unit MailTemplateLog;

{
  A logging file for the mail template program.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - October 17 1999.
  Version 1.1 - April 25, 2000.
}

interface

uses
  SysUtils;

type
  TMailTemplateLog = class(TObject)
  private
    FLogFile: Text;
    function TimeStamp: string;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Log(Message: string);
    procedure LogTest(FromEmail, ToEmail, Subject, Message: string);
    procedure Error(Error: Exception);
  end;

implementation

resourcestring
  LogFrom    = 'From: ';
  LogMessage = 'Message: ';
  LogSubject = 'Subject: ';
  LogTo      = 'To: ';
  TestOnly   = 'Test only----------------------------------------------------';

const
  LogExt     = '.log';
  LogFormat  = '.yyyymmddhhnnss';
  TimeFormat = '[hh:nn:ss] ';

{ TMailTemplateLog ------------------------------------------------------------}

{ Open the log file }
constructor TMailTemplateLog.Create;
var
  Filename: string;
begin
  inherited Create;
  Filename := ChangeFileExt(ExtractFileName(ParamStr(0)),
    FormatDateTime(LogFormat, Now) + LogExt);
  AssignFile(FLogFile, Filename);
  Rewrite(FLogFile);
end;

{ Close the log file }
destructor TMailTemplateLog.Destroy;
begin
  CloseFile(FLogFile);
  inherited Destroy;
end;

{ Write an error message }
procedure TMailTemplateLog.Error(Error: Exception);
begin
  Log(Error.Message);
end;

{ Write a log message }
procedure TMailTemplateLog.Log(Message: string);
begin
  Writeln(FLogFile, TimeStamp + Message);
  Flush(FLogFile);
end;

{ Write a testing message }
procedure TMailTemplateLog.LogTest(FromEmail, ToEmail, Subject, Message: string);
begin
  Writeln(FLogFile, TestOnly);
  Writeln(FLogFile, LogFrom + FromEmail);
  Writeln(FLogFile, LogTo + ToEmail);
  Writeln(FLogFile, LogSubject + Subject);
  Writeln(FLogFile, LogMessage + Message);
  { Ensure it gets written out }
  Flush(FLogFile);
end;

{ Return the current time }
function TMailTemplateLog.TimeStamp: string;
begin
  Result := FormatDateTime(TimeFormat, Now);
end;

end.
