unit MWApp1;

{
  Movie-watchers. A demonstration of XML document processing.
  This front-end provides access to the movie-watcher objects.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 3 July 1999.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls, ExtCtrls, Grids, MWObjs;

type
  { The user interface }
  TfrmMovieWatchers = class(TForm)
    tabNavigation: TTabControl;
    lbxNavigation: TListBox;
    pgcDetails: TPageControl;
      tshMovie: TTabSheet;
        Label1: TLabel;
        edtTitle: TEdit;
        Label2: TLabel;
        edtRating: TEdit;
        Label3: TLabel;
        edtLength: TEdit;
        Label4: TLabel;
        edtDirector: TEdit;
        Label5: TLabel;
        lbxStars: TListBox;
        Label6: TLabel;
        memSynopsis: TMemo;
        Label7: TLabel;
        lbxCinemas: TListBox;
      tshCinema: TTabSheet;
        Label8: TLabel;
        edtName: TEdit;
        Label9: TLabel;
        edtPhone: TEdit;
        Label10: TLabel;
        edtAddress: TEdit;
        Label11: TLabel;
        memDirections: TMemo;
        cbxDisabledAccess: TCheckBox;
        cbxCandyBar: TCheckBox;
        Label13: TLabel;
        stgPricing: TStringGrid;
        Label12: TLabel;
        lbxMovies: TListBox;
      tshScreening: TTabSheet;
        Label14: TLabel;
        edtMovie: TEdit;
        Label15: TLabel;
        edtCinema: TEdit;
        Label16: TLabel;
        edtDates: TEdit;
        Label17: TLabel;
        edtSound: TEdit;
        cbxNoPasses: TCheckBox;
        Label18: TLabel;
        lbxSessions: TListBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure tabNavigationChange(Sender: TObject);
    procedure lbxNavigationClick(Sender: TObject);
    procedure lbxCinemasDblClick(Sender: TObject);
    procedure lbxCinemasKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure lbxMoviesDblClick(Sender: TObject);
    procedure lbxMoviesKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edtMovieDblClick(Sender: TObject);
    procedure edtMovieKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edtCinemaDblClick(Sender: TObject);
    procedure edtCinemaKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure lbxSessionsDblClick(Sender: TObject);
    procedure lbxSessionsKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    FMovies: TStringList;
    FCinemas: TStringList;
    FScreenings: TStringList;
    procedure ShowList(ListType: Integer; Name: string);
    procedure ShowMovie(Movie: TMovie);
    procedure ShowCinema(Cinema: TCinema);
    procedure ShowScreening(Screening: TScreening);
  public
  end;

var
  frmMovieWatchers: TfrmMovieWatchers;

implementation

{$R *.DFM}

resourcestring
  { Tab names }
  Movies     = '&Movies';
  Cinemas    = '&Cinemas';
  Screenings = '&Screenings';
  { Pricing headings }
  Name     = 'Name';
  Period   = 'Period';
  Adult    = 'Adult';
  Child    = 'Child';
  Discount = 'Discount';
  { Miscellaneous }
  ScreeningDesc = '%s at %s';

const
  { Tab indexes }
  MoviesTab     = 0;
  CinemasTab    = 1;
  ScreeningsTab = 2;
  { Miscellaneous }
  TimeFormat    = 'h:nn';
  DollarsFormat = '%6.2f';
  Tab           = #09;

{ TfrmMovieWatchers -----------------------------------------------------------}

{ Initialisation - load XML document and process }
procedure TfrmMovieWatchers.FormCreate(Sender: TObject);
begin
  tabNavigation.Tabs[0] := Movies;
  tabNavigation.Tabs[1] := Cinemas;
  tabNavigation.Tabs[2] := Screenings;
  with stgPricing do
  begin
    ColWidths[0] := 50;
    ColWidths[1] := 83;
    ColWidths[2] := 40;
    ColWidths[3] := 40;
    ColWidths[4] := 40;
    Cells[0, 0]  := Name;
    Cells[1, 0]  := Period;
    Cells[2, 0]  := Adult;
    Cells[3, 0]  := Child;
    Cells[4, 0]  := Discount;
  end;
  { Create the internal lists }
  FMovies     := TStringList.Create;
  FCinemas    := TStringList.Create;
  FScreenings := TStringList.Create;
  { Open the XML file }
  try
    LoadDocument(ParamStr(1), FMovies, FCinemas, FScreenings);
  except on Error: Exception do
    begin
      MessageDlg(Error.Message, mtError, [mbOK], 0);
      Halt;
    end;
  end;
  { Start out on the movies }
  ShowList(MoviesTab, '');
end;

{ Release resources }
procedure TfrmMovieWatchers.FormDestroy(Sender: TObject);
var
  Index: Integer;
begin
  for Index := 0 to FMovies.Count - 1 do
    TMovie(FMovies.Objects[Index]).Free;
  FMovies.Free;
  for Index := 0 to FCinemas.Count - 1 do
    TCinema(FCinemas.Objects[Index]).Free;
  FCinemas.Free;
  for Index := 0 to FScreenings.Count - 1 do
    TScreening(FScreenings.Objects[Index]).Free;
  FScreenings.Free;
end;

{ Show list of items and select one }
procedure TfrmMovieWatchers.ShowList(ListType: Integer; Name: string);
begin
  tabNavigation.TabIndex  := ListType;
  tabNavigationChange(tabNavigation);
  lbxNavigation.ItemIndex := lbxNavigation.Items.IndexOf(Name);
  if lbxNavigation.ItemIndex = -1 then
    lbxNavigation.ItemIndex := 0;
  lbxNavigationClick(lbxNavigation);
end;

{ Show selected details in listbox }
procedure TfrmMovieWatchers.tabNavigationChange(Sender: TObject);
begin
  with lbxNavigation do
  begin
    Items.BeginUpdate;
    Items.Clear;
    if tabNavigation.TabIndex = MoviesTab then
      Items := FMovies
    else if tabNavigation.TabIndex = CinemasTab then
      Items := FCinemas
    else if tabNavigation.TabIndex = ScreeningsTab then
      Items := FScreenings;
    Items.EndUpdate;
  end;
  lbxNavigation.ItemIndex := 0;
  lbxNavigationClick(lbxNavigation);
  ActiveControl           := lbxNavigation;
end;

{ Display details for a movie }
procedure TfrmMovieWatchers.ShowMovie(Movie: TMovie);
var
  Index: Integer;
begin
  with Movie do
  begin
    edtTitle.Text          := Name;
    edtRating.Text         := MovieRatingText[Rating];
    edtLength.Text         := FormatDateTime(TimeFormat, Length);
    edtDirector.Text       := Director;
    lbxStars.Items         := Stars;
    memSynopsis.Lines.Text := Synopsis;
    { Show which cinemas it is playing at }
    with lbxCinemas.Items do
    begin
      BeginUpdate;
      Clear;
      for Index := 0 to FScreenings.Count - 1 do
        if TScreening(FScreenings.Objects[Index]).Movie = Movie then
          AddObject(TScreening(FScreenings.Objects[Index]).Cinema.Name,
            FScreenings.Objects[Index]);
      if Count > 0 then
        lbxCinemas.ItemIndex := 0;
      EndUpdate;
    end;
  end;
  pgcDetails.ActivePage := tshMovie;
end;

{ Display details for a cinema }
procedure TfrmMovieWatchers.ShowCinema(Cinema: TCinema);
var
  Index: Integer;
begin
  with Cinema do
  begin
    edtName.Text              := Name;
    edtPhone.Text             := Phone;
    edtAddress.Text           := Address;
    memDirections.Lines.Text  := Directions;
    cbxCandyBar.Checked       := CandyBar;
    cbxDisabledAccess.Checked := DisabledAccess;
    with stgPricing do
    begin
      RowCount := Pricing.Count + 1;
      for Index := 0 to Pricing.Count - 1 do
        with TPrice(Pricing.Objects[Index]) do
        begin
          Cells[0, Index + 1] := Name;
          Cells[1, Index + 1] := Period;
          Cells[2, Index + 1] := Format(DollarsFormat, [Adult]);
          Cells[3, Index + 1] := Format(DollarsFormat, [Child]);
          Cells[4, Index + 1] := Format(DollarsFormat, [Discount]);
        end;
    end;
    { Show which movies it screens }
    with lbxMovies.Items do
    begin
      BeginUpdate;
      Clear;
      for Index := 0 to FScreenings.Count - 1 do
        if TScreening(FScreenings.Objects[Index]).Cinema = Cinema then
          AddObject(TScreening(FScreenings.Objects[Index]).Movie.Name,
            FScreenings.Objects[Index]);
      if Count > 0 then
        lbxMovies.ItemIndex := 0;
      EndUpdate;
    end;
  end;
  pgcDetails.ActivePage := tshCinema;
end;

{ Display details for a screening }
procedure TfrmMovieWatchers.ShowScreening(Screening: TScreening);
var
  Index: Integer;
begin
  with Screening do
  begin
    edtMovie.Text       := Movie.Name;
    edtCinema.Text      := Cinema.Name;
    edtDates.Text       := DateToStr(StartDate) + ' - ' + DateToStr(EndDate);
    edtSound.Text       := DigitalSound;
    cbxNoPasses.Checked := NoPasses;
    with lbxSessions.Items do
    begin
      BeginUpdate;
      Clear;
      for Index := 0 to Screening.Showing.Count - 1 do
        Add(Screening.Showing[Index] + Tab +
          TPrice(Screening.Showing.Objects[Index]).Name);
      if Count > 0 then
        lbxSessions.ItemIndex := 0;
      EndUpdate;
    end;
  end;
  pgcDetails.ActivePage := tshScreening;
end;

{ Select an item to display its details }
procedure TfrmMovieWatchers.lbxNavigationClick(Sender: TObject);
begin
  with lbxNavigation do
  begin
    if ItemIndex < 0 then
      ItemIndex := 0;
    if tabNavigation.TabIndex = MoviesTab then
      ShowMovie(TMovie(Items.Objects[ItemIndex]))
    else if tabNavigation.TabIndex = CinemasTab then
      ShowCinema(TCinema(Items.Objects[ItemIndex]))
    else if tabNavigation.TabIndex = ScreeningsTab then
      ShowScreening(TScreening(Items.Objects[ItemIndex]));
  end;
end;

{ Go to the screening details for a movie }
procedure TfrmMovieWatchers.lbxCinemasDblClick(Sender: TObject);
begin
  ShowList(ScreeningsTab, Format(ScreeningDesc,
    [edtTitle.Text, lbxCinemas.Items[lbxCinemas.ItemIndex]]));
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.lbxCinemasKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxCinemasDblClick(lbxCinemas);
end;

{ Go to the screening details for a cinema }
procedure TfrmMovieWatchers.lbxMoviesDblClick(Sender: TObject);
begin
  ShowList(ScreeningsTab, Format(ScreeningDesc,
    [lbxMovies.Items[lbxMovies.ItemIndex], edtName.Text]));
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.lbxMoviesKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxMoviesDblClick(lbxMovies);
end;

{ Go to the movie details }
procedure TfrmMovieWatchers.edtMovieDblClick(Sender: TObject);
begin
  ShowList(MoviesTab, edtMovie.Text);
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.edtMovieKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_RETURN then
    edtMovieDblClick(edtMovie);
end;

{ Go to the cinema details }
procedure TfrmMovieWatchers.edtCinemaDblClick(Sender: TObject);
begin
  ShowList(CinemasTab, edtCinema.Text);
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.edtCinemaKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    edtCinemaDblClick(edtCinema);
end;

{ Go to the cinema/pricing details }
procedure TfrmMovieWatchers.lbxSessionsDblClick(Sender: TObject);
var
  Index: Integer;
  Pricing: string;
begin
  with lbxSessions do
  begin
    Index   := Pos(Tab, Items[ItemIndex]);
    Pricing := Copy(Items[ItemIndex], Index + 1, Length(Items[ItemIndex]));
  end;
  ShowList(CinemasTab, edtCinema.Text);
  with stgPricing do
    for Index := 1 to RowCount - 1 do
      if Cells[0, Index] = Pricing then
      begin
        Row := Index;
        Exit;
      end;
end;

{ Enter acts like a double-click }
procedure TfrmMovieWatchers.lbxSessionsKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxSessionsDblClick(lbxSessions);
end;

end.
