unit MWObjs;

{
  Movie-watchers objects. A demonstration of XML document processing.
  This unit contains definitions for objects that correspond to the
  elements in the XML document. It includes a procedure to extract
  them and place them in suitable lists for further processing.
  It uses a SAX for Pascal reader to load the document.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 28 December, 2002.
}

interface

uses
  Classes, SysUtils, SAX, SAXHelpers, SAXMS;

type
  TMovieRating = (mrUnrated, mrG, mrPG, mrPG13, mrR);

const
  { Text equivalents for the ratings }
  MovieRatingText: array [TMovieRating] of string[7] =
    ('Unrated', 'G', 'PG', 'PG-13', 'R');

type
  { Details about a movie }
  TMovie = class(TObject)
  private
    FId: string;
    FName: string;
    FRating: TMovieRating;
    FLength: TDateTime;
    FDirector: string;
    FStars: TStringList;
    FSynopsis: string;
    function GetRatingText: string;
    procedure SetRatingText(RatingText: string);
  public
    constructor Create(Id: string);
    destructor Destroy; override;
    property Id: string read FId write FId;
    property Name: string read FName write FName;
    property Rating: TMovieRating read FRating write FRating;
    property RatingText: string read GetRatingText write SetRatingText;
    property Length: TDateTime read FLength write FLength;
    property Director: string read FDirector write FDirector;
    property Stars: TStringList read FStars write FStars;
    property Synopsis: string read FSynopsis write FSynopsis;
  end;

  { Details about a cinema }
  TCinema = class(TObject)
  private
    FId: string;
    FName: string;
    FPhone: string;
    FAddress: string;
    FDirections: string;
    FCandyBar: Boolean;
    FDisabledAccess: Boolean;
    FPricing: TStringList;
  public
    constructor Create(Id: string);
    destructor Destroy; override;
    property Id: string read FId write FId;
    property Name: string read FName write FName;
    property Phone: string read FPhone write FPhone;
    property Address: string read FAddress write FAddress;
    property Directions: string read FDirections write FDirections;
    property CandyBar: Boolean read FCandyBar write FCandyBar;
    property DisabledAccess: Boolean read FDisabledAccess
      write FDisabledAccess;
    property Pricing: TStringList read FPricing write FPricing;
  end;

  { Details about a cinema's pricing scheme }
  TPrice = class(TObject)
  private
    FId: string;
    FName: string;
    FPeriod: string;
    FAdult: Double;
    FChild: Double;
    FDiscount: Double;
  public
    constructor Create(Id: string);
    property Id: string read FId write FId;
    property Name: string read FName write FName;
    property Period: string read FPeriod write FPeriod;
    property Adult: Double read FAdult write FAdult;
    property Child: Double read FChild write FChild;
    property Discount: Double read FDiscount write FDiscount;
  end;

  { Details about what is showing where and when }
  TScreening = class(TObject)
  private
    FMovie: TMovie;
    FCinema: TCinema;
    FStartDate: TDateTime;
    FEndDate: TDateTime;
    FDigitalSound: string;
    FNoPasses: Boolean;
    FShowing: TStringList;
  public
    constructor Create(Movie: TMovie; Cinema: TCinema);
    destructor Destroy; override;
    property Movie: TMovie read FMovie write FMovie;
    property Cinema: TCinema read FCinema write FCinema;
    property StartDate: TDateTime read FStartDate write FStartDate;
    property EndDate: TDateTime read FEndDate write FEndDate;
    property DigitalSound: string read FDigitalSound write FDigitalSound;
    property NoPasses: Boolean read FNoPasses write FNoPasses;
    property Showing: TStringList read FShowing write FShowing;
  end;

{ Load XML document and process into string lists
  with references to the appropriate objects }
procedure LoadDocument(URI: string;
  MoviesList, CinemasList, ScreeningsList: TStringList);

implementation

resourcestring
  InvalidDocument = 'Invalid movie-watcher XML document loaded'#13#10;
  ScreeningDesc   = '%s at %s';

const
  { Movie-watcher element names }
  MWMovie          = 'movie';
  MWId             = 'id';
  MWName           = 'name';
  MWRating         = 'rating';
  MWLength         = 'length';
  MWDirector       = 'director';
  MWStar           = 'star';
  MWSynopsis       = 'synopsis';
  MWCinema         = 'cinema';
  MWPhone          = 'phone';
  MWAddress        = 'address';
  MWDirections     = 'directions';
  MWFacilities     = 'facilities';
  MWCandyBar       = 'candy-bar';
  MWDisabledAccess = 'disabled-access';
  MWPrices         = 'prices';
  MWPeriod         = 'period';
  MWAdult          = 'adult';
  MWChild          = 'child';
  MWDiscount       = 'discount';
  MWScreening      = 'screening';
  MWMovieId        = 'movie-id';
  MWCinemaId       = 'cinema-id';
  MWStartDate      = 'start-date';
  MWEndDate        = 'end-date';
  MWDigitalSound   = 'digital-sound';
  MWNoPasses       = 'no-passes';
  MWSession        = 'session';
  MWPriceId        = 'price-id';

{ TMovie ----------------------------------------------------------------------}

{ Initialisation - set the Id }
constructor TMovie.Create(Id: string);
begin
  inherited Create;
  Self.Id := Id;
  FStars  := TStringList.Create;
end;

{ Release resources }
destructor TMovie.Destroy;
begin
  FStars.Free;
  inherited Destroy;
end;

{ Translate enumerated type to movie rating text }
function TMovie.GetRatingText: string;
begin
  Result := MovieRatingText[Rating];
end;

{ Translate movie rating text to enumerated type }
procedure TMovie.SetRatingText(RatingText: string);
var
  mrRating: TMovieRating;
begin
  for mrRating := mrR downto mrG do
    if MovieRatingText[mrRating] = RatingText then
    begin
      Rating := mrRating;
      Exit;
    end;
  Rating := mrUnrated;
end;

{ TCinema ---------------------------------------------------------------------}

{ Initialisation - set the Id }
constructor TCinema.Create(Id: string);
begin
  inherited Create;
  Self.Id  := Id;
  FPricing := TStringList.Create;
end;

{ Release resources }
destructor TCinema.Destroy;
var
  Index: Integer;
begin
  for Index := 0 to FPricing.Count - 1 do
    TPrice(FPricing.Objects[Index]).Free;
  FPricing.Free;
  inherited Destroy;
end;

{ TPrice ----------------------------------------------------------------------}

{ Initialisation - set the Id }
constructor TPrice.Create(Id: string);
begin
  inherited Create;
  Self.Id := Id;
end;

{ TScreening ------------------------------------------------------------------}

{ Initialisation - set the movie and cinema }
constructor TScreening.Create(Movie: TMovie; Cinema: TCinema);
begin
  inherited Create;
  Self.Movie  := Movie;
  Self.Cinema := Cinema;
  FShowing    := TStringList.Create;
end;

{ Release resources }
destructor TScreening.Destroy;
begin
  FShowing.Free;
  inherited Destroy;
end;

{ TMWContentHandler -----------------------------------------------------------}

type
  { A SAX content handler that knows about movie-watcher documents }
  TMWContentHandler = class(TDefaultHandler)
  private
    FCinema: TCinema;
    FCinemas: TList;
    FMovie: TMovie;
    FMovies: TList;
    FPrice: TPrice;
    FScreening: TScreening;
    FScreenings: TList;
    FText: string;
  public
    constructor Create;
    destructor Destroy; override;
    property Cinemas: TList read FCinemas;
    property Movies: TList read FMovies;
    property Screenings: TList read FScreenings;
    { IContentHandler }
    procedure Characters(const ch: SAXString); override;
    procedure EndElement(const uri, localName, qName: SAXString); override;
    procedure StartElement(const uri, localName, qName: SAXString;
      const atts: IAttributes); override;
  end;

{ Initialisation - allocate lists }
constructor TMWContentHandler.Create;
begin
  inherited Create;
  FCinemas    := TList.Create;
  FMovies     := TList.Create;
  FScreenings := TList.Create;
end;

{ Release resources }
destructor TMWContentHandler.Destroy;
begin
  FCinemas.Free;
  FMovies.Free;
  FScreenings.Free;
  inherited Destroy;
end;

{ Accumulate text content }
procedure TMWContentHandler.Characters(const ch: SAXString);
begin
  FText := FText + ch;
end;

{ Save text content to appropriate property }
procedure TMWContentHandler.EndElement(const uri, localName, qName: SAXString);

  { Replace consecutive white space with one space }
  function Normalize(const Text: string): string;
  const
    Blanks = [#1..#32];
  var
    Index: Integer;
  begin
    Result := Text;
    if Length(Text) < 2 then
      Exit;
    for Index := Length(Result) downto 2 do
      if (Result[Index] in Blanks) and (Result[Index - 1] in Blanks) then
      begin
        Result[Index - 1] := ' ';
        Delete(Result, Index, 1);
      end;
  end;

  { Return the accumulated text and clear for next time }
  function ReadAndClearText: string;
  begin
    Result := Trim(Normalize(FText));
    FText  := '';
  end;

begin
  if qName = MWMovie then
    FMovie := nil
  else if qName = MWMovie then
    FCinema := nil
  else if qName = MWPrices then
    FPrice := nil
  else if qName = MWScreening then
    FScreening := nil
  else if qName = MWName then
  begin
    if Assigned(FMovie) then
      FMovie.Name := ReadAndClearText
    else if Assigned(FPrice) then
      FPrice.Name := ReadAndClearText
    else if Assigned(FCinema) then
      FCinema.Name := ReadAndClearText;
  end
  else if qName = MWLength then
    FMovie.Length := StrToInt(ReadAndClearText) / 24 / 60
  else if qName = MWDirector then
    FMovie.Director := ReadAndClearText
  else if qName = MWStar then
    FMovie.Stars.Add(ReadAndClearText)
  else if qName = MWSynopsis then
    FMovie.Synopsis := ReadAndClearText
  else if qName = MWPhone then
    FCinema.Phone := ReadAndClearText
  else if qName = MWAddress then
    FCinema.Address := ReadAndClearText
  else if qName = MWDirections then
    FCinema.Directions := ReadAndClearText
  else if qName = MWCandyBar then
    FCinema.CandyBar := True
  else if qName = MWDisabledAccess then
    FCinema.DisabledAccess := True
  else if qName = MWPeriod then
    FPrice.Period := ReadAndClearText
  else if qName = MWAdult then
    FPrice.Adult := StrToFloat(ReadAndClearText)
  else if qName = MWChild then
    FPrice.Child := StrToFloat(ReadAndClearText)
  else if qName = MWDiscount then
    FPrice.Discount := StrToFloat(ReadAndClearText)
  else if qName = MWStartDate then
    FScreening.StartDate := StrToDateTime(ReadAndClearText)
  else if qName = MWEndDate then
    FScreening.EndDate := StrToDateTime(ReadAndClearText)
  else if qName = MWNoPasses then
    FScreening.NoPasses := True
  else if qName = MWDigitalSound then
    FScreening.DigitalSound := ReadAndClearText
  else if qName = MWSession then
    FScreening.Showing.AddObject(ReadAndClearText, FPrice);
end;

{ Create objects as necessary for document elements }
procedure TMWContentHandler.StartElement(
    const uri, localName, qName: SAXString; const atts: IAttributes);

  { Locate the movie with the given identifier }
  function FindMovie(Id: string): TMovie;
  var
    Index: Integer;
  begin
    Result := nil;
    for Index := 0 to FMovies.Count - 1 do
      if TMovie(FMovies[Index]).Id = Id then
      begin
        Result := TMovie(FMovies[Index]);
        Exit;
      end;
  end;

  { Locate the cinema with the given identifier }
  function FindCinema(Id: string): TCinema;
  var
    Index: Integer;
  begin
    Result := nil;
    for Index := 0 to FCinemas.Count - 1 do
      if TCinema(FCinemas[Index]).Id = Id then
      begin
        Result := TCinema(FCinemas[Index]);
        Exit;
      end;
  end;

  { Locate the pricing scheme with the given identifier }
  function FindPrice(PriceId: string): TPrice;
  var
    Index, Index2: Integer;
  begin
    Result := nil;
    for Index := 0 to FCinemas.Count - 1 do
      with TCinema(FCinemas[Index]) do
      begin
        Index2 := Pricing.IndexOf(PriceId);
        if Index2 > -1 then
        begin
          Result := TPrice(Pricing.Objects[Index2]);
          Exit;
        end;
      end;
  end;

begin
  if qName = MWMovie then
  begin
    FMovie            := TMovie.Create(atts.getValue(MWId));
    FMovie.RatingText := atts.getValue(MWRating);
    FMovies.Add(FMovie);
  end
  else if qName = MWCinema then
  begin
    FCinema := TCinema.Create(atts.getValue(MWId));
    FCinemas.Add(FCinema);
  end
  else if qName = MWPrices then
  begin
    FPrice := TPrice.Create(atts.getValue(MWId));
    FCinema.Pricing.AddObject(atts.getValue(MWId), FPrice);
  end
  else if qName = MWScreening then
  begin
    FScreening := TScreening.Create(FindMovie(atts.getValue(MWMovieId)),
      FindCinema(atts.getValue(MWCinemaId)));
    FScreenings.Add(FScreening);
  end
  else if qName = MWSession then
    FPrice := FindPrice(atts.getValue(MWPriceId));
end;

{ Loading data ----------------------------------------------------------------}

{ Load XML document and process into string lists
  with references to the appropriate objects }
procedure LoadDocument(URI: string;
  MoviesList, CinemasList, ScreeningsList: TStringList);
var
  Index: Integer;
  XMLReader: IXMLReader;
  Handler: TMWContentHandler;
begin
  { Create the XML parser }
  Handler := TMWContentHandler.Create;
  try
    XMLReader                := GetSAXVendor.XMLReader;
    XMLReader.ContentHandler := Handler;
    { And parse the document }
    XMLReader.parse(URI);
    with Handler do
    begin
      { Are they all here? }
      if (Movies.Count = 0) or (Cinemas.Count = 0) or
          (Screenings.Count = 0) then
        raise Exception.Create(InvalidDocument + URI);
      { Step through the handler's lists and convert to output format }
      for Index := 0 to Movies.Count - 1  do
        MoviesList.AddObject(TMovie(Movies[Index]).Name, Movies[Index]);
      for Index := 0 to Cinemas.Count - 1 do
        CinemasList.AddObject(TCinema(Cinemas[Index]).Name, Cinemas[Index]);
      for Index := 0 to Screenings.Count - 1 do
        ScreeningsList.AddObject(Format(ScreeningDesc,
          [TScreening(Screenings[Index]).Movie.Name,
          TScreening(Screenings[Index]).Cinema.Name]), Screenings[Index]);
    end;
  finally
    if Handler.RefCount = 0 then
      Handler.Free;
  end;
end;

initialization
  ShortDateFormat := 'MM/dd/yyyy';
end.
