unit OXMLView1;

{
  A generic viewer for XML documents.
  Based on Open XML interfaces.
  Requires XDOM v2.4 package from Open XML.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 6 July, 1999.
  Updated 24 May 2003, XDOM v2.4.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Menus, ComCtrls, Grids,
{$IFDEF VER120}  { Delphi 4 }
  ImgList,
{$ENDIF}
  XDOM_2_4;

type
  TfrmXMLViewer = class(TForm)
    pgcMain: TPageControl;
      tshStructure: TTabSheet;
        trvXML: TTreeView;
        pgcDetails: TPageControl;
          tshDocument: TTabSheet;
            Label1: TLabel;
            edtDocType: TEdit;
            cbxStandAlone: TCheckBox;
            Label2: TLabel;
            edtPublicId: TEdit;
            Label3: TLabel;
            edtSystemId: TEdit;
            Label4: TLabel;
            edtVersion: TEdit;
            Label5: TLabel;
            edtEncoding: TEdit;
            Label6: TLabel;
            stgEntities: TStringGrid;
            Label7: TLabel;
            stgNotations: TStringGrid;
          tshElement: TTabSheet;
            pnlNames: TPanel;
              Label8: TLabel;
              edtURI: TEdit;
              Label9: TLabel;
              edtLocalName: TEdit;
            stgAttributes: TStringGrid;
          tshText: TTabSheet;
            lblNodeType: TLabel;
            memText: TMemo;
      tshSource: TTabSheet;
        memSource: TRichEdit;
    mnuMain: TMainMenu;
      mniFile: TMenuItem;
        mniOpen: TMenuItem;
        mniSep1: TMenuItem;
        mniKeepCDATASections: TMenuItem;
        mniKeepComments: TMenuItem;
        mniKeepEntityReferences: TMenuItem;
        mniSuppressNamespaceDeclarations: TMenuItem;
        mniSep2: TMenuItem;
        mniExit: TMenuItem;
      mniView: TMenuItem;
        mniExpandAll: TMenuItem;
        mniCollapseAll: TMenuItem;
        mniSep3: TMenuItem;
        mniViewSource: TMenuItem;
    imlXML: TImageList;
    dlgOpen: TOpenDialog;
    domImplementation: TDomImplementation;
    xmlParser: TXmlToDomParser;
    xmlDomReader: TXmlStandardDomReader;
    xmlDocBuilder: TXmlDocBuilder;
    mniValidate: TMenuItem;
    mniReplaceEntityReferences: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mniOpenClick(Sender: TObject);
    procedure mniOptionClick(Sender: TObject);
    procedure mniExitClick(Sender: TObject);
    procedure mniExpandAllClick(Sender: TObject);
    procedure mniCollapseAllClick(Sender: TObject);
    procedure mniViewSourceClick(Sender: TObject);
    procedure trvXMLChange(Sender: TObject; Node: TTreeNode);
    procedure xmlParserExternalEntity(sender: TObject;
      const parentSystemId: WideString; var publicId, systemId: WideString;
      var stream: TStream; var action: TXmlParserAction);
  private
    FList: TList;
    procedure ClearTree;
    procedure LoadDoc(const Filename: string);
  public
  end;

var
  frmXMLViewer: TfrmXMLViewer;

implementation

{$R *.DFM}

resourcestring
  AttributeDesc   = 'Attribute';
  CommentDesc     = 'Comment';
  DefinitionDesc  = 'Definition';
  DTDDesc         = 'DTD';
  ElementDeclDesc = 'Element Declaration';
  EntityRefDesc   = 'Entity Declaration/Reference';
  ExternalDesc    = 'External';
  InstructionDesc = 'Processing Instruction';
  InternalDesc    = 'Internal';
  NameDesc        = 'Name';
  NotationDesc    = 'Notation';
  PublicDesc      = 'Public Id';
  SystemDesc      = 'System Id';
  TextDesc        = 'Text/CDATA Section';
  ValueDesc       = 'Value';
  XMLDocDesc      = 'XML Document';

const
  AttrTypes: array [TdomASDataType] of string =
    ('string', 'NOTATION', 'ID', 'IDREF', 'IDREFS', 'ENTITY', 'ENTITIES',
    'NMTOKEN', 'NMTOKENS');
  ContentTypes: array [TdomASContentType] of string =
    ('ANY', 'EMPTY', '', '#PCDATA', '', '');
  Frequencies: array [TdomASFrequency] of string = ('', '?', '+', '*');
  ModelSeparator: array [TdomASContentModelType] of string = ('|', '', ',');
  ValueConstraints: array [TdomASValueConstraint] of string =
    ('', '#FIXED', '#IMPLIED', '#REQUIRED');

{ TNodeInfo -------------------------------------------------------------------}

type
  { XML node types }
  TNodeType = (ntDocument, ntElement, ntComment, ntInstruction,
    ntText, ntCData, ntDTD, ntElementDecl, ntAttributeDecl, ntEntityRef);

  { XML node.

    Nodes have a name and/or a value depending on their type.
    Attributes are name=value pairs in a string list.
  }
  TNodeInfo = class(TObject)
  private
    FAttributes: TStringList;
    FLocalName: string;
    FName: string;
    FNamespaceURI: string;
    FNodeType: TNodeType;
    FValue: string;
  public
    constructor Create(const NodeType: TNodeType;
      Name, NamespaceURI, LocalName, Value: string; Attributes: TStrings);
    destructor Destroy; override;
    property Attributes: TStringList read FAttributes write FAttributes;
    property LocalName: string read FLocalName write FLocalName;
    property Name: string read FName write FName;
    property NamespaceURI: string read FNamespaceURI write FNamespaceURI;
    property NodeType: TNodeType read FNodeType write FNodeType;
    property Value: string read FValue write FValue;
  end;

{ Initialisation }
constructor TNodeInfo.Create(const NodeType: TNodeType;
  Name, NamespaceURI, LocalName, Value: string; Attributes: TStrings);
begin
  inherited Create;
  FLocalName    := LocalName;
  FName         := Name;
  FNamespaceURI := NamespaceURI;
  FNodeType     := NodeType;
  FValue        := Value;
  FAttributes   := TStringList.Create;
  if Assigned(Attributes) then
    FAttributes.Assign(Attributes);
end;

{Release resources }
destructor TNodeInfo.Destroy;
begin
  FAttributes.Free;
  inherited Destroy;
end;

{ TfrmXMLViewer ---------------------------------------------------------------}

{ Initialisation - try to load an XML document on start up }
procedure TfrmXMLViewer.FormCreate(Sender: TObject);
begin
  Caption := 'XML Viewer (Open XML v' + domImplementation.xdomVersion + ')';
  FList   := TList.Create;
  with stgEntities do
  begin
    Cells[0, 0] := NameDesc;
    Cells[1, 0] := PublicDesc;
    Cells[2, 0] := SystemDesc;
    Cells[3, 0] := NotationDesc;
  end;
  with stgNotations do
  begin
    Cells[0, 0] := NameDesc;
    Cells[1, 0] := PublicDesc;
    Cells[2, 0] := SystemDesc;
  end;
  with stgAttributes do
  begin
    Cells[0, 0] := AttributeDesc;
    Cells[1, 0] := ValueDesc;
  end;
  if ParamStr(1) <> '' then
  begin
    LoadDoc(ParamStr(1));
    dlgOpen.InitialDir := ExtractFilePath(ParamStr(1));
  end
  else
    dlgOpen.InitialDir := ExtractFilePath(Application.ExeName);
end;

{ Release resources }
procedure TfrmXMLViewer.FormDestroy(Sender: TObject);
begin
  ClearTree;
  FList.Free;
end;

{ Release resources }
procedure TfrmXMLViewer.ClearTree;
var
  Index: Integer;
begin
  for Index := 0 to FList.Count - 1 do
    TNodeInfo(FList[Index]).Free;
  FList.Clear;
  trvXML.OnChange := nil;
  trvXML.Items.Clear;
  trvXML.OnChange := trvXMLChange;
end;

{ Load an XML document }
procedure TfrmXMLViewer.LoadDoc(const Filename: string);
const
  ResolveOption: array [Boolean] of TdomEntityResolveOption =
    (erExpand, erReplace);
var
  xmlDoc: TdomDocument;

  { Initialise document-wide details for display }
  procedure InitDocumentDetails;
  begin
    { Clear entries }
    edtDocType.Text       := '';
    edtPublicId.Text      := '';
    edtSystemId.Text      := '';
    edtVersion.Text       := '';
    edtEncoding.Text      := '';
    cbxStandAlone.Checked := False;
    with stgEntities do
    begin
      RowCount := 2;
      Rows[1].Clear;
    end;
    with stgNotations do
    begin
      RowCount := 2;
      Rows[1].Clear;
    end;
    ClearTree;
  end;

  { Add a TNodeInfo to the tree view }
  function AddNodeInfo(Parent: TTreeNode; const Name: string;
    NodeInfo: TNodeInfo): TTreeNode;
  begin
    FList.Add(NodeInfo);
    Result := trvXML.Items.AddChildObject(Parent, Name, NodeInfo);
    with Result do
    begin
      ImageIndex    := Ord(NodeInfo.NodeType);
      SelectedIndex := ImageIndex;
    end;
  end;

  { Compile the content model for an element declaration }
  function CompileContentModel(Model: TdomASContentModel;
    ContentType: TdomASContentType): string;
  var
    Index: Integer;
  begin
    if not Assigned(Model) then
      Result := ContentTypes[ContentType]
    else if Model.subModels.length = 0 then
    begin
      if Model.name = '' then
        Result := ContentTypes[ContentType]
      else
        Result := Model.name + Frequencies[Model.frequency];
    end
    else
    begin
      if ContentType = AS_MIXED_CONTENTTYPE then
        Result :=
          ModelSeparator[Model.contentModelType] + ContentTypes[ContentType]
      else
        Result := '';
      for Index := 0 to Model.subModels.length - 1 do
        Result := Result + ModelSeparator[Model.contentModelType] +
          CompileContentModel(Model.subModels.item(Index) as TdomASContentModel,
          AS_UNKNOWN_CONTENTTYPE);
      Result := '(' + Copy(Result, 2, Length(Result) - 1) + ')' +
        Frequencies[Model.frequency];
    end;
  end;

  { Add DTD definitions (abstract schema model) to the tree }
  procedure ProcessASModel(Model: TdomASModel; TreeParent: TTreeNode;
    Internal: Boolean);
  var
    ModelNode: TTreeNode;
    Index, Index2, Index3: Integer;
    NameDisplay, ValueDisplay: string;
    Attribs: TStringList;
  begin
    if Internal then
      ValueDisplay := InternalDesc
    else
      ValueDisplay := ExternalDesc;
    ModelNode := AddNodeInfo(TreeParent, ValueDisplay, TNodeInfo.Create(
      ntEntityRef, ValueDisplay, '', '', '', nil));
    { Add notations }
    for Index := 0 to Model.notationDecls.length - 1 do
      with Model.notationDecls.item(Index) as TdomASNotationDecl do
      begin
        NameDisplay := name;
        with stgNotations do
        begin
          if Cells[0, RowCount - 1] <> '' then
            RowCount := RowCount + 1;
          Cells[0, RowCount - 1] := NameDisplay;
          Cells[1, RowCount - 1] := publicId;
          Cells[2, RowCount - 1] := systemId;
        end;
      end;
    { Add entities }
    for Index := 0 to Model.entityDecls.length - 1 do
      with Model.entityDecls.item(Index) as TdomASEntityDecl do
      begin
        NameDisplay := name;
        with stgEntities do
        begin
          if notationName <> '' then
          begin
            { Unparsed entity }
            if Cells[0, RowCount - 1] <> '' then
              RowCount := RowCount + 1;
            Cells[0, RowCount - 1] := NameDisplay;
            Cells[1, RowCount - 1] := publicId;
            Cells[2, RowCount - 1] := systemId;
            Cells[3, RowCount - 1] := notationName;
          end
          else
          begin
            if entityType = AS_INTERNAL_ENTITY then
              { Internal parsed entity }
              ValueDisplay := replacementText
            else
            begin
              { External parsed entity }
              if publicId <> '' then
                ValueDisplay := 'PUBLIC ' + publicId + ' ' + systemId
              else
                ValueDisplay := 'SYSTEM ' + systemId;
            end;
            AddNodeInfo(ModelNode, NameDisplay, TNodeInfo.Create(
              ntEntityRef, NameDisplay, '', '', ValueDisplay, nil));
          end;
        end;
      end;
    { Add element and attribute declarations }
    Attribs := TStringList.Create;
    try
      for Index := 0 to Model.elementDecls.length - 1 do
        with Model.elementDecls.item(Index) as TdomASElementDecl do
        begin
          AddNodeInfo(ModelNode, name, TNodeInfo.Create(
            ntElementDecl, name, '', '',
            CompileContentModel(contentModel, contentType), nil));
          NameDisplay := name;
          if attributeDecls.length > 0 then
          begin
            Attribs.Clear;
            for Index2 := 0 to attributeDecls.length - 1 do
              with attributeDecls.item(Index2) as TdomASAttributeDecl do
              begin
                if enumAttr.count > 0 then
                begin
                  ValueDisplay := '';
                  for Index3 := 0 to enumAttr.count - 1 do
                    ValueDisplay := ValueDisplay + '|' + enumAttr[Index3];
                  ValueDisplay := '(' +
                    Copy(ValueDisplay, 2, Length(ValueDisplay) - 1) + ')';
                end
                else
                  ValueDisplay := AttrTypes[attrType];
                Attribs.Values[name] := ValueDisplay + ' ' +
                  ValueConstraints[constraintType] + ' ' + attrValue;
              end;
            AddNodeInfo(ModelNode, NameDisplay, TNodeInfo.Create(
              ntAttributeDecl, NameDisplay, '', '', '', Attribs));
          end;
        end;
    finally
      Attribs.Free;
    end;
  end;

  { Add current node to the treeview and then recurse through children }
  procedure AddNodeToTree(Node: TdomNode; TreeParent: TTreeNode);
  var
    Index: Integer;
    NameDisplay: string;
    NewNode: TTreeNode;
    Attribs: TStringList;
  begin
    { Generate name for display in the tree }
    if Node.nodeType in [ntText_Node, ntComment_Node, ntCDATA_Section_Node] then
    begin
      if Length(Node.nodeValue) > 20 then
        NameDisplay := Copy(Node.nodeValue, 1, 17) + '...'
      else
        NameDisplay := Node.nodeValue;
    end
    else
      NameDisplay := Node.nodeName;
    { Create storage for later display of node values }
    case Node.NodeType of
      ntElement_Node:
        with Node as TdomElement do
        begin
          Attribs := TStringList.Create;
          try
            { Add list of attributes and their values }
            for Index := 0 to Node.attributes.length - 1 do
              with Node.attributes.item(Index) do
                Attribs.Values[NodeName] := nodeValue;
            NewNode := AddNodeInfo(TreeParent, NameDisplay, TNodeInfo.Create(
              ntElement, tagName, namespaceURI, localName, '', Attribs));
          finally
            Attribs.Free;
          end;
        end;
      ntText_Node:
        with Node as TdomText do
          NewNode := AddNodeInfo(TreeParent, NameDisplay, TNodeInfo.Create(
            ntText, '', '', '', data, nil));
      ntCDATA_Section_Node:
        with Node as TdomCDATASection do
          NewNode := AddNodeInfo(TreeParent, NameDisplay, TNodeInfo.Create(
            ntCData, '', '', '', data, nil));
      ntEntity_Reference_Node:
        NewNode := AddNodeInfo(TreeParent, NameDisplay, TNodeInfo.Create(
          ntEntityRef, Node.nodeName, '', '', '', nil));
      ntProcessing_Instruction_Node:
        with Node as TdomProcessingInstruction do
          NewNode := AddNodeInfo(TreeParent, NameDisplay, TNodeInfo.Create(
            ntInstruction, target, '', '', data, nil));
      ntComment_Node:
        with Node as TdomComment do
          NewNode := AddNodeInfo(TreeParent, NameDisplay, TNodeInfo.Create(
            ntComment, '', '', '', data, nil));
      ntDocument_Node:
        with Node as TdomDocument do
        begin
          edtSystemId.Text := FileName;
          edtVersion.Text  := version;
          edtEncoding.Text := encoding;
          case standalone of
            STANDALONE_YES:         cbxStandAlone.State := cbChecked;
            STANDALONE_NO:          cbxStandAlone.State := cbUnchecked;
            STANDALONE_UNSPECIFIED: cbxStandAlone.State := cbGrayed;
          end;
          NewNode := AddNodeInfo(TreeParent, XMLDocDesc, TNodeInfo.Create(
            ntDocument, XMLDocDesc, '', '', '', nil));
        end;
      ntDocument_Type_Decl_Node:
        with Node as TdomDocumentTypeDecl do
        begin
          edtDocType.Text := name;
          NewNode := AddNodeInfo(TreeParent, DTDDesc, TNodeInfo.Create(
            ntDTD, DTDDesc, '', '', name, nil));
          { Add abstract schema models }
          for Index := 0 to ownerDocument.ASModels.length - 1 do
            ProcessASModel(ownerDocument.ASModels[Index], NewNode,
              ownerDocument.ASModels[Index].location = ownerDocument.systemId);
        end;
      else
      begin
        NewNode := TreeParent;
        OutputDebugString(PChar(NameDisplay + ' ' +
          IntToStr(Ord(Node.nodeType)) + ' ' +
          IntToStr(Node.childNodes.length)));
      end;
    end;
    { And recurse through children }
    if Node.hasChildNodes then
      for Index := 0 to Node.childNodes.length - 1 do
        AddNodeToTree(Node.childNodes.item(Index), NewNode);
  end;

begin
  Screen.Cursor := crHourGlass;
  try
    pgcDetails.ActivePage := tshDocument;
    { Initialise document-wide details for display }
    InitDocumentDetails;
    { Load the source document }
    memSource.Lines.LoadFromFile(Filename);
    dlgOpen.Filename := Filename;
    trvXML.Items.BeginUpdate;
    try
      domImplementation.Clear;
      { Set parser options }
      xmlParser.KeepCDATASections := mniKeepCDATASections.Checked;
      xmlParser.KeepComments      := mniKeepComments.Checked;
      xmlParser.KeepEntityRefs    := mniKeepEntityReferences.Checked;
      xmlDomReader.SuppressXmlns  := mniSuppressNamespaceDeclarations.Checked;
      { Create the new document }
      xmlDoc := domImplementation.createDocument('dummy', nil);
      xmlDoc.clear;
      { Parse the file and then handle namespaces }
      xmlDocBuilder.referenceNode := xmlDoc;
      xmlDomReader.parse(xmlParser.fileToDom(Filename));
      { Parse the DTD/schema }
      xmlDoc.prepareASModels;
      { And expand entity references within the hierarchy }
      xmlDoc.resolveEntityReferences(
        ResolveOption[mniReplaceEntityReferences.Checked]);
      { Add the structure to the tree view }
      AddNodeToTree(xmlDoc, nil);
      { Validate if requested }
      if mniValidate.Checked then
        xmlDoc.validate(ResolveOption[mniReplaceEntityReferences.Checked]);
      trvXML.Items[0].Expand(False);
    finally
      trvXML.Items.EndUpdate;
    end;
  finally
    Screen.Cursor := crDefault;
  end;
end;

{ Resolve external path - for local references }
procedure TfrmXMLViewer.xmlParserExternalEntity(Sender: TObject;
  const ParentSystemId: WideString; var PublicId, SystemId: WideString;
  var Stream: TStream; var Action: TXmlParserAction);
begin
  if (SystemId <> '') and (Pos('://', SystemId) = 0) then
    try
      SystemId := ExtractFileDir(ParentSystemId) + '\' + SystemID;
      Stream   := TFileStream.Create(SystemId, fmOpenRead);
      Action   := paOK;
    except
      Action   := paFail;
    end;
end;

{ Select a file to open }
procedure TfrmXMLViewer.mniOpenClick(Sender: TObject);
begin
  with dlgOpen do
    if Execute then
      LoadDoc(Filename);
end;

{ Toggle parser flags }
procedure TfrmXMLViewer.mniOptionClick(Sender: TObject);
begin
  with Sender as TMenuItem do
    Checked := not Checked;
end;

{ Exit the application }
procedure TfrmXMLViewer.mniExitClick(Sender: TObject);
begin
  Close;
end;

{ Expand all nodes below the current one }
procedure TfrmXMLViewer.mniExpandAllClick(Sender: TObject);
begin
  if Assigned(trvXML.Selected) then
    trvXML.Selected.Expand(True);
end;

{ Collapse all nodes below the current one }
procedure TfrmXMLViewer.mniCollapseAllClick(Sender: TObject);
begin
  if Assigned(trvXML.Selected) then
    trvXML.Selected.Collapse(True);
end;

{ Toggle between structure and source }
procedure TfrmXMLViewer.mniViewSourceClick(Sender: TObject);
begin
  mniViewSource.Checked := not mniViewSource.Checked;
  if mniViewSource.Checked then
    pgcMain.ActivePage := tshSource
  else
    pgcMain.ActivePage := tshStructure;
end;

{ Display details for the selected XML element }
procedure TfrmXMLViewer.trvXMLChange(Sender: TObject; Node: TTreeNode);
var
  Index: Integer;
begin
  with TNodeInfo(trvXML.Selected.Data) do
    case NodeType of
      ntDocument:
        { Show document details, including entities and notations }
        pgcDetails.ActivePage := tshDocument;
      ntElement, ntAttributeDecl:
        begin
          { Show element details, including attributes }
          pgcDetails.ActivePage := tshElement;
          edtURI.Text           := NamespaceURI;
          if LocalName <> '' then
            edtLocalName.Text   := LocalName
          else
            edtLocalName.Text   := Name;
          with stgAttributes do
          begin
            if NodeType = ntElement then
              Cells[1, 0] := ValueDesc
            else
              Cells[1, 0] := DefinitionDesc;
            if Attributes.Count = 0 then
              RowCount := 2
            else
              RowCount := Attributes.Count + 1;
            Rows[1].Clear;
            for Index := 0 to Attributes.Count - 1 do
            begin
              Cells[0, Index + 1] := Attributes.Names[Index];
              Cells[1, Index + 1] :=
                Attributes.Values[Attributes.Names[Index]];
            end;
          end;
        end;
      else
        begin
          { Show details for other nodes - text type }
          pgcDetails.ActivePage := tshText;
          memText.Lines.Text    := Value;
          case NodeType of
            ntComment:     lblNodeType.Caption := CommentDesc;
            ntInstruction: lblNodeType.Caption := InstructionDesc;
            ntEntityRef:   lblNodeType.Caption := EntityRefDesc;
            ntElementDecl: lblNodeType.Caption := ElementDeclDesc;
            else           lblNodeType.Caption := TextDesc;
          end;
        end;
    end;
end;

end.
