unit TextGen1;

{
  Demonstrate the generation of an XML document from a database as text.
  Requires 'movie-watcher' alias to be set up in BDE.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written February 16, 1999.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, DBTables, StdCtrls, ExtCtrls, CommonXML;

type
  TfrmTextXML = class(TForm)
    memXML: TMemo;
    pnlButtons: TPanel;
      btnGenerate: TButton;
      btnSave: TButton;
    dlgSave: TSaveDialog;
    procedure btnGenerateClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
  private
  public
  end;

var
  frmTextXML: TfrmTextXML;

implementation

{$R *.DFM}

resourcestring
  { XML document fragments }
  CinemasClosing    = '</cinemas>'#13;
  CinemasOpening    = '<cinemas>'#13;
  CinemaTag         = '  <cinema id="C%d">'#13 +
                      '    <name>%s</name>'#13 +
                      '    <phone>%s</phone>'#13 +
                      '    <address>%s</address>'#13 +
                      '    <directions>%s</directions>'#13 +
                      '    <facilities>%s%s</facilities>'#13 +
                      '    <pricing>'#13 +
                      '%s' +
                      '    </pricing>'#13 +
                      '  </cinema>'#13;
  DocumentOpening   = '<movie-watcher>'#13;
  DocumentClosing   = '</movie-watcher>'#13;
  EntityDecl        = '<!ENTITY %s SYSTEM "%s" NDATA %s>'#13;
  MovieIdFormat     = 'M%d';
  MoviesClosing     = '</movies>'#13;
  MoviesOpening     = '<movies>'#13;
  MovieTag          = '  <movie id="M%d" rating="%s"%s%s>'#13 +
                      '    <name>%s</name>'#13 +
                      '    <length>%d</length>'#13 +
                      '    <director>%s</director>'#13 +
                      '    <starring>%s</starring>'#13 +
                      '    <synopsis><![CDATA[%s]]></synopsis>'#13 +
                      '  </movie>'#13;
  PricingTag        = '      <prices id="P%d">'#13 +
                      '        <name>%s</name>'#13 +
                      '        <period>%s</period>'#13 +
                      '        <adult>%f</adult>'#13 +
                      '        <child>%f</child>'#13 +
                      '        <discount>%f</discount>'#13 +
                      '      </prices>'#13;
  ScreeningsClosing = '</screenings>'#13;
  ScreeningsOpening = '<screenings>'#13;
  ScreeningTag      = '  <screening movie-id="M%d" cinema-id="C%d">'#13 +
                      '    <start-date>%s</start-date>'#13 +
                      '    <end-date>%s</end-date>'#13 +
                      '    <features>'#13 +
                      '      <digital-sound>%s</digital-sound>'#13 +
                      '    </features>'#13 +
                      '    <restrictions>%s</restrictions>'#13 +
                      '    <sessions>'#13 +
                      '%s' +
                      '    </sessions>'#13 +
                      '  </screening>'#13;
  SessionTag        = '      <session price-id="P%d">%s</session>'#13;
  StarTag           = '<star>%s</star>';
  TimeFormat        = 'h:nn AM/PM';
  XMLProlog         = '<?xml %s?>'#13 +
                      '<!DOCTYPE %s SYSTEM "%s" ['#13 +
                      '<!NOTATION HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN">'#13 +
                      '<!NOTATION JPEG SYSTEM "iview.exe">'#13 +
                      '%s]>'#13 +
                      '<!-- %s -->'#13 +
                      '<?%s %s?>'#13;

{ Generate the XML document as text }
procedure TfrmTextXML.btnGenerateClick(Sender: TObject);

  { Include attribute entity reference only if present }
  function GetOptEntityAttr(Sender: TField; Reference: string): string;
  begin
    if Sender.AsString <> '' then
      Result := ' ' + ModifyName(Sender.FieldName) + '="' + Reference + '"'
    else
      Result := '';
  end;

  { Include empty field tag only if flag in DB set }
  function GetOptElement(Sender: TField): string;
  begin
    if Sender.AsBoolean then
      Result := '<' + ModifyName(Sender.FieldName) + '/>'
    else
      Result := '';
  end;

  { Compile entities for the movie }
  function GenerateEntities: string;
  begin
    Result := '';
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        if FieldByName(LogoURLField).AsString <> '' then
          Result := Result + Format(EntityDecl, [Format(MovieIdFormat,
            [FieldByName(MovieIdField).AsInteger]) + 'Logo',
            FieldByName(LogoURLField).AsString, JPEGType]);
        if FieldByName(URLField).AsString <> '' then
          Result := Result + Format(EntityDecl, [Format(MovieIdFormat,
            [FieldByName(MovieIdField).AsInteger]) + 'Url',
            FieldByName(URLField).AsString, HTMLType]);
        Next;
      end;
    end;
  end;

  { Compile elements for the stars of the movie }
  function GenerateStars: string;
  begin
    Result := '';
    with datCommonXML.qryStars do
    begin
      First;
      while not EOF do
      begin
        Result := Result + Format(StarTag, [FieldByName(StarField).AsString]);
        Next;
      end;
    end;
  end;

  { Generate elements for each movie }
  function GenerateMovies: string;
  var
    MovieId: Integer;
  begin
    Result := MoviesOpening;
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        MovieId := FieldByName(MovieIdField).AsInteger;
        Result := Result + Format(MovieTag, [MovieId,
          FieldByName(RatingField).AsString,
          GetOptEntityAttr(FieldByName(LogoURLField),
            Format(MovieIdFormat, [MovieId]) + 'Logo'),
          GetOptEntityAttr(FieldByName(URLField),
            Format(MovieIdFormat, [MovieId]) + 'Url'),
          FieldByName(NameField).AsString, FieldByName(LengthField).AsInteger,
          FieldByName(DirectorField).AsString, GenerateStars,
          FieldByName(SynopsisField).AsString]);
        Next;
      end;
    end;
    Result := Result + MoviesClosing;
  end;

  { Compile elements for the pricing schemes }
  function GeneratePricing: string;
  begin
    Result := '';
    with datCommonXML.qryPricing do
    begin
      First;
      while not EOF do
      begin
        Result := Result + Format(PricingTag,
          [FieldByName(PricingIdField).AsInteger,
          FieldByName(NameField).AsString, FieldByName(PeriodField).AsString,
          FieldByName(AdultField).AsFloat, FieldByName(ChildField).AsFloat,
          FieldByName(DiscountField).AsFloat]);
        Next;
      end;
    end;
  end;

  { Generate elements for each cinema }
  function GenerateCinemas: string;
  begin
    Result := CinemasOpening;
    with datCommonXML.qryCinema do
    begin
      First;
      while not EOF do
      begin
        Result := Result + Format(CinemaTag,
          [FieldByName(CinemaIdField).AsInteger,
          FieldByName(NameField).AsString,
          FieldByName(PhoneField).AsString,
          FieldByName(AddressField).AsString,
          FieldByName(DirectionsField).AsString,
          GetOptElement(FieldByName(CandyBarField)),
          GetOptElement(FieldByName(DisabledField)), GeneratePricing]);
        Next;
      end;
    end;
    Result := Result + CinemasClosing;
  end;

  { Compile elements for the sessions for each screening }
  function GenerateSessions: string;
  begin
    Result := '';
    with datCommonXML.qrySessions do
    begin
      First;
      while not EOF do
      begin
        Result := Result + Format(SessionTag,
          [FieldByName(PricingIdField).AsInteger,
          FormatDateTime(TimeFormat, FieldByName(TimeField).AsDateTime)]);
        Next;
      end;
    end;
  end;

  { Generate elements for each screening }
  function GenerateScreenings: string;
  begin
    Result := ScreeningsOpening;
    with datCommonXML.qryScreening do
    begin
      First;
      while not EOF do
      begin
        Result := Result + Format(ScreeningTag,
          [FieldByName(MovieIdField).AsInteger,
          FieldByName(CinemaIdField).AsInteger,
          FieldByName(StartDateField).AsString,
          FieldByName(EndDateField).AsString,
          FieldByName(DigSoundField).AsString,
          GetOptElement(FieldByName(NoPassesField)), GenerateSessions]);
        Next;
      end;
    end;
    Result := Result + ScreeningsClosing;
  end;

begin
  Screen.Cursor       := crHourGlass;
  btnGenerate.Enabled := False;
  try
    memXML.Lines.Text := Format(XMLProlog, [XMLPrologAttrs, MovieWatcherTag,
      XMLDTDFile, GenerateEntities, XMLComment, XMLStyleTag, XMLStyleAttrs]) +
      DocumentOpening + GenerateMovies + GenerateCinemas +
      GenerateScreenings + DocumentClosing;
  finally
    btnGenerate.Enabled := True;
    Screen.Cursor       := crDefault;
  end;
end;

{ Save the generated XML }
procedure TfrmTextXML.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
    if Execute then
      memXML.Lines.SaveToFile(Filename);
end;

end.
