unit CSXMLDOM;

{
  Adapter for CUESoft XML for Delphi's XML DOM interfaces.

  Written by Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 21 August 2002.
}

interface

uses
  Windows, SysUtils, Classes, XMLDOM, XmlObjModel, XslProcessor, XslFilterHTML;

const
  sCUEXml = 'CUEXml';  { Do not localize }

type

{ ICSNodeRef }

  ICSNodeRef = interface
    ['{7A53E169-3107-4DEB-A6A6-ADED0F25080C}']
    function GetXMLDOMNode: TXmlNode;
  end;

{ TCSDOMInterface }

  TCSDOMInterface = class(TInterfacedObject)
  end;

{ TCSDOMImplementation }

  TCSDOMImplementation = class(TCSDOMInterface, IDOMImplementation)
  private
    FDOMImpl: TXmlDOMImplementation;
  protected
    { IDOMImplementation }
    function hasFeature(const feature, version: DOMString): WordBool;
    function createDocumentType(const qualifiedName, publicId,
      systemId: DOMString): IDOMDocumentType; safecall;
    function createDocument(const namespaceURI, qualifiedName: DOMString;
      doctype: IDOMDocumentType): IDOMDocument; safecall;
  public
    constructor Create(DOMImplementation: TXmlDomImplementation);
    destructor Destroy; override;
    property DOMImpl: TXmlDOMImplementation read FDOMImpl;
  end;

{ TCSDOMNode }

  TCSDOMNodeClass = class of TCSDOMNode;

  TCSDOMDocument = class;

  TCSDOMNode = class(TCSDOMInterface,
    ICSNodeRef, IDOMNode, IDOMNodeEx, IDOMNodeSelect)
  private
    FAttributes: IDOMNamedNodeMap;
    FChildNodes: IDOMNodeList;
    FNode: TXmlNode;
    FOwnerDocument: TCSDOMDocument;
  protected
    { IXpNodeRef }
    function GetXMLDOMNode: TXmlNode;
    { IDOMNode }
    function get_nodeName: DOMString; virtual; safecall;
    function get_nodeValue: DOMString; safecall;
    procedure set_nodeValue(value: DOMString);
    function get_nodeType: DOMNodeType; virtual; safecall;
    function get_parentNode: IDOMNode; safecall;
    function get_childNodes: IDOMNodeList; virtual; safecall;
    function get_firstChild: IDOMNode; safecall;
    function get_lastChild: IDOMNode; safecall;
    function get_previousSibling: IDOMNode; safecall;
    function get_nextSibling: IDOMNode; safecall;
    function get_attributes: IDOMNamedNodeMap; safecall;
    function get_ownerDocument: IDOMDocument; safecall;
    function get_namespaceURI: DOMString; safecall;
    function get_prefix: DOMString; safecall;
    function get_localName: DOMString; safecall;
    function insertBefore(const newChild, refChild: IDOMNode): IDOMNode;
      safecall;
    function replaceChild(const newChild, oldChild: IDOMNode): IDOMNode;
      safecall;
    function removeChild(const childNode: IDOMNode): IDOMNode; safecall;
    function appendChild(const newChild: IDOMNode): IDOMNode; safecall;
    function hasChildNodes: WordBool; virtual; safecall;
    function cloneNode(deep: WordBool): IDOMNode; safecall;
    procedure normalize;
    function supports(const feature, version: DOMString): WordBool;
    { IDOMNodeEx }
    function get_text: DOMString; safecall;
    function get_xml: DOMString; safecall;
    procedure set_text(const Value: DOMString); safecall;
    procedure transformNode(const stylesheet: IDOMNode; var output: WideString);
      overload;
    procedure transformNode(const stylesheet: IDOMNode;
      const output: IDOMDocument); overload;
    { IDOMNodeSelect }
    function selectNode(const nodePath: WideString): IDOMNode; safecall;
    function selectNodes(const nodePath: WideString): IDOMNodeList; safecall;
  public
    constructor Create(const Node: TXmlNode; const Document: TCSDOMDocument);
      virtual;
    property Node: TXmlNode read FNode;
  end;

{ TCSDOMNodeList }

  TCSDOMNodeList = class(TCSDOMInterface, IDOMNodeList)
  private
    FNodeList: TXmlNodeList;
    FOwnerDocument: TCSDOMDocument;
  protected
    { IDOMNodeList }
    function get_item(index: Integer): IDOMNode; safecall;
    function get_length: Integer; safecall;
  public
    constructor Create(const NodeList: TXmlNodeList;
      const Document: TCSDOMDocument);
    property NodeList: TXmlNodeList read FNodeList;
  end;

{ TCSDOMNamedNodeMap }

  TCSDOMNamedNodeMap = class(TCSDOMInterface, IDOMNamedNodeMap)
  private
    FNamedNodeMap: TXmlNamedNodeMap;
    FOwnerDocument: TCSDOMDocument;
  protected
    { IDOMNamedNodeMap }
    function get_item(index: Integer): IDOMNode; safecall;
    function get_length: Integer;
    function getNamedItem(const name: DOMString): IDOMNode; safecall;
    function setNamedItem(const newItem: IDOMNode): IDOMNode; safecall;
    function removeNamedItem(const name: DOMString): IDOMNode; safecall;
    function getNamedItemNS(const namespaceURI, localName: DOMString):
      IDOMNode; safecall;
    function setNamedItemNS(const arg: IDOMNode): IDOMNode; safecall;
    function removeNamedItemNS(const namespaceURI, localName: DOMString):
      IDOMNode; safecall;
  public
    constructor Create(const NamedNodeMap: TXmlNamedNodeMap;
      const Document: TCSDOMDocument);
    property NamedNodeMap: TXmlNamedNodeMap read FNamedNodeMap;
  end;

{ TCSDOMCharacterData }

  TCSDOMCharacterData = class(TCSDOMNode, IDOMCharacterData)
  private
    function GetCharacterData: TXmlCharacterData;
  protected
    { IDOMCharacterData }
    function get_data: DOMString;
    procedure set_data(const data: DOMString);
    function get_length: Integer;
    function substringData(offset, count: Integer): DOMString;
    procedure appendData(const data: DOMString);
    procedure insertData(offset: Integer; const data: DOMString);
    procedure deleteData(offset, count: Integer);
    procedure replaceData(offset, count: Integer; const data: DOMString);
  public
    property CharacterData: TXmlCharacterData read GetCharacterData;
  end;

{ TCSDOMAttr }

  TCSDOMAttr = class(TCSDOMNode, IDOMAttr)
  private
    function GetAttribute: TXmlAttribute;
  protected
    { Property Get/Set }
    function get_name: DOMString;
    function get_specified: WordBool;
    function get_value: DOMString;
    procedure set_value(const attributeValue: DOMString);
    function get_ownerElement: IDOMElement;
    { Properties }
    property name: DOMString read get_name;
    property specified: WordBool read get_specified;
    property value: DOMString read get_value write set_value;
    property ownerElement: IDOMElement read get_ownerElement;
  public
    property Attribute: TXmlAttribute read GetAttribute;
  end;

{ TCSDOMElement }

  TCSDOMElement = class(TCSDOMNode, IDOMElement)
  private
    function GetElement: TXmlElement;
  protected
    { IDOMElement }
    function get_tagName: DOMString; safecall;
    function getAttribute(const name: DOMString): DOMString; safecall;
    procedure setAttribute(const name, value: DOMString);
    procedure removeAttribute(const name: DOMString);
    function getAttributeNode(const name: DOMString): IDOMAttr; safecall;
    function setAttributeNode(const newAttr: IDOMAttr): IDOMAttr; safecall;
    function removeAttributeNode(const oldAttr: IDOMAttr): IDOMAttr; safecall;
    function getElementsByTagName(const name: DOMString): IDOMNodeList; safecall;
    function getAttributeNS(const namespaceURI, localName: DOMString): DOMString; safecall;
    procedure setAttributeNS(const namespaceURI, qualifiedName, value: DOMString);
    procedure removeAttributeNS(const namespaceURI, localName: DOMString);
    function getAttributeNodeNS(const namespaceURI, localName: DOMString): IDOMAttr; safecall;
    function setAttributeNodeNS(const newAttr: IDOMAttr): IDOMAttr; safecall;
    function getElementsByTagNameNS(const namespaceURI,
      localName: DOMString): IDOMNodeList; safecall;
    function hasAttribute(const name: DOMString): WordBool; safecall;
    function hasAttributeNS(const namespaceURI, localName: DOMString): WordBool;
    procedure normalize;
  public
    property Element: TXmlElement read GetElement;
  end;

{ TCSDOMText }

  TCSDOMText = class(TCSDOMCharacterData, IDOMText)
  protected
    function splitText(offset: Integer): IDOMText; safecall;
  end;

{ TCSDOMComment }

  TCSDOMComment = class(TCSDOMCharacterData, IDOMComment)
  end;

{ TCSDOMCDATASection }

  TCSDOMCDATASection = class(TCSDOMText, IDOMCDATASection)
  end;

{ TCSDOMDocumentType }

  TCSDOMDocumentType = class(TCSDOMNode, IDOMDocumentType)
  private
    function GetDocumentType: TXmlDocumentType;
  protected
    { IDOMDocumentType }
    function get_name: DOMString; safecall;
    function get_entities: IDOMNamedNodeMap; safecall;
    function get_notations: IDOMNamedNodeMap; safecall;
    function get_publicId: DOMString; safecall;
    function get_systemId: DOMString; safecall;
    function get_internalSubset: DOMString; safecall;
  public
    property DocumentType: TXmlDocumentType read GetDocumentType;
  end;

{ TCSDOMNotation }

  TCSDOMNotation = class(TCSDOMNode, IDOMNotation)
  private
    function GetNotation: TXmlNotation;
  protected
    { IDOMNotation }
    function get_publicId: DOMString; safecall;
    function get_systemId: DOMString; safecall;
  public
    property Notation: TXmlNotation read GetNotation;
  end;

{ TCSDOMEntity }

  TCSDOMEntity = class(TCSDOMNode, IDOMEntity)
  private
    function GetEntity: TXmlEntity;
  protected
    { IDOMEntity }
    function get_publicId: DOMString; safecall;
    function get_systemId: DOMString; safecall;
    function get_notationName: DOMString; safecall;
  public
    property Entity: TXmlEntity read GetEntity;
  end;

{ TCSDOMEntityReference }

  TCSDOMEntityReference = class(TCSDOMNode, IDOMEntityReference)
  end;

{ TCSDOMProcessingInstruction }

  TCSDOMProcessingInstruction = class(TCSDOMNode, IDOMProcessingInstruction)
  private
    function GetProcessingInstruction: TXmlProcessingInstruction;
  protected
    { IDOMProcessingInstruction }
    function get_target: DOMString; safecall;
    function get_data: DOMString; safecall;
    procedure set_data(const value: DOMString);
  public
    property ProcessingInstruction: TXmlProcessingInstruction
      read GetProcessingInstruction;
  end;

{ TCSDOMDocumentFragment }

  TCSDOMDocumentFragment = class(TCSDOMNode, IDOMDocumentFragment)
  end;

{ TCSDOMDocument }

  TCSDOMDocument = class(TCSDOMNode, IDOMDocument, IDOMParseOptions,
    IDOMPersist, IDOMParseError, IDOMXMLProlog)
  private
    FEncoding: DOMString;
    FGotProlog: Boolean;
    FStandalone: DOMString;
    FVersion: DOMString;
    FXMLObjModel: TXmlObjModel;
    function GetDocument: TXmlDocument;
  protected
    procedure InvalidDocument(oOwner: TObject; wCode: Integer; oNode: TXmlNode;
      var bStop: Boolean);
    procedure GetProlog;
    { IDOMNode }
    function get_nodeName: DOMString; override; safecall;
    { IDOMDocument }
    function get_doctype: IDOMDocumentType; safecall;
    function get_domImplementation: IDOMImplementation; safecall;
    function get_documentElement: IDOMElement; safecall;
    procedure set_documentElement(const IDOMElement: IDOMElement);
    function createElement(const tagName: DOMString): IDOMElement; safecall;
    function createDocumentFragment: IDOMDocumentFragment; safecall;
    function createTextNode(const data: DOMString): IDOMText; safecall;
    function createComment(const data: DOMString): IDOMComment; safecall;
    function createCDATASection(const data: DOMString): IDOMCDATASection; safecall;
    function createProcessingInstruction(const target,
      data: DOMString): IDOMProcessingInstruction; safecall;
    function createAttribute(const name: DOMString): IDOMAttr; safecall;
    function createEntityReference(const name: DOMString): IDOMEntityReference; safecall;
    function getElementsByTagName(const tagName: DOMString): IDOMNodeList; safecall;
    function importNode(importedNode: IDOMNode; deep: WordBool): IDOMNode; safecall;
    function createElementNS(const namespaceURI,
      qualifiedName: DOMString): IDOMElement; safecall;
    function createAttributeNS(const namespaceURI,
      qualifiedName: DOMString): IDOMAttr; safecall;
    function getElementsByTagNameNS(const namespaceURI,
      localName: DOMString): IDOMNodeList; safecall;
    function getElementById(const elementId: DOMString): IDOMElement;
    { IDOMParseOptions }
    function get_async: Boolean;
    function get_preserveWhiteSpace: Boolean;
    function get_resolveExternals: Boolean;
    function get_validate: Boolean;
    procedure set_async(Value: Boolean);
    procedure set_preserveWhiteSpace(Value: Boolean);
    procedure set_resolveExternals(Value: Boolean);
    procedure set_validate(Value: Boolean);
    { IDOMPersist }
    function get_xml: DOMString; safecall;
    function asyncLoadState: Integer; safecall;
    function load(source: OleVariant): WordBool; safecall;
    function loadFromStream(const stream: TStream): WordBool; safecall;
    function loadxml(const Value: DOMString): WordBool; safecall;
    procedure save(destination: OleVariant); safecall;
    procedure saveToStream(const stream: TStream); safecall;
    procedure set_OnAsyncLoad(const Sender: TObject;
      EventHandler: TAsyncEventHandler); safecall;
    { IDOMParseError }
    function get_errorCode: Integer;
    function get_url: WideString; safecall;
    function get_reason: WideString; safecall;
    function get_srcText: WideString; safecall;
    function get_line: Integer;
    function get_linepos: Integer;
    function get_filepos: Integer;
    { IDOMXMLProlog }
    function get_Encoding: DOMString; safecall;
    function get_Standalone: DOMString; safecall;
    function get_Version: DOMString; safecall;
    procedure set_Encoding(const Value: DOMString); safecall;
    procedure set_Standalone(const Value: DOMString); safecall;
    procedure set_Version(const Value: DOMString); safecall;
  public
    constructor Create(const Node: TXmlNode; const Document: TCSDOMDocument);
      override;
    destructor Destroy; override;
    property Document: TXmlDocument read GetDocument;
  end;

{ TCSDOMImplementationFactory }

  TCSDOMImplementationFactory = class(TDOMVendor)
  public
    function DOMImplementation: IDOMImplementation; override;
    function Description: String; override;
  end;

var
  CSXML_DOM: TCSDOMImplementationFactory;

implementation

resourcestring
  sNodeExpected = 'Node cannot be null';

{ Utility Functions -----------------------------------------------------------}

function MakeNode(const Node: TXmlNode; const Document: TCSDOMDocument):
  IDOMNode;
const
  NodeClasses: array [ELEMENT_NODE..NOTATION_NODE] of TCSDOMNodeClass =
    (TCSDOMElement, TCSDOMAttr, TCSDOMText, TCSDOMCDataSection,
     TCSDOMEntityReference, TCSDOMEntity, TCSDOMProcessingInstruction,
     TCSDOMComment, TCSDOMDocument, TCSDOMDocumentType,
     TCSDOMDocumentFragment, TCSDOMNotation);
begin
  if Assigned(Node) then
    Result := NodeClasses[Node.nodeType].Create(Node, Document)
  else
    Result := nil;
end;

function MakeNodeList(const NodeList: TXmlNodeList;
  const Document: TCSDOMDocument): IDOMNodeList;
begin
  Result := TCSDOMNodeList.Create(NodeList, Document);
end;

function MakeNamedNodeMap(const NamedNodeMap: TXmlNamedNodeMap;
  const Document: TCSDOMDocument): IDOMNamedNodeMap;
begin
  Result := TCSDOMNamedNodeMap.Create(NamedNodeMap, Document);
end;

function GetNode(const Node: IDOMNode): TXmlNode;
begin
  if not Assigned(Node) then
    raise DOMException.Create(sNodeExpected);
  Result := (Node as ICSNodeRef).GetXMLDOMNode;
end;

function FixLineBreaks(const Value: string): string;
var
  Index: Integer;
begin
  Result := Value;
  while True do
  begin
    Index := Pos(#0, Result);
    if Index = 0 then
      Exit;
    Result[Index] := #13;
  end;
end;

{ TCSDOMImplementation ------------------------------------------------------}

constructor TCSDOMImplementation.Create(
  DOMImplementation: TXmlDomImplementation);
begin
  inherited Create;
  FDOMImpl := DOMImplementation;
end;

destructor TCSDOMImplementation.Destroy;
begin
  FDOMImpl.Free;
  inherited Destroy;
end;

function TCSDOMImplementation.createDocument(const namespaceURI,
  qualifiedName: DOMString; doctype: IDOMDocumentType): IDOMDocument;
var
  Document: TCSDOMDocument;
  XML: string;
begin
  Document := TCSDOMDocument.Create(nil, nil);
  Document.FOwnerDocument := Document;
  if qualifiedName <> '' then
  begin
    XML := '<' + qualifiedName;
    if namespaceURI <> '' then
      XML := XML +
        ' xmlns:' + ExtractPrefix(qualifiedName) + '="' + namespaceURI + '"';
    XML := XML + '/>';
    Document.loadxml(XML);
  end;
  Result := Document;
end;

function TCSDOMImplementation.createDocumentType(const qualifiedName,
  publicId, systemId: DOMString): IDOMDocumentType;
begin
  DOMVendorNotSupported('createDocumentType', sCUEXml); { Do not localize }
end;

function TCSDOMImplementation.hasFeature(
  const feature, version: DOMString): WordBool;
begin
  Result := DOMImpl.hasFeature(feature, version);
end;

{ TCSDOMNode ----------------------------------------------------------------}

constructor TCSDOMNode.Create(const Node: TXmlNode;
  const Document: TCSDOMDocument);
begin
  Assert(Assigned(Node));
  FNode          := Node;
  FOwnerDocument := Document;
  inherited Create;
end;

function TCSDOMNode.appendChild(const newChild: IDOMNode): IDOMNode;
begin
  Node.appendChild(GetNode(newChild));
  Result := newChild
end;

function TCSDOMNode.cloneNode(Deep: WordBool): IDOMNode;
begin
  Result := MakeNode(Node.CloneNode(deep), FOwnerDocument);
end;

function TCSDOMNode.get_attributes: IDOMNamedNodeMap;
begin
  if not Assigned(FAttributes) and Assigned(Node.Attributes) then
    FAttributes := MakeNamedNodeMap(Node.Attributes, FOwnerDocument);
  Result := FAttributes;
end;

function TCSDOMNode.get_childNodes: IDOMNodeList;
begin
  if not Assigned(FChildNodes) then
    FChildNodes := MakeNodeList(Node.ChildNodes, FOwnerDocument);
  Result := FChildNodes;
end;

function TCSDOMNode.get_firstChild: IDOMNode;
begin
  Result := MakeNode(Node.FirstChild, FOwnerDocument);
end;

function TCSDOMNode.get_lastChild: IDOMNode;
begin
  Result := MakeNode(Node.LastChild, FOwnerDocument);
end;

function TCSDOMNode.get_localName: DOMString;
begin
  Result := Node.BaseName;
end;

function TCSDOMNode.get_namespaceURI: DOMString;
begin
  Result := Node.Namespace;
end;

function TCSDOMNode.get_nextSibling: IDOMNode;
begin
  Result := MakeNode(Node.NextSibling, FOwnerDocument);
end;

function TCSDOMNode.get_nodeName: DOMString;
begin
  Result := Node.NodeName;
end;

function TCSDOMNode.get_nodeType: DOMNodeType;
begin
  Result := Node.NodeType;
end;

function TCSDOMNode.get_nodeValue: DOMString;
begin
  Result := Node.NodeValue;
end;

function TCSDOMNode.get_ownerDocument: IDOMDocument;
begin
  Result := FOwnerDocument;
end;

function TCSDOMNode.get_parentNode: IDOMNode;
begin
  Result := MakeNode(Node.ParentNode, FOwnerDocument);
end;

function TCSDOMNode.get_prefix: DOMString;
begin
  Result := Node.Prefix;
end;

function TCSDOMNode.get_previousSibling: IDOMNode;
begin
  Result := MakeNode(Node.PreviousSibling, FOwnerDocument);
end;

function TCSDOMNode.hasChildNodes: WordBool;
begin
  Result := Node.HasChildNodes;
end;

function TCSDOMNode.insertBefore(const newChild, refChild: IDOMNode): IDOMNode;
begin
  Node.InsertBefore(GetNode(newChild), GetNode(refChild));
  Result := newChild;
end;

procedure TCSDOMNode.normalize;
begin
  if Node is TXmlElement then
    TXmlElement(Node).Normalize
  else
    DOMVendorNotSupported('normalize', sCUEXml); { Do not localize }
end;

function TCSDOMNode.removeChild(const childNode: IDOMNode): IDOMNode;
begin
  Result := MakeNode(Node.RemoveChild(GetNode(childNode)), FOwnerDocument);
end;

function TCSDOMNode.replaceChild(const newChild, oldChild: IDOMNode): IDOMNode;
begin
  Result := MakeNode(Node.ReplaceChild(GetNode(newChild), GetNode(oldChild)),
    FOwnerDocument);
end;

procedure TCSDOMNode.set_nodeValue(value: DOMString);
begin
  Node.NodeValue := value;
end;

function TCSDOMNode.supports(const feature, version: DOMString): WordBool;
begin
  Result := CSXML_DOM.DOMImplementation.hasFeature(feature, version);
end;

function TCSDOMNode.GetXMLDOMNode: TXmlNode;
begin
  Result := Node;
end;

{ IDOMNodeEx Interface }

function TCSDOMNode.get_text: DOMString;
begin
  Result := Node.Text;
end;

procedure TCSDOMNode.set_text(const Value: DOMString);
var
  Index: Integer;
begin
  for Index := Node.ChildNodes.Length - 1 downto 0 do
    Node.RemoveChild(Node.ChildNodes.Item(0));
  Node.AppendChild(Node.OwnerDocument.CreateTextNode(Value));
end;

function TCSDOMNode.get_xml: DOMString;
begin
  Result := Node.XmlDocument;
  if Result[Length(Result)] = #0 then
    Delete(Result, Length(Result), 1);
end;

procedure TCSDOMNode.transformNode(const stylesheet: IDOMNode;
  var output: WideString);
var
  XSLProc: TXslProcessor;
begin
  XSLProc := TXslProcessor.Create(nil);
  with XSLProc do
    try
      try
        Filter      := TXslFilterHTML.Create(XSLProc);
        RaiseErrors := True;
        StyleData   := GetNode(stylesheet).XmlDocument;
        XmlObjModel := FOwnerDocument.FXMLObjModel;
        ApplyStyle;
        output      := FixLineBreaks(TXslFilterHTML(Filter).HTMLDocument);
      except on E: Exception do
        raise DOMException.Create(E.Message);
      end;
    finally
      Free;
    end;
end;

procedure TCSDOMNode.transformNode(const stylesheet: IDOMNode;
  const output: IDOMDocument);
var
  XSLOutput: WideString;
  XMLObjModel: TXmlObjModel;
begin
  transformNode(stylesheet, XSLOutput);
  XMLObjModel := TXmlObjModel.Create(nil);
  with XMLObjModel do
    try
      LoadMemory(@XSLOutput[1]);
      output.importNode(MakeNode(Document, FOwnerDocument), True);
    finally
      Free;
    end;
end;

{ IDOMNodeSelect }

function TCSDOMNode.selectNode(const nodePath: WideString): IDOMNode;
begin
  if (Node is TXmlDocument) and (TXmlDocument(Node).DocumentElement <> nil) then
    Result := MakeNode(TXmlDocument(Node).DocumentElement.
      SelectSingleNode(nodePath), FOwnerDocument)
  else if Node is TXmlElement then
    Result := MakeNode(TXmlElement(Node).SelectSingleNode(nodePath),
      FOwnerDocument)
  else
    DOMVendorNotSupported('selectNode', sCUEXml); { Do not localize }
end;

function TCSDOMNode.selectNodes(const nodePath: WideString): IDOMNodeList;
begin
  if (Node is TXmlDocument) and (TXmlDocument(Node).DocumentElement <> nil) then
    Result := MakeNodeList(TXmlDocument(Node).DocumentElement.
      SelectNodes(nodePath), FOwnerDocument)
  else if Node is TXmlElement then
    Result := MakeNodeList(TXmlElement(Node).SelectNodes(nodePath),
      FOwnerDocument)
  else
    DOMVendorNotSupported('selectNodes', sCUEXml); { Do not localize }
end;

{ TCSDOMNodeList ------------------------------------------------------------}

constructor TCSDOMNodeList.Create(const NodeList: TXmlNodeList;
      const Document: TCSDOMDocument);
begin
  inherited Create;
  FNodeList      := NodeList;
  FOwnerDocument := Document;
end;

function TCSDOMNodeList.get_item(index: Integer): IDOMNode;
begin
  Result := MakeNode(NodeList.Item(index), FOwnerDocument);
end;

function TCSDOMNodeList.get_length: Integer;
begin
  Result := NodeList.Length;
end;

{ TCSDOMNamedNodeMap --------------------------------------------------------}

constructor TCSDOMNamedNodeMap.Create(const NamedNodeMap: TXmlNamedNodeMap;
      const Document: TCSDOMDocument);
begin
  inherited Create;
  FNamedNodeMap  := NamedNodeMap;
  FOwnerDocument := Document;
end;

function TCSDOMNamedNodeMap.get_item(index: Integer): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.Item(index), FOwnerDocument);
end;

function TCSDOMNamedNodeMap.get_length: Integer;
begin
  Result := NamedNodeMap.Length;
end;

function TCSDOMNamedNodeMap.getNamedItem(const name: DOMString): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.GetNamedItem(name), FOwnerDocument);
end;

function TCSDOMNamedNodeMap.getNamedItemNS(
  const namespaceURI, localName: DOMString): IDOMNode;
var
  Index: Integer;
begin
  Result := nil;
  for Index := 0 to NamedNodeMap.Length - 1 do
    if (NamedNodeMap.Item(Index).BaseName = localName) and
        (NamedNodeMap.Item(Index).Namespace = namespaceURI) then
    begin
      Result := MakeNode(NamedNodeMap.Item(Index), FOwnerDocument);
      Exit;
    end;
end;

function TCSDOMNamedNodeMap.removeNamedItem(const name: DOMString): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.RemoveNamedItem(name), FOwnerDocument);
end;

function TCSDOMNamedNodeMap.removeNamedItemNS(
  const namespaceURI, localName: DOMString): IDOMNode;
var
  Index: Integer;
begin
  Result := nil;
  for Index := 0 to NamedNodeMap.Length - 1 do
    if (NamedNodeMap.Item(Index).BaseName = localName) and
        (NamedNodeMap.Item(Index).Namespace = namespaceURI) then
    begin
      Result := MakeNode(NamedNodeMap.RemoveNamedItem(
        NamedNodeMap.Item(Index).NodeName), FOwnerDocument);
      Exit;
    end;
end;

function TCSDOMNamedNodeMap.setNamedItem(const newItem: IDOMNode): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.SetNamedItem(GetNode(newItem)),
    FOwnerDocument);
end;

function TCSDOMNamedNodeMap.setNamedItemNS(const arg: IDOMNode): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.SetNamedItem(GetNode(arg)), FOwnerDocument);
end;

{ TCSDOMCharacterData -------------------------------------------------------}

function TCSDOMCharacterData.GetCharacterData: TXmlCharacterData;
begin
  Result := Node as TXmlCharacterData;
end;

procedure TCSDOMCharacterData.appendData(const data: DOMString);
begin
  CharacterData.AppendData(data);
end;

procedure TCSDOMCharacterData.deleteData(offset, count: Integer);
begin
  CharacterData.DeleteData(offset, count);
end;

function TCSDOMCharacterData.get_data: DOMString;
begin
  Result := CharacterData.Data;
end;

function TCSDOMCharacterData.get_length: Integer;
begin
  Result := CharacterData.Length;
end;

procedure TCSDOMCharacterData.insertData(offset: Integer;
  const data: DOMString);
begin
  CharacterData.InsertData(offset, data);
end;

procedure TCSDOMCharacterData.replaceData(offset, count: Integer;
  const data: DOMString);
begin
  CharacterData.ReplaceData(offset, count, data);
end;

procedure TCSDOMCharacterData.set_data(const data: DOMString);
begin
  CharacterData.Data := data;
end;

function TCSDOMCharacterData.substringData(offset, count: Integer): DOMString;
begin
  Result := CharacterData.SubStringData(offset, count);
end;

{ TCSDOMAttr ----------------------------------------------------------------}

function TCSDOMAttr.GetAttribute: TXmlAttribute;
begin
  Result := Node as TXmlAttribute;
end;

function TCSDOMAttr.get_name: DOMString;
begin
  Result := Attribute.Name;
end;

function TCSDOMAttr.get_ownerElement: IDOMElement;
begin
  Result := nil;
end;

function TCSDOMAttr.get_specified: WordBool;
begin
  Result := Attribute.Specified;
end;

function TCSDOMAttr.get_value: DOMString;
begin
  Result := Attribute.Value;
end;

procedure TCSDOMAttr.set_value(const attributeValue: DOMString);
begin
  Attribute.Value := attributeValue;
end;

{ TCSDOMElement -------------------------------------------------------------}

function TCSDOMElement.GetElement: TXmlElement;
begin
  Result := Node as TXmlElement;
end;

function TCSDOMElement.get_tagName: DOMString;
begin
  Result := Element.TagName;
end;

function TCSDOMElement.getAttribute(const name: DOMString): DOMString;
begin
  Result := Element.GetAttribute(name);
end;

function TCSDOMElement.getAttributeNS(
  const namespaceURI, localName: DOMString): DOMString;
var
  Index: Integer;
begin
  Result := '';
  for Index := 0 to Element.Attributes.Length - 1 do
    if (Element.Attributes.Item(Index).BaseName = localName) and
        (Element.Attributes.Item(Index).Namespace = namespaceURI) then
    begin
      Result := Element.GetAttribute(Element.Attributes.Item(Index).NodeName);
      Exit;
    end;
end;

function TCSDOMElement.getAttributeNode(const name: DOMString): IDOMAttr;
begin
  Result := MakeNode(Element.GetAttributeNode(name), FOwnerDocument) as IDOMAttr;
end;

function TCSDOMElement.getAttributeNodeNS(
  const namespaceURI, localName: DOMString): IDOMAttr;
var
  Index: Integer;
begin
  Result := nil;
  for Index := 0 to Element.Attributes.Length - 1 do
    if (Element.Attributes.Item(Index).BaseName = localName) and
        (Element.Attributes.Item(Index).Namespace = namespaceURI) then
    begin
      Result := MakeNode(Element.Attributes.Item(Index), FOwnerDocument)
        as IDOMAttr;
      Exit;
    end;
end;

function TCSDOMElement.getElementsByTagName(const name: DOMString):
  IDOMNodeList;
begin
  Result := MakeNodeList(Element.GetElementsByTagName(name), FOwnerDocument);
end;

function TCSDOMElement.getElementsByTagNameNS(
  const namespaceURI, localName: DOMString): IDOMNodeList;
begin
  DOMVendorNotSupported('getElementsByTagNameNS', sCUEXml); { Do not localize }
end;

function TCSDOMElement.hasAttribute(const name: DOMString): WordBool;
begin
  Result := (getAttributeNode(name) <> nil);
end;

function TCSDOMElement.hasAttributeNS(
  const namespaceURI, localName: DOMString): WordBool;
begin
  Result := (getAttributeNodeNS(namespaceURI, localName) <> nil);
end;

procedure TCSDOMElement.removeAttribute(const name: DOMString);
begin
  Element.RemoveAttribute(name);
end;

function TCSDOMElement.removeAttributeNode(const oldAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(Element.RemoveAttributeNode(
    GetNode(oldAttr) as TXmlAttribute), FOwnerDocument) as IDOMAttr;
end;

procedure TCSDOMElement.removeAttributeNS(
  const namespaceURI, localName: DOMString);
var
  Attr: IDOMAttr;
begin
  Attr := getAttributeNodeNS(localName, namespaceURI);
  if Attr <> nil then
    removeAttributeNode(Attr);
end;

procedure TCSDOMElement.setAttribute(const name, value: DOMString);
begin
  Element.SetAttribute(name, value);
end;

function TCSDOMElement.setAttributeNode(const newAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(Element.SetAttributeNode(
    GetNode(newAttr) as TXmlAttribute), FOwnerDocument) as IDOMAttr;
end;

function TCSDOMElement.setAttributeNodeNS(const newAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(Element.SetAttributeNode(
    GetNode(newAttr) as TXmlAttribute), FOwnerDocument) as IDOMAttr;
end;

procedure TCSDOMElement.setAttributeNS(
  const namespaceURI, qualifiedName, value: DOMString);
var
  Attr: IDOMAttr;
begin
  Attr := getAttributeNodeNS(namespaceURI, qualifiedName);
  if Attr <> nil then
    Attr.nodeValue := value
  else
    setAttribute(qualifiedName, value);
end;

procedure TCSDOMElement.normalize;
begin
  Element.Normalize;
end;

{ TCSDOMText ----------------------------------------------------------------}

function TCSDOMText.splitText(offset: Integer): IDOMText;
begin
  Result := MakeNode((Node as TXmlText).SplitText(offset), FOwnerDocument)
    as IDOMText;
end;

{ TCSDOMDocumentType --------------------------------------------------------}

function TCSDOMDocumentType.GetDocumentType: TXmlDocumentType;
begin
  Result := Node as TXmlDocumentType;
end;

function TCSDOMDocumentType.get_entities: IDOMNamedNodeMap;
begin
  Result := MakeNamedNodeMap(DocumentType.Entities, FOwnerDocument);
end;

function TCSDOMDocumentType.get_internalSubset: DOMString;
begin
  Result := DocumentType.XmlDocument;
end;

function TCSDOMDocumentType.get_name: DOMString;
begin
  Result := DocumentType.Name;
end;

function TCSDOMDocumentType.get_notations: IDOMNamedNodeMap;
begin
  Result := MakeNamedNodeMap(DocumentType.Notations, FOwnerDocument);
end;

function TCSDOMDocumentType.get_publicId: DOMString;
begin
  Result := '';
end;

function TCSDOMDocumentType.get_systemId: DOMString;
begin
  Result := '';
end;

{ TCSDOMNotation ------------------------------------------------------------}

function TCSDOMNotation.GetNotation: TXmlNotation;
begin
  Result := Node as TXmlNotation;
end;

function TCSDOMNotation.get_publicId: DOMString;
begin
  Result := Notation.PublicId;
end;

function TCSDOMNotation.get_systemId: DOMString;
begin
  Result := Notation.SystemId;
end;

{ TCSDOMEntity --------------------------------------------------------------}

function TCSDOMEntity.GetEntity: TXmlEntity;
begin
  Result := Node as TXmlEntity;
end;

function TCSDOMEntity.get_notationName: DOMString;
begin
  Result := Entity.NotationName;
end;

function TCSDOMEntity.get_publicId: DOMString;
begin
  Result := Entity.PublicId;
end;

function TCSDOMEntity.get_systemId: DOMString;
begin
  Result := Entity.SystemId;
end;

{ TCSDOMProcessingInstruction -----------------------------------------------}

function TCSDOMProcessingInstruction.GetProcessingInstruction:
  TXmlProcessingInstruction;
begin
  Result := Node as TXmlProcessingInstruction;
end;

function TCSDOMProcessingInstruction.get_data: DOMString;
begin
  Result := ProcessingInstruction.Data;
end;

function TCSDOMProcessingInstruction.get_target: DOMString;
begin
  Result := ProcessingInstruction.Target;
end;

procedure TCSDOMProcessingInstruction.set_data(const value: DOMString);
begin
  ProcessingInstruction.Data := value;
end;

{ TCSDOMDocument ------------------------------------------------------------}

constructor TCSDOMDocument.Create(const Node: TXmlNode;
  const Document: TCSDOMDocument);
begin
  FXMLObjModel             := TXmlObjModel.Create(nil);
  FXMLObjModel.RaiseErrors := False;
  if Assigned(Node) then
    inherited Create(Node, Self)
  else
    inherited Create(FXMLObjModel.Document, Self);
end;

destructor TCSDOMDocument.Destroy;
begin
  FXMLObjModel.Free;
  inherited Destroy;
end;

function TCSDOMDocument.GetDocument: TXmlDocument;
begin
  Result := Node as TXmlDocument;
end;

function TCSDOMDocument.createAttribute(const name: DOMString): IDOMAttr;
begin
  Result := IDOMAttr(MakeNode(Document.CreateAttribute(name), FOwnerDocument));
end;

function TCSDOMDocument.createAttributeNS(const namespaceURI,
  qualifiedName: DOMString): IDOMAttr;
begin
  Result := createAttribute(qualifiedName);
end;

function TCSDOMDocument.createCDATASection(const data: DOMString):
  IDOMCDATASection;
begin
  Result := IDOMCDATASection(MakeNode(Document.CreateCDATASection(data),
    FOwnerDocument));
end;

function TCSDOMDocument.createComment(const data: DOMString): IDOMComment;
begin
  Result := IDOMComment(MakeNode(Document.CreateComment(data), FOwnerDocument));
end;

function TCSDOMDocument.createDocumentFragment: IDOMDocumentFragment;
begin
  Result := IDOMDocumentFragment(MakeNode(Document.CreateDocumentFragment,
    FOwnerDocument));
end;

function TCSDOMDocument.createElement(const tagName: DOMString): IDOMElement;
begin
  Result := IDOMElement(MakeNode(Document.CreateElement(tagName),
    FOwnerDocument));
end;

function TCSDOMDocument.createElementNS(
  const namespaceURI, qualifiedName: DOMString): IDOMElement;
begin
  Result := createElement(qualifiedName);
end;

function TCSDOMDocument.createEntityReference(const name: DOMString):
  IDOMEntityReference;
begin
  Result := IDOMEntityReference(MakeNode(Document.CreateEntityReference(name),
    FOwnerDocument));
end;

function TCSDOMDocument.createProcessingInstruction(
  const target, data: DOMString): IDOMProcessingInstruction;
begin
  Result := IDOMProcessingInstruction(MakeNode(
    Document.CreateProcessingInstruction(target, data), FOwnerDocument));
end;

function TCSDOMDocument.createTextNode(const data: DOMString): IDOMText;
begin
  Result := IDOMText(MakeNode(Document.CreateTextNode(data), FOwnerDocument));
end;

function TCSDOMDocument.get_doctype: IDOMDocumentType;
begin
  Result := IDOMDocumentType(MakeNode(Document.DocType, FOwnerDocument));
end;

function TCSDOMDocument.get_documentElement: IDOMElement;
begin
  Result := IDOMElement(MakeNode(Document.DocumentElement, FOwnerDocument));
end;

function TCSDOMDocument.get_domImplementation: IDOMImplementation;
begin
  Result := TCSDOMImplementation.Create(Document.DomImplementation);
end;

function TCSDOMDocument.get_nodeName: DOMString;
begin
  Result := '#document';
end;

function TCSDOMDocument.getElementById(const elementId: DOMString):
  IDOMElement;
begin
  Result := MakeNode(Document.DocumentElement.
    SelectSingleNode('//*[@id="' + elementId + '"]'), FOwnerDocument)
    as IDOMElement;
end;

function TCSDOMDocument.getElementsByTagName(const tagName: DOMString):
  IDOMNodeList;
begin
  Result := MakeNodeList(Document.GetElementsByTagName(tagName), FOwnerDocument);
end;

function TCSDOMDocument.getElementsByTagNameNS(
  const namespaceURI, localName: DOMString): IDOMNodeList;
begin
  DOMVendorNotSupported('getElementsByTagNameNS', sCUEXml); { Do not localize }
end;

function TCSDOMDocument.importNode(importedNode: IDOMNode; deep: WordBool):
  IDOMNode;
var
  NewNode: TXmlNode;
begin
  NewNode := GetNode(importedNode).CloneNode(deep);
  Document.ForceOwnerDocument(NewNode);
  Result  := MakeNode(NewNode, FOwnerDocument);
end;

procedure TCSDOMDocument.set_documentElement(const IDOMElement: IDOMElement);
begin
  if Assigned(Document.DocumentElement) then
    Document.RemoveChild(Document.DocumentElement);
  Document.AppendChild(GetNode(IDOMElement));
end;

{ IDOMParseOptions Interface }

function TCSDOMDocument.get_async: Boolean;
begin
  Result := False;
end;

function TCSDOMDocument.get_preserveWhiteSpace: Boolean;
begin
  Result := not FXMLObjModel.NormalizeData;
end;

function TCSDOMDocument.get_resolveExternals: Boolean;
begin
  Result := False;
end;

function TCSDOMDocument.get_validate: Boolean;
begin
  Result := False;
end;

procedure TCSDOMDocument.set_async(Value: Boolean);
begin
  if Value <> get_async then
    DOMVendorNotSupported('set_async', sCUEXml); { Do not localize }
end;

procedure TCSDOMDocument.set_preserveWhiteSpace(Value: Boolean);
begin
  FXMLObjModel.NormalizeData := not Value;
end;

procedure TCSDOMDocument.set_resolveExternals(Value: Boolean);
begin
  if Value <> get_resolveExternals then
    DOMVendorNotSupported('set_resolveExternals', sCUEXml); { Do not localize }
end;

procedure TCSDOMDocument.set_validate(Value: Boolean);
begin
  if Value <> get_validate then
    DOMVendorNotSupported('set_validate', sCUEXml); { Do not localize }
end;

{ IDOMPersist interface }

function TCSDOMDocument.asyncLoadState: Integer;
begin
  Result := 0;
end;

function TCSDOMDocument.get_xml: DOMString;
var
  Index: Integer;
  XML: string;
begin
  Result := '';
  if FVersion <> '' then
    Result := Result + ' ' + sVersion + '="' + FVersion + '"';
  if FEncoding <> '' then
    Result := Result + ' ' + sEncoding + '="' + FEncoding + '"';
  if FStandalone <> '' then
    Result := Result + ' ' + sStandalone + '="' + FStandalone + '"';
  if Result <> '' then
    Result := '<?xml' + Result + '?>'#13;
  for Index := 0 to Document.ChildNodes.Length - 1 do
    Result := Result + FixLineBreaks(Document.ChildNodes.Item(Index).XmlDocument);
end;

procedure TCSDOMDocument.InvalidDocument(oOwner: TObject;
  wCode: Integer; oNode: TXmlNode; var bStop: Boolean);
begin
end;

function TCSDOMDocument.load(source: OleVariant): WordBool;
var
  XMLDoc: DOMString;
begin
  FGotProlog := False;
  XMLDoc     := source;
  Result     := FXMLObjModel.LoadDataSource(XMLDoc);
end;

function TCSDOMDocument.loadFromStream(const stream: TStream): WordBool;
var
  MemStream: TMemoryStream;
begin
  FGotProlog := False;
  MemStream  := TMemoryStream.Create;
  try
    MemStream.CopyFrom(stream, 0);
    Result := FXMLObjModel.LoadMemory(MemStream.Memory);
  finally
    MemStream.Free;
  end;
end;

function TCSDOMDocument.loadxml(const Value: DOMString): WordBool;
var
  XML: WideString;
begin
  FGotProlog := False;
  XML        := WideChar($FEFF) + Value;
  Result     := FXMLObjModel.LoadMemory(@XML[1]);
end;

procedure TCSDOMDocument.save(destination: OleVariant);
begin
  destination := Document.XmlDocument;
end;

procedure TCSDOMDocument.saveToStream(const stream: TStream);
var
  StrStream: TStringStream;
begin
  StrStream := TStringStream.Create(Document.XmlDocument);
  try
    Stream.CopyFrom(StrStream, 0);
  finally
    StrStream.Free;
  end;
end;

procedure TCSDOMDocument.set_OnAsyncLoad(const Sender: TObject;
  EventHandler: TAsyncEventHandler);
begin
  //
end;

{ IDOMParseError }

function TCSDOMDocument.get_errorCode: Integer;
begin
  Result := 0;
end;

function TCSDOMDocument.get_filepos: Integer;
begin
  Result := -1;
end;

function FindEmbeddedValue(const Text, Name: WideString): WideString;
var
  Index: Integer;
  Value: WideString;
begin
  Result := '';
  if Text = '' then
    Exit;
  Value := Text;
  Index := Pos(Name, Value);
  if Index > 0 then
    Delete(Value, 1, Index + Length(Name) - 1);
  Index := Pos(' ', Value);
  if Index > 0 then
    Result := Copy(Value, 1, Index - 1);
end;

function TCSDOMDocument.get_line: Integer;
begin
  Result := -1;
  try
    Result := StrToInt(FindEmbeddedValue(get_reason, 'Line: '));
  except on EConvertError do
    // Ignore
  end;
end;

function TCSDOMDocument.get_linepos: Integer;
begin
  Result := -1;
  try
    Result := StrToInt(FindEmbeddedValue(get_reason, 'Col: '));
  except on EConvertError do
    // Ignore
  end;
end;

function TCSDOMDocument.get_reason: WideString;
begin
  Result := FXMLObjModel.Errors.Text;
end;

function TCSDOMDocument.get_srcText: WideString;
begin
  Result := '';
end;

function TCSDOMDocument.get_url: WideString;
begin
  Result := '';
end;

{ IDOMXMLProlog Interface }

procedure TCSDOMDocument.GetProlog;
var
  Data: string;
  Attrs: TStringList;

  procedure ExtractAttrs(const Data: DOMString; const Attrs: TStringList);
  const
    Whitespace = [#9, #10, #13, ' '];
    NameChars  = ['A'..'Z', 'a'..'z', '0'..'9', '_'];
  var
    Index, Start, Len: Integer;
    Name: string;
    Quote: WideChar;
  begin
    Index := 1;
    Len   := Length(Data);
    repeat
      while (Index <= Len) and (Char(Data[Index]) in Whitespace) do
        Inc(Index);
      Start := Index;
      while (Index <= Len) and (Char(Data[Index]) in NameChars) do
        Inc(Index);
      Name := Copy(Data, Start, Index - Start);
      while (Index <= Len) and (Char(Data[Index]) in Whitespace) do
        Inc(Index);
      if Data[Index] <> '=' then
        raise DOMException.Create('Expected "="');
      Inc(Index);
      while (Index <= Len) and (Char(Data[Index]) in Whitespace) do
        Inc(Index);
      Quote := Data[Index];
      Inc(Index);
      Start := Index;
      while (Index <= Len) and (Data[Index] <> Quote) do
        Inc(Index);
      Attrs.Values[Name] := Copy(Data, Start, Index - Start);
      Inc(Index);
      while (Index <= Len) and (Char(Data[Index]) in Whitespace) do
        Inc(Index);
    until Index > Length(Data);
  end;

begin
  if FGotProlog then
    Exit;
  FGotProlog := True;
  if (Document.FirstChild.NodeType <> PROCESSING_INSTRUCTION_NODE) or
      (TXmlProcessingInstruction(Document.FirstChild).Target <> sXML) then
    Exit;
  Data  := TXmlProcessingInstruction(Document.FirstChild).Data;
  Attrs := TStringList.Create;
  try
    ExtractAttrs(Data, Attrs);
    FVersion    := Attrs.Values[sVersion];
    FEncoding   := Attrs.Values[sEncoding];
    FStandalone := Attrs.Values[sStandalone];
  finally
    Attrs.Free;
  end;
end;

function TCSDOMDocument.get_Encoding: DOMString;
begin
  GetProlog;
  Result := FEncoding;
end;

function TCSDOMDocument.get_Standalone: DOMString;
begin
  GetProlog;
  Result := FStandalone;
end;

function TCSDOMDocument.get_Version: DOMString;
begin
  GetProlog;
  Result := FVersion;
end;

procedure TCSDOMDocument.set_Encoding(const Value: DOMString);
begin
  FEncoding := Value;
end;

procedure TCSDOMDocument.set_Standalone(const Value: DOMString);
begin
  FStandalone := Value;
end;

procedure TCSDOMDocument.set_Version(const Value: DOMString);
begin
  FVersion := Value;
end;

{ TCSDOMImplementationFactory -------------------------------------------------}

function TCSDOMImplementationFactory.DOMImplementation: IDOMImplementation;
begin
  Result := TCSDOMImplementation.Create(TXmlDomImplementation.Create);
end;

function TCSDOMImplementationFactory.Description: String;
begin
  Result := sCUEXml;
end;

initialization
  CSXML_DOM   := TCSDOMImplementationFactory.Create;
  RegisterDOMVendor(CSXML_DOM);
finalization
  UnRegisterDOMVendor(CSXML_DOM);
  CSXML_DOM.Free;
end.
