unit OXDOMGen1;

{
  Demonstrate the generation of an XML document from a database
  using the Open XML Document Object Model (DOM).
  Requires 'movie-watcher' alias to be set up in BDE.
  Requires XDOM v2.4 package from Open XML.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written November 16, 2002.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, DBTables, StdCtrls, ExtCtrls, XDOM_2_4, CommonXML;

type
  TfrmDOMXML = class(TForm)
    memXML: TMemo;
    pnlButtons: TPanel;
      btnGenerate: TButton;
      btnSave: TButton;
    dlgSave: TSaveDialog;
    procedure btnGenerateClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
  private
  public
  end;

var
  frmDOMXML: TfrmDOMXML;

implementation

{$R *.DFM}

const
  EntityDecl   = '<!ENTITY %s SYSTEM "%s" NDATA %s>'#13;
  NotationDecl = '<!NOTATION %s %s>'#13;

{ Generate the XML document as text }
procedure TfrmDOMXML.btnGenerateClick(Sender: TObject);
var
  XMLImpl: TDomImplementation;
  XMLDoc: TdomDocument;
  XMLWriter: TDomToXmlParser;
  XMLText: string;

  { Add a simple element that only contains text }
  procedure AddSimpleElement(Parent: TdomElement; Field: TField;
    AsCDATA: Boolean = False);
  var
    Internal: TdomNode;
  begin
    Internal := Parent.appendChild(
      XMLDoc.createElement(ModifyName(Field.FieldName)));
    if AsCDATA then
      Internal.appendChild(XMLDoc.createCDATASection(Field.DisplayText))
    else
      Internal.appendChild(XMLDoc.createTextNode(Field.DisplayText));
  end;

  { Include empty field tag only if flag in DB set }
  procedure AddOptElement(Parent: TdomElement; Field: TField);
  begin
    if Field.AsBoolean then
      Parent.appendChild(
        Parent.ownerDocument.createElement(ModifyName(Field.FieldName)));
  end;

  { Generate XML prolog, style sheet reference, and main element }
  procedure GenerateHeaders;
  var
    BaseId, InternalSubset: string;

    function GenerateExternalRef(PublicId, SystemId: string): string;
    begin
      if PublicId <> '' then
        Result := 'PUBLIC "' + PublicId + '"'
      else
        Result := 'SYSTEM';
      if SystemId <> '' then
        Result := Result + ' "' + SystemId + '"';
    end;

  begin
    with XMLDoc do
    begin
      { Set prolog attributes }
      version    := '1.0';
      encoding   := 'UTF-8';
      standalone := STANDALONE_NO;
      { Compile internal subset of the DTD - notation declarations }
      InternalSubset :=
        Format(NotationDecl, [HTMLType, GenerateExternalRef(HTMLPubId, HTMLSysId)]) +
        Format(NotationDecl, [JPEGType, GenerateExternalRef(JPEGPubId, JPEGSysId)]);
      with datCommonXML.qryMovie do
      begin
        First;
        while not EOF do
        begin
          BaseId := FieldByName(MovieIdField).DisplayText;
          { Add unparsed entity declarations }
          if FieldByName(LogoURLField).AsString <> '' then
            InternalSubset := InternalSubset + Format(EntityDecl,
              [BaseId + 'Logo', FieldByName(LogoURLField).DisplayText, JPEGType]);
          if FieldByName(URLField).AsString <> '' then
            InternalSubset := InternalSubset + Format(EntityDecl,
              [BaseId + 'Url', FieldByName(URLField).DisplayText, HTMLType]);
          Next;
        end;
      end;
      { Cannot add DOCTYPE when any other content }
      removeChild(documentElement);
      { Add DOCTYPE }
      appendChild(createDocumentTypeDecl(
        MovieWatcherTag, '', XMLDTDFile, internalSubset));
      { Replace document element }
      appendChild(createElement(MovieWatcherTag));
      { Add other document-level item }
      insertBefore(createComment(XMLComment), documentElement);
      insertBefore(createProcessingInstruction(
        XMLStyleTag, XMLStyleAttrs), documentElement);
    end;
  end;

  { Compile elements for the stars of the movie }
  procedure GenerateStars(Starring: TdomElement);
  begin
    with datCommonXML.qryStars do
    begin
      First;
      while not EOF do
      begin
        AddSimpleElement(Starring, FieldByName(StarField));
        Next;
      end;
    end;
  end;

  { Generate elements for each movie }
  procedure GenerateMovies;
  var
    Movies, Movie: TdomElement;
    BaseId: string;
  begin
    Movies := TdomElement(XMLDoc.documentElement.appendChild(
      XMLDoc.createElement(MoviesTag)));
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        Movie := TdomElement(Movies.appendChild(
          XMLDoc.createElement(MovieTag)));
        BaseId := FieldByName(MovieIdField).DisplayText;
        Movie.setAttribute(Id, BaseId);
        Movie.setAttribute(Rating, FieldByName(RatingField).DisplayText);
        if FieldByName(LogoURLField).AsString <> '' then
          Movie.setAttribute(ModifyName(FieldByName(LogoURLField).FieldName),
            BaseId + 'Logo');
        if FieldByName(URLField).AsString <> '' then
          Movie.setAttribute(ModifyName(FieldByName(URLField).FieldName),
            BaseId + 'Url');
        AddSimpleElement(Movie, FieldByName(NameField));
        AddSimpleElement(Movie, FieldByName(LengthField));
        AddSimpleElement(Movie, FieldByName(DirectorField));
        GenerateStars(TdomElement(Movie.appendChild(
          XMLDoc.createElement(StarringTag))));
        AddSimpleElement(Movie, FieldByName(SynopsisField), True);
        Next;
      end;
    end;
  end;

  { Compile elements for the pricing schemes }
  procedure GeneratePricing(Pricing: TdomElement);
  var
    Price: TdomElement;
  begin
    with datCommonXML.qryPricing do
    begin
      First;
      while not EOF do
      begin
        Price := TdomElement(Pricing.appendChild(
          Pricing.ownerDocument.createElement(PriceTag)));
        Price.setAttribute(Id, FieldByName(PricingIdField).DisplayText);
        AddSimpleElement(Price, FieldByName(NameField));
        AddSimpleElement(Price, FieldByName(PeriodField));
        AddSimpleElement(Price, FieldByName(AdultField));
        AddSimpleElement(Price, FieldByName(ChildField));
        if FieldByName(DiscountField).AsFloat <> 0 then
          AddSimpleElement(Price, FieldByName(DiscountField));
        Next;
      end;
    end;
  end;

  { Generate elements for each cinema }
  procedure GenerateCinemas;
  var
    Cinemas, Cinema, Facilities: TdomElement;
  begin
    Cinemas := TdomElement(XMLDoc.documentElement.appendChild(
      XMLDoc.createElement(CinemasTag)));
    with datCommonXML.qryCinema do
    begin
      First;
      while not EOF do
      begin
        Cinema := TdomElement(Cinemas.appendChild(
          XMLDoc.createElement(CinemaTag)));
        Cinema.setAttribute(Id, FieldByName(CinemaIdField).DisplayText);
        AddSimpleElement(Cinema, FieldByName(NameField));
        AddSimpleElement(Cinema, FieldByName(PhoneField));
        AddSimpleElement(Cinema, FieldByName(AddressField));
        AddSimpleElement(Cinema, FieldByName(DirectionsField));
        Facilities := TdomElement(Cinema.appendChild(
          XMLDoc.createElement(FacilitiesTag)));
        AddOptElement(Facilities, FieldByName(CandyBarField));
        AddOptElement(Facilities, FieldByName(DisabledField));
        GeneratePricing(TdomElement(Cinema.appendChild(
          XMLDoc.createElement(PricingTag))));
        Next;
      end;
    end;
  end;

  { Compile elements for the sessions for each screening }
  procedure GenerateSessions(Sessions: TdomElement);
  var
    Session: TdomElement;
  begin
    with datCommonXML.qrySessions do
    begin
      First;
      while not EOF do
      begin
        Session := TdomElement(Sessions.appendChild(
          Sessions.ownerDocument.createElement(SessionTag)));
        Session.setAttribute(PricingId,
          FieldByName(PricingIdField).DisplayText);
        AddSimpleElement(Session, FieldByName(TimeField));
        Next;
      end;
    end;
  end;

  { Generate elements for each screening }
  procedure GenerateScreenings;
  var
    Screenings, Screening, Internal: TdomElement;
  begin
    Screenings := TdomElement(XMLDoc.documentElement.appendChild(
      XMLDoc.createElement(ScreeningsTag)));
    with datCommonXML.qryScreening do
    begin
      First;
      while not EOF do
      begin
        Screening := TdomElement(Screenings.appendChild(
          XMLDoc.createElement(ScreeningTag)));
        Screening.setAttribute(MovieId, FieldByName(MovieIdField).DisplayText);
        Screening.setAttribute(CinemaId,
          FieldByName(CinemaIdField).DisplayText);
        AddSimpleElement(Screening, FieldByName(StartDateField));
        AddSimpleElement(Screening, FieldByName(EndDateField));
        Internal := TdomElement(Screenings.appendChild(
          XMLDoc.createElement(FeaturesTag)));
        AddSimpleElement(Internal, FieldByName(DigSoundField));
        Internal := TdomElement(Screenings.appendChild(
          XMLDoc.createElement(RestrictionsTag)));
        AddOptElement(Internal, FieldByName(NoPassesField));
        GenerateSessions(TdomElement(Screenings.appendChild(
          XMLDoc.createElement(SessionsTag))));
        Next;
      end;
    end;
  end;

begin
  Screen.Cursor       := crHourglass;
  btnGenerate.Enabled := False;
  try
    XMLImpl := TDomImplementation.create(nil);
    try
      { Instantiate the DOM }
      XMLDoc := XMLImpl.createDocument(MovieWatcherTag, nil);
      { Generate the structure }
      GenerateHeaders;
      GenerateMovies;
      GenerateCinemas;
      GenerateScreenings;
      { And convert to XML }
      XMLWriter := TDomToXmlParser.create(nil);
      try
        XMLWriter.DOMImpl := XMLImpl;
        XMLWriter.writeToString(XMLDoc, 'UTF-8', XMLText);
        memXML.Lines.Text := XMLText;
      finally
        XMLWriter.Free;
      end;
    finally
      { Release the DOM }
      XMLImpl.FreeDocument(XMLDoc);
      XMLImpl.Free;
    end;
  finally
    btnGenerate.Enabled := True;
    Screen.Cursor       := crDefault;
  end;
end;

{ Save the generated XML }
procedure TfrmDOMXML.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
    if Execute then
      memXML.Lines.SaveToFile(Filename);
end;

end.
