unit FindMoviesIntf;

{
  Invokable interface IFindMovies for SOAP Web service.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 17 September, 2002.
}

interface

uses
  InvokeRegistry, Types, XSBuiltIns;

type
  { Enumerated type for ratings }
  TMovieRating = (mrUnknown, mrNR, mrG, mrPG, mrPG13, mrR);

  { Class to hold details about a movie - must derive from TRemotable }
  TMovieDetails = class(TRemotable)
  private
    FDirector: string;
    FLength: Integer;
    FName: string;
    FRating: TMovieRating;
    FStars: TStringDynArray;
    FSynopsis: string;
    FURL: string;
  published
    property Name: string read FName write FName;
    property Length: Integer read FLength write FLength;
    property Rating: TMovieRating read FRating write FRating;
    property Director: string read FDirector write FDirector;
    property Stars: TStringDynArray read FStars write FStars;
    property Synopsis: string read FSynopsis write FSynopsis;
    property URL: string read FURL write FURL;
  end;

  { Custom exception }
  EFindMoviesException = class(ERemotableException)
  private
    FExtraMessage: string;
  public
    constructor Create(const ExtraMessage: string);
  published
    property ExtraMessage: string read FExtraMessage write FExtraMessage;
  end;

  { Invokable interfaces must derive from IInvokable }
  IFindMovies = interface(IInvokable)
  ['{FFD442FB-0E60-4631-9A2F-75AD8DF9B982}']
    function FindMovies(const Rating: TMovieRating): TStringDynArray; stdcall;
    function GetMovieDetails(const MovieName: string): TMovieDetails; stdcall;
  end;

const
  NamespaceURI = 'http://www.the_movies.com/FindMovies';
  Ratings: array [TMovieRating] of string = ('', 'NR', 'G', 'PG', 'PG-13', 'R');

implementation

resourcestring
  NoMoviesFound = 'No movies found';
  
{ EFindMoviesException --------------------------------------------------------}

constructor EFindMoviesException.Create(const ExtraMessage: string);
begin
  inherited Create(NoMoviesFound);
  FExtraMessage := ExtraMessage;
end;

initialization
  { Invokable interfaces must be registered }
  InvRegistry.RegisterInterface(TypeInfo(IFindMovies), NamespaceURI, '',
    'Find a list of movies with a given rating.'#13 +
    'If the rating is blank, all movies are returned.'#13 +
    'Find details about an individual movie, given its name.');

  { New data types must also be registered }
  RemClassRegistry.RegisterXSInfo(TypeInfo(TMovieRating), NamespaceURI);
  RemClassRegistry.RegisterXSClass(TMovieDetails, NamespaceURI);

  { Register the custom exception type }
  RemTypeRegistry.RegisterXSClass(EFindMoviesException, NamespaceURI);
end.
