unit Exams;

{
  Examination objects.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 22 October, 1999.
}

interface

uses
  Classes, SysUtils, Windows, ActiveX, MSXML2_TLB;

type
  { Possible answer presentation types }
  TAnswerType = (atCheckbox, atRadio, atText);

  { A possible answer }
  TAnswer = class(TObject)
  private
    FCorrect: Boolean;
    FValue: string;
  public
    constructor Create(Value: string; Correct: Boolean);
    property Correct: Boolean read FCorrect write FCorrect;
    property Value: string read FValue write FValue;
  end;

  { A single question }
  TQuestion = class(TObject)
  private
    FAnswers: TList;
    FAnswerType: TAnswerType;
    FExplanation: string;
    FId: string;
    FQuery: string;
    FValidAnswers: TStringList;
    procedure Add(Answer: TAnswer);
    function GetAnswer(Index: Integer): TAnswer;
    function GetAnswerCount: Byte;
    function GetValidAnswers: TStringList;
  public
    constructor Create(Id: string);
    destructor Destroy; override;
    property AnswerCount: Byte read GetAnswerCount;
    property Answers[Index: Integer]: TAnswer read GetAnswer;
    property AnswerType: TAnswerType read FAnswerType write FAnswerType;
    property Explanation: string read FExplanation write FExplanation;
    property Id: string read FId write FId;
    property Query: string read FQuery write FQuery;
    property ValidAnswers: TStringList read GetValidAnswers;
    function IsValid(Response: string): Boolean;
  end;

  { The entire exam }
  TExam = class(TObject)
  private
    FDescription: string;
    FInstructions: string;
    FPassMark: Byte;
    FQuestions: TList;
    FStrictOrder: Boolean;
    FTitle: string;
    procedure Add(Question: TQuestion);
    function GetQuestion(Index: Integer): TQuestion;
    function GetQuestionById(Id: string): TQuestion;
    function GetQuestionCount: Byte;
  public
    constructor Create;
    destructor Destroy; override;
    property Description: string read FDescription write FDescription;
    property Instructions: string read FInstructions write FInstructions;
    property PassMark: Byte read FPassMark write FPassMark;
    property QuestionCount: Byte read GetQuestionCount;
    property QuestionById[Id: string]: TQuestion read GetQuestionById;
    property Questions[Index: Integer]: TQuestion read GetQuestion;
    property StrictOrder: Boolean read FStrictOrder write FStrictOrder;
    property Title: string read FTitle write FTitle;
  end;

  { The user's answers to the questions }
  TUserSession = class(TPersistent)
  private
    FAnswers: TStringList;
    FExam: TExam;
    function GetAnswer(QuestionId: string): string;
    function GetAnswered: Integer;
    function GetCorrect: Integer;
    function GetIsAnswered(QuestionId: string): Boolean;
    function GetQuestionId(Index: Integer): string;
    function GetQuestionCount: Integer;
    function GetScore: Integer;
    procedure SetAnswer(QuestionId, UserAnswer: string);
  public
    constructor Create(Exam: TExam);
    destructor Destroy; override;
    property Answer[QuestionId: string]: string read GetAnswer write SetAnswer;
    property Answered: Integer read GetAnswered;
    property Correct: Integer read GetCorrect;
    property Exam: TExam read FExam;
    property IsAnswered[QuestionId: string]: Boolean read GetIsAnswered;
    property QuestionId[Index: Integer]: string read GetQuestionId;
    property QuestionCount: Integer read GetQuestionCount;
    property Score: Integer read GetScore;
    procedure InitialiseQuestions;
  end;

  { Unit-specific exceptions }
  EExamException = class(Exception);

{ Read an XML file containing the exam specification
  and convert it into the above objects }
function LoadExam(FileName: string): TExam;

implementation

resourcestring
  CannotParse      = 'Cannot parse XML document %s'#13#10'%s';
  MissingAnswers   = 'Missing answers for question %s';
  MissingQuery     = 'Missing query for question %s';
  MissingQuestions = 'Missing questions';
  NotAnExam        = 'Document %s doesn''t contain an examination';

const
  { XML elements/attributes }
  AnswerTag       = 'answer';
  AnswersTag      = 'answers';
  CorrectAttr     = 'correct';
  DescriptionTag  = 'description';
  ExamTag         = 'exam';
  ExplanationTag  = 'explanation';
  IdAttr          = 'id';
  InstructionsTag = 'instructions';
  PassMarkAttr    = 'pass_mark';
  QueryTag        = 'query';
  QuestionTag     = 'question';
  StrictOrderAttr = 'strict_order';
  TitleTag        = 'title';
  TypeAttr        = 'type';
  { Attribute values }
  AnswerTypes: array [TAnswerType] of string = ('checkbox', 'radio', 'text');
  TrueValue       = 'true';
  { Flagging text for unanswered questions }
  Unanswered      = '<*Unanswered*>';

{ TAnswer ---------------------------------------------------------------------}

{ Initialisation }
constructor TAnswer.Create(Value: string; Correct: Boolean);
begin
  inherited Create;
  FValue   := Value;
  FCorrect := Correct;
end;

{ TQuestion -------------------------------------------------------------------}

{ Initialisation }
constructor TQuestion.Create(Id: string);
begin
  inherited Create;
  FAnswers      := TList.Create;
  FId           := Id;
  FValidAnswers := TStringList.Create;
end;

{ Release resources }
destructor TQuestion.Destroy;
var
  Index: Integer;
begin
  for Index := 0 to FAnswers.Count - 1 do
    TAnswer(FAnswers[Index]).Free;
  FAnswers.Free;
  FValidAnswers.Free;
  inherited Destroy;
end;

{ Add a new answer }
procedure TQuestion.Add(Answer: TAnswer);
begin
  FAnswers.Add(Answer);
  FValidAnswers.Clear;
end;

{ Return the specified answer }
function TQuestion.GetAnswer(Index: Integer): TAnswer;
begin
  Result := TAnswer(FAnswers[Index]);
end;

{ Return the number of answers }
function TQuestion.GetAnswerCount: Byte;
begin
  Result := FAnswers.Count;
end;

{ Return the valid answer(s) in string format }
function TQuestion.GetValidAnswers: TStringList;
var
  Index: Integer;
  ValidAnswers: string;
begin
  if FValidAnswers.Count = 0 then
  begin
    ValidAnswers := '';
    for Index := 0 to AnswerCount - 1 do
      { For each correct answer}
      if Answers[Index].Correct then
        { Add complete answer if text type }
        if AnswerType = atText then
          FValidAnswers.Add(Answers[Index].Value)
        { Otherwise string together numeric positions }
        else
          ValidAnswers := ValidAnswers + ',' + IntToStr(Index);
    if ValidAnswers <> '' then
      FValidAnswers.Add(Copy(ValidAnswers, 2, Length(ValidAnswers)));
  end;
  Result := FValidAnswers;
end;

{ Is this response a valid answer? }
function TQuestion.IsValid(Response: string): Boolean;
begin
  Result := (ValidAnswers.IndexOf(Response) > -1);
end;

{ TExam -----------------------------------------------------------------------}

{ Initialisation }
constructor TExam.Create;
begin
  inherited Create;
  FQuestions := TList.Create;
end;

{ Release resources }
destructor TExam.Destroy;
var
  Index: Integer;
begin
  for Index := 0 to FQuestions.Count - 1 do
    TQuestion(FQuestions[Index]).Free;
  FQuestions.Free;
  inherited Destroy;
end;

{ Add a new question }
procedure TExam.Add(Question: TQuestion);
begin
  FQuestions.Add(Question);
end;

{ Return the specified question based on its position }
function TExam.GetQuestion(Index: Integer): TQuestion;
begin
  Result := TQuestion(FQuestions[Index]);
end;

{ Return the specified question based on its id }
function TExam.GetQuestionById(Id: string): TQuestion;
var
  Index: Integer;
begin
  for Index := 0 to FQuestions.Count - 1 do
  begin
    Result := TQuestion(FQuestions[Index]);
    if Result.Id = Id then
      Exit;
  end;
  Result := nil;
end;

{ Return the number of questions }
function TExam.GetQuestionCount: Byte;
begin
  Result := FQuestions.Count;
end;

{ TUserSession ----------------------------------------------------------------}

{ Initialisation }
constructor TUserSession.Create(Exam: TExam);
begin
  inherited Create;
  FAnswers := TStringList.Create;
  FExam    := Exam;
  InitialiseQuestions;
end;

{ Release resources }
destructor TUserSession.Destroy;
begin
//  FAnswers.SaveToFile('answers.txt');
  FAnswers.Free;
  inherited Destroy;
end;

{ Return the answer for the specified question }
function TUserSession.GetAnswer(QuestionId: string): string;
begin
  Result := FAnswers.Values[QuestionId];
  if Result = Unanswered then
    Result := '';
end;

{ Return the number of answered questions }
function TUserSession.GetAnswered: Integer;
var
  Index: Integer;
begin
  Result := 0;
  for Index := 0 to QuestionCount - 1 do
    if Answer[QuestionId[Index]] <> '' then
      Inc(Result);
end;

{ Return the number of correct answers }
function TUserSession.GetCorrect: Integer;
var
  Index: Integer;
begin
  Result := 0;
  for Index := 0 to QuestionCount - 1 do
    if Exam.QuestionById[QuestionId[Index]].
        IsValid(Answer[QuestionId[Index]]) then
      Inc(Result);
end;

{ Has this question been answered? }
function TUserSession.GetIsAnswered(QuestionId: string): Boolean;
begin
  Result := (Answer[QuestionId] <> '');
end;

{ Return the id for the specified question }
function TUserSession.GetQuestionId(Index: Integer): string;
begin
  Result := FAnswers.Names[Index];
end;

{ Return the number of questions }
function TUserSession.GetQuestionCount: Integer;
begin
  Result := FAnswers.Count;
end;

{ Return the current score }
function TUserSession.GetScore: Integer;
begin
  Result := Round(Correct / QuestionCount * 100);
end;

{ Randomise the order of the questions (if appropriate) }
procedure TUserSession.InitialiseQuestions;
var
  Index: Integer;
begin
  { Load the question ids as unanswered }
  FAnswers.Clear;
  for Index := 0 to Exam.QuestionCount - 1 do
    FAnswers.Values[Exam.Questions[Index].Id] := Unanswered;

  { If cannot reorder or nothing to reorder then finished }
  if Exam.StrictOrder or (Exam.QuestionCount < 2) then
    Exit;

  { Shuffle the questions }
  Randomize;
  for Index := FAnswers.Count - 1 downto 1 do
    FAnswers.Exchange(Random(Index + 1), Index);
end;

{ Add a user's answer }
procedure TUserSession.SetAnswer(QuestionId, UserAnswer: string);
begin
  { Only save the answer if not already there }
  if (Answer[QuestionId] = '') and (UserAnswer <> '') then
    FAnswers.Values[QuestionId] := UserAnswer;
end;

{ LoadExam --------------------------------------------------------------------}

{ Read an XML file containing the exam specification
  and convert it into the above objects }
function LoadExam(FileName: string): TExam;
var
  XMLDoc: IXMLDOMDocument;
  Exam: TExam;
  ExamNode: Integer;

  { Retrieve the attribute value from the element }
  function Attribute(Node: IXMLDOMNode; Attribute: string): string;
  var
    AttrNode: IXMLDOMNode;
  begin
    AttrNode := Node.Attributes.GetNamedItem(Attribute);
    if Assigned(AttrNode) then
      Result := AttrNode.Text
    else
      Result := '';
  end;

  { Extract the details for a question }
  procedure GetQuestion(QuestionNode: IXMLDOMElement);
  var
    Question: TQuestion;
    QstNode: Integer;
    AnsType: string;
    AnswerType: TAnswerType;

    { Extract the details for a series of answers }
    procedure GetAnswers(Answers: IXMLDOMElement);
    var
      AnsNode: Integer;
      Answer: TAnswer;
    begin
      if Answers.HasChildNodes then
        with Answers.ChildNodes do
          { Get each possible answer }
          for AnsNode := 0 to Length - 1 do
          begin
            if Item[AnsNode].NodeName = AnswerTag then
            begin
              Answer := TAnswer.Create(Item[AnsNode].FirstChild.NodeValue,
                (Attribute(Item[AnsNode], CorrectAttr) = TrueValue));
              try
                { And add it to the question }
                Question.Add(Answer);
              except
                Answer.Free;
                raise;
              end;
            end;
          end;
    end;

  begin
    { Create the question }
    Question := TQuestion.Create(Attribute(QuestionNode, IdAttr));
    try
      if QuestionNode.HasChildNodes then
        with QuestionNode.ChildNodes do
          for QstNode := 0 to Length - 1 do
            { Get question values from child nodes }
            if Item[QstNode].NodeName = QueryTag then
              Question.Query := Item[QstNode].FirstChild.NodeValue
            else if Item[QstNode].NodeName = ExplanationTag then
              Question.Explanation := Item[QstNode].FirstChild.NodeValue
            else if Item[QstNode].NodeName = AnswersTag then
            begin
              { Get type of answers }
              Question.AnswerType := atText;
              AnsType := Attribute(Item[QstNode], TypeAttr);
              for AnswerType := Low(TAnswerType) to High(TAnswerType) do
                if AnswerTypes[AnswerType] = AnsType then
                begin
                  Question.AnswerType := AnswerType;
                  Break;
                end;
              { Then load the actual values }
              GetAnswers(IXMLDOMElement(Item[QstNode]));
            end;

      { Validations }
      if Question.Query = '' then
        raise EExamException.Create(Format(MissingQuery, [Question.Id]));
      if Question.AnswerCount = 0 then
        raise EExamException.Create(Format(MissingAnswers, [Question.Id]));

      { And add the question to the exam }
      Exam.Add(Question);
    except
      Question.Free;
      raise;
    end;
  end;

begin
  { Create the exam }
  Exam := TExam.Create;
  try
    { Create the XML parser }
    XMLDoc := CoDOMDocument.Create;

    try
      try
        { And parse the XML document }
        if not XMLDoc.Load(FileName) then
          Abort;
      except
        raise EExamException.Create(
          Format(CannotParse, [Filename, XMLDoc.ParseError.Reason]));
      end;
      if XMLDoc.DocumentElement.NodeName <> ExamTag then
        raise EExamException.Create(Format(NotAnExam, [Filename]));

      { Get the exam attributes }
      try
        Exam.PassMark :=
          StrToInt(Attribute(XMLDoc.DocumentElement, PassMarkAttr));
      except
        Exam.PassMark := 100;  { % }
      end;
      Exam.StrictOrder :=
        (Attribute(XMLDoc.DocumentElement, StrictOrderAttr) = TrueValue);

      { Load child elements }
      if XMLDoc.DocumentElement.HasChildNodes then
        with XMLDoc.DocumentElement.ChildNodes do
          for ExamNode := 0 to Length - 1 do
            { Get exam values from child nodes }
            if Item[ExamNode].NodeName = TitleTag then
              Exam.Title := Item[ExamNode].FirstChild.NodeValue
            else if Item[ExamNode].NodeName = DescriptionTag then
              Exam.Description := Item[ExamNode].FirstChild.NodeValue
            else if Item[ExamNode].NodeName = InstructionsTag then
              Exam.Instructions := Item[ExamNode].FirstChild.NodeValue
            { Process each question }
            else if Item[ExamNode].NodeName = QuestionTag then
              GetQuestion(IXMLDOMElement(Item[ExamNode]));

      { Validations }
      if Exam.QuestionCount = 0 then
        raise EExamException.Create(MissingQuestions);
    finally
      XMLDoc := nil;
    end;
  except
    Exam.Free;
    raise;
  end;

  Result := Exam;
end;

initialization
  { Initialise COM }
  CoInitialize(nil);
finalization
  { Tidy up }
  CoUninitialize();
end.
