package wood.keith.opentools.browser;

import java.io.PrintStream;
import javax.swing.JOptionPane;

import com.borland.primetime.Command;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.BrowserAdapter;
import com.borland.primetime.ide.NodeViewer;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.Project;
import com.borland.primetime.util.VetoException;

/**
 * Demonstrate the BrowserListener interface.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  26 March 2002
 */
public class BrowserTrace extends BrowserAdapter implements Command {

  private static final String VERSION = "1.0";
  private static final String LEADER  = "BrowserTrace: ";
  private static final String DESCR   =
    "Display browser events to standard output";
  private static final String USAGE   =
    "  Registers a global BrowserListener and writes all browser events " +
    "to standard output.";
  private static final BrowserTrace TRACER = new BrowserTrace();

  /**
   * Register the new command with JBuilder.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    PrimeTime.registerCommand("browsertrace", TRACER);
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded browsertrace command v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  /**
   * Print out more detailed help on this command.
   *
   * @param  out  the stream to write the help to
   */
  public void printCommandHelp(PrintStream out) { out.println(USAGE); }

  /**
   * Return the text for the -help command.
   *
   * @return  a basic description of the command
   */
  public String getCommandDescription() { return DESCR; }

  /**
   * Register the browser listener.
   *
   * @param  params  an array of parameter values for this command
   */
  public void invokeCommand(String[] args) {
    Browser.addStaticBrowserListener(TRACER);
  }

  /**
   * Does this command accept any arguments
   *
   * @return  false, since it does not accept parameters.
   */
  public boolean takesCommandArguments() {
    return false;
  }

  public void browserViewerDeactivating(Browser browser, Node node,
      NodeViewer viewer) throws VetoException {
    System.out.println(LEADER + "viewerDeactivating " +
      browser.getBrowserTitle() + " " + node.getDisplayName() + " " +
      (viewer == null ? "null" : viewer.getViewerTitle()));
    check(browser, "Allow browserViewerDeactivating");
  }

  public void browserViewerActivated(Browser browser, Node node,
      NodeViewer viewer) {
    System.out.println(LEADER + "viewerActivated " +
      browser.getBrowserTitle() + " " + node.getDisplayName() + " " +
      (viewer == null ? "null" : viewer.getViewerTitle()));
  }

  public void browserNodeClosed(Browser browser, Node node) {
    System.out.println(LEADER + "nodeClosed " + browser.getBrowserTitle() +
      " " + (node == null ? "null" : node.getDisplayName()));
  }

  public void browserNodeActivated(Browser browser, Node node) {
    System.out.println(LEADER + "nodeActivated " + browser.getBrowserTitle() +
      " " + (node == null ? "null" : node.getDisplayName()));
  }

  public void browserProjectClosed(Browser browser, Project project) {
    System.out.println(LEADER + "projectClosed " + browser.getBrowserTitle() +
      " " + (project == null ? "null" : project.getDisplayName()));
  }

  public void browserProjectActivated(Browser browser, Project project) {
    System.out.println(LEADER + "projectActivated " + browser.getBrowserTitle() +
      " " + (project == null ? "null" : project.getDisplayName()));
  }

  public void browserClosed(Browser browser) {
    System.out.println(LEADER + "closed " + browser.getBrowserTitle());
  }

  public void browserClosing(Browser browser) throws VetoException {
    System.out.println(LEADER + "closing " + browser.getBrowserTitle());
    check(browser, "Allow browserClosing");
  }

  public void browserDeactivated(Browser browser) {
    System.out.println(LEADER + "deactivated " + browser.getBrowserTitle());
  }

  public void browserActivated(Browser browser) {
    System.out.println(LEADER + "activated " + browser.getBrowserTitle());
  }

  public void browserOpened(Browser browser) {
    System.out.println(LEADER + "opened " + browser.getBrowserTitle());
  }

  private void check(Browser browser, String message) throws VetoException {
    if (JOptionPane.showConfirmDialog(browser, message, LEADER,
        JOptionPane.YES_NO_OPTION) == JOptionPane.NO_OPTION) {
      throw new VetoException(message);
    }
  }
}