package wood.keith.opentools.buildnumber;

import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;

import com.borland.jbuilder.build.DependencyBuilder;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.build.Builder;
import com.borland.primetime.build.BuildListener;
import com.borland.primetime.build.BuilderManager;
import com.borland.primetime.build.BuildProcess;
import com.borland.primetime.build.BuildTask;
import com.borland.primetime.build.Phase;
import com.borland.primetime.node.Node;

/**
 *  Title:       Build Number OpenTool
 *  Description: This simple opentool provides a build number incremented
 *               automatically with each build/make.
 *               New version for JBuilder 7 only.
 *  Copyright:   Copyright (c) 2001
 *
 * @author  Joi Ellis
 * @created July 11, 2001
 * @version 1.4
 * @author  Keith Wood (kbwood@iprimus.com.au)
 * @version 1.4A
 */
public class BuildNumberBuilder extends Builder {

  /** The name of this build step. */
  private static final String TARGET = "buildnumber";

  /**
   * Registers this class as a new type of Builder.
   *
   * @param  major  OTAPI Major version
   * @param  minor  OTAPI Minor version
   */
  public static void initOpenTool( byte major, byte minor ) {
    if ( major == 4 && minor >= 4 ) {  // JBuilder 7 +
      if ( PrimeTime.isVerbose() ) {
        BuildNumberNode.printAbout( major, minor );
      }
      BuilderManager.registerBuilder(new BuildNumberBuilder());
      DependencyBuilder.registerTarget(TARGET, Phase.COMPILE_PHASE);
    } else {
      System.out.println( "BuildNumberOT not loaded: unexpected OTAPI version." );
    }
  }

  /**
   * Decide whether this node is handled by this builder.
   *
   * @param  node  the node being built
   * @return  true if this node is built here, false otherwise
   */
  public boolean isMakeable(Node node) {
    return (node instanceof BuildNumberNode);
  }

  /**
   * Called once for every node in the project.
   *
   * Nodes we recognize are stuffed into a BuildNumberBuildTask and added to
   * the pile of tasks to complete for this build.
   *
   * @param  process  The current BuildProcess
   * @param  node     The node being offered to us for building
   */
  public void updateBuildProcess( BuildProcess process, Node node ) {
    if (!isMakeable(node)) {
      return;
    }

    BuildNumberNode.blat("Processing " + node.getLongDisplayName());
    BuildNumberBuildTask task = (BuildNumberBuildTask)
      process.getFirstBuildTask(TARGET, BuildNumberBuildTask.class);
    if (task == null) {
      // Create the task if not already there
      task = (BuildNumberBuildTask)
        process.createTask(BuildNumberBuildTask.class, TARGET);
    }
    // Add this node to it
    task.addBuildSource((BuildNumberNode)node);
    BuildNumberNode.blat("Task added for " + node.getLongDisplayName());

    // Look for pre- and post-build commands
    BuildNumberBuildListener listener =
      (BuildNumberBuildListener)process.getProperty(this);
    if (listener == null) {
      // Create a new listener if not there
      listener = new BuildNumberBuildListener();
      process.addBuildListener(listener);
      process.putProperty(this, listener);
    }
    try {
      Properties props = ((BuildNumberNode)node).getProps();
      // Note pre- and post-build commands
      String prop = props.getProperty("prebuild");
      if (prop != null && prop.trim().length() > 0) {
        listener.addPreBuildAction(prop);
      }
      prop = props.getProperty("postbuild");
      if (prop != null && prop.trim().length() > 0) {
        listener.addPostBuildAction(prop);
      }
    }
    catch (Exception ex) {
      if (PrimeTime.isVerbose()) {
        ex.printStackTrace();
      }
    }
  }
}

/**
 * A listener for build events to run commands before or after a build.
 */
class BuildNumberBuildListener extends BuildListener {

  private Vector preBuildActions = new Vector();
  private Vector postBuildActions = new Vector();

  /**
   * Add a new command to run before the build.
   *
   * @param  action  the command
   */
  public void addPreBuildAction(String action) {
    preBuildActions.addElement(action);
  }

  /**
   * Add a new command to run after the build.
   *
   * @param  action  the command
   */
  public void addPostBuildAction(String action) {
    postBuildActions.addElement(action);
  }

  /**
   * Called when the build begins.
   *
   * @param  process  the process being built
   */
  public void buildStart(BuildProcess process) {
    BuildNumberNode.blat("Running preBuild actions");
    runActions(preBuildActions);
  }

  /**
   * Called when the build completes.
   *
   * @param  process  the process being built
   */
  public void buildFinish(BuildProcess process) {
    BuildNumberNode.blat("Running postBuild actions");
    runActions(postBuildActions);
  }

  /**
   * Run a list of commands before or after a build.
   *
   * @param  actions  the list of commands
   */
  private void runActions(Vector actions) {
    Iterator iter = actions.iterator();
    int rc = 0;
    while (iter.hasNext()) {
      String item = (String)iter.next();
      BuildNumberNode.blat("Running " + item);
      try {
        Process proc = Runtime.getRuntime().exec(item);
        rc = proc.waitFor();
        BuildNumberNode.blat(item + " return code: " + rc);
      }
      catch (Exception ex) {
        System.out.println("BuildNumberOT: Error running: " + item);
        ex.printStackTrace();
      }
    }
    actions.removeAllElements();
  }
}
