package wood.keith.opentools.buildnumber;

import com.borland.jbuilder.build.CopiedResource;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.ide.BrowserIcons;
import com.borland.primetime.node.DuplicateNodeException;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.Project;
import com.borland.primetime.node.TextFileNode;
import com.borland.primetime.vfs.Url;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import javax.swing.Icon;
import java.util.Properties;
import java.io.ByteArrayOutputStream;
import com.borland.primetime.vfs.ReadOnlyException;
import java.io.IOException;
import java.util.Date;

/**
 *  Title:       Build Number OpenTool
 *  Description: This simple opentool provides a build number incremented
 *               automatically with each build/make.
 *  Copyright:   Copyright (c) 2001
 *
 * This class defines the new node type.  It also provides some static package
 * methods for printing the copyright and OTAPI version.
 *
 * @author     Joi Ellis
 * @created    July 11, 2001
 * @version    1.4A
 */

public class BuildNumberNode extends TextFileNode implements CopiedResource {

  public static String version = "1.4A";
  public static String buildNo = "0";
  private static URL buildNoURL;
  private static Properties buildProps = new Properties();
  protected static boolean DEBUG;


  /**
   *  Registers this class as a new type of Node.
   *
   * @param  major  OTAPI Major Version
   * @param  minor  OTAPI Minor Version
   */
  public static void initOpenTool( byte major, byte minor ) {
    BuildNumberNode.DEBUG=Boolean.getBoolean("BuildNumberOT.debug");
    if ( major == 4 ) {
      if ( PrimeTime.isVerbose() ) {
        loadBuildNo();
        printAbout( major, minor );
      }
      registerFileNodeClass( "num", "Build Number", BuildNumberNode.class,
        BrowserIcons.ICON_REBUILD );
    } else {
        System.out.println( "BuildNumberOT not loaded: unexpected OTAPI version." );
    }
  }


  /**
   *  Reads the build.num resource file.
   *
   * @return    String containing the package build number.
   */
  private static void loadBuildNo() {
    InputStream in = null;
    try {
      in = BuildNumberNode.class.getResourceAsStream("build.num");
      buildProps.load( in );
      in.close();
    } catch ( Exception ex ) {
    }
    version = buildProps.getProperty("version");
    buildNo = buildProps.getProperty("build","0");
  }


  /**
   *  Constructor for the BuildNumberNode object
   *
   * @param  project                     Project being built
   * @param  node                        Node (file) being built
   * @param  url                         Url of the file being built
   * @exception  DuplicateNodeException  Description of Exception
   */
  public BuildNumberNode( Project project, Node node, Url url ) throws DuplicateNodeException {
    super( project, node, url );
  }


  /**
   *  Gets the displayIcon attribute of the BuildNumberNode object
   *
   * @return    The displayIcon to show in the project pane
   */
  public Icon getDisplayIcon() {
    return BrowserIcons.ICON_REBUILD;
  }


  /**
   *  Prints a simple copyright banner to System.out.
   *
   * @param  major  Description of Parameter
   * @param  minor  Description of Parameter
   */
  static void printAbout( byte major, byte minor ) {
    System.out.println( "BuildNumberOT Version " + version + " (Build " + buildNo + ")" );
    System.out.println( "  Copyright (c) 2001,2002, Joi Ellis.  All rights reserved." );
    System.out.println( "  JBuilder OTAPI Major.Minor: " + major + "." + minor );
    if ( BuildNumberNode.DEBUG ) {
      System.out.println("  DEBUGGING OUTPUT ENABLED.");
    }

  }
  private static String test() {
    Properties props = new Properties();
    InputStream in = null;
    try {
      in = BuildNumberNode.class.getResourceAsStream("build.num");
      props.load( in );
      in.close();
    } catch ( Exception ex ) {
    }
    return props.getProperty("build","0");
  }

  protected Properties getProps(  ) throws Exception {
    Properties props = new Properties();
    InputStream in = getBuffer().getInputStream();
    props.load( in );
    in.close();
    return props;
  }
  protected void putProps( Properties buildNodeProps ) throws ReadOnlyException,
  IOException {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      buildNodeProps.store(baos,"Contents updated by BuildNumberOT");
      getBuffer().setContent( baos.toByteArray() );
      getBuffer().save();
      baos.close();
  }

  protected void updateProps() throws IOException, ReadOnlyException, Exception {
    Properties props = getProps();
    BuildNumberNode.incrementProps( props );
    putProps( props );
  }

  protected static void setDefaults( Properties buildNodeProps ) {
    buildNodeProps.setProperty("build","0");
    buildNodeProps.setProperty("version","0.0.0");
    buildNodeProps.setProperty("date","");
    buildNodeProps.setProperty("epoch","0");
    buildNodeProps.setProperty("prebuild","");
    buildNodeProps.setProperty("postbuild","");
    buildNodeProps.setProperty("comment","");
    buildNodeProps.setProperty("disabled","false");
  }

  protected static void incrementProps( Properties buildNodeProps ) {
      int i = 0;
      if ( buildNodeProps.isEmpty() ) {
        setDefaults( buildNodeProps );
      }

      try {
        i = Integer.parseInt(buildNodeProps.getProperty("build","0"));
        i++;
      } catch ( NumberFormatException nfe ) {
        i = 0;
      }
      buildNodeProps.setProperty("build", Integer.toString(i));

      if ( buildNodeProps.containsKey("version") ) {
        int buildCount = 0;
        String data = buildNodeProps.getProperty("version");
        i = data.lastIndexOf(".");
        String prefix = data.substring(0,i+1);
        try {
          buildCount = Integer.parseInt( data.substring(i+1) );
        } catch ( NumberFormatException nfe ) {
          buildCount = 0;
        }
        buildCount++;
        buildNodeProps.setProperty("version", prefix + buildCount );
      }

      Date now = new Date();
      if ( buildNodeProps.containsKey("date") ) {
        buildNodeProps.setProperty("date", now.toString() );
      }

      if ( buildNodeProps.containsKey("epoch") ) {
        buildNodeProps.setProperty("epoch", Long.toString( now.getTime() ));
      }
  }
  protected static void blat( String text ) {
    if ( PrimeTime.isVerbose() && BuildNumberNode.DEBUG ) {
      System.out.println( text );
    }
  }
}
