package wood.keith.opentools.consoledisplay;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.StringTokenizer;

import com.borland.jbuilder.node.JBProject;
import com.borland.primetime.Command;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.BrowserAdapter;
import com.borland.primetime.ide.Message;
import com.borland.primetime.ide.MessageCategory;
import com.borland.primetime.ide.MessageView;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.TextFileNode;
import com.borland.primetime.vfs.Url;

/**
 * Example of a command line processor OpenTool.
 * It redirects console output into a message view.
 * Double-click on stack trace messages to open source.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  13 September 2001
 * @version  2.0  25 February 2002
 */
public class ConsoleDisplay implements Command {

  private static final String VERSION = "2.0";
  private static final String LEADER  = "ConsoleDisplay: ";
  private static final String DESCR   =
    "Redirect console output to a message view";
  private static final String USAGE   =
    "  Redirects the normal console output to a tab in the message pane.\n" +
    "  Optional parameters let you direct just normal output or just\n" +
    "  error output. The default is to send both.\n" +
    "  The name of the message tab reflects the parameter selection.\n\n" +
    "  -consoledisp [out | err]";

  /**
   * Register the new command with JBuilder.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    PrimeTime.registerCommand("consoledisp", new ConsoleDisplay());
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded consoledisp command v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  public ConsoleDisplay() {
  }

  /**
   * Print out more detailed help on this command.
   *
   * @param  out  the stream to write the help to
   */
  public void printCommandHelp(PrintStream out) { out.println(USAGE); }

  /**
   * Return the text for the -help command.
   *
   * @return  a basic description of the command
   */
  public String getCommandDescription() { return DESCR; }

  /**
   * Check parameters and then redirect console output accordingly.
   *
   * @param  params  an array of parameter values for this command
   */
  public void invokeCommand(String[] params) {
    if (params.length > 1) {
      System.out.println(LEADER + USAGE);
      return;
    }
    String param = "";
    if (params.length == 1) {
      param = params[0];
      if (!param.equals("out") && !param.equals("err")) {
        System.err.println(LEADER + USAGE);
        return;
      }
    }
    boolean redirectOut = !param.equals("err");
    boolean redirectErr = !param.equals("out");
    // Create output stream for console
    final MessageOutputStream msgOut = new MessageOutputStream(
      !redirectOut ? "Err" : (!redirectErr ? "Out" : "Console"));
    PrintStream out = new PrintStream(msgOut);
    if (redirectOut) {
      System.setOut(out);
    }
    if (redirectErr) {
      System.setErr(out);
    }
    // Need to wait until the browser opens to display the first console output
    Browser.addStaticBrowserListener(new BrowserAdapter() {
      public void browserOpened(Browser browser) {
        Browser.removeStaticBrowserListener(this);
        msgOut.opened();
      }
    });
  }

  /**
   * Does this command accept any arguments
   *
   * @return  true, since it does accept parameters.
   */
  public boolean takesCommandArguments() { return true; }
}

/**
 * A replacement output stream that sends data to a message view.
 */
class MessageOutputStream extends OutputStream {

  private static final int MIN_SIZE = 80;
  private MessageView mview = null;
  private MessageCategory mcat;
  private StringBuffer buffer = new StringBuffer(MIN_SIZE);

  public MessageOutputStream(String category) {
    mcat = new MessageCategory(category);
  }

  /**
   * Check to see if the message view is available yet.
   *
   * @return  the message view or null if not available
   */
  private MessageView getView() {
    if (mview == null) {
      Browser browser = Browser.getActiveBrowser();
      if (browser != null) {
        mview = browser.getMessageView();
      }
    }
    return mview;
  }

  /**
   * Force the display of any accumulated text.
   */
  public void opened() {
    try {
      write('\n');
    }
    catch (IOException ex) {
      // Ignore
    }
  }

  /**
   * Accumulate characters until a line is completed, then send it
   * to the message view. If the message view is not yet available,
   * then continue accumulating text until it is.
   *
   * @param  b  the character to write out
   */
  public void write(int b) throws IOException {
    switch (b) {
      case '\n':  // LF
        MessageView view = getView();
        if (view == null) {
          buffer.append("\n");
        }
        else {
          StringTokenizer tokens = new StringTokenizer(buffer.toString(), "\n");
          while (tokens.hasMoreTokens()) {
            view.addMessage(mcat, new ConsoleMessage(tokens.nextToken()));
          }
          buffer.setLength(0);
          buffer.ensureCapacity(MIN_SIZE);
        }
        break;
      case '\r':  // CR
        break;
      case '\t':  // Tab
        buffer.append("    ");
        break;
      default:
        buffer.append((char)b);
    }
  }
}

/**
 * A message that knows how to load and highlight error lines
 * within Java classes.
 */
class ConsoleMessage extends Message implements ActionListener {

  public ConsoleMessage(String text) {
    super(text);
    setMessageAction(this);
  }

  /**
   * Scan the given line and open the associated Java class if present.
   */
  public void actionPerformed(ActionEvent e) {
    String command = e.getActionCommand();
    // Extract the class reference
    if (!(command.trim().startsWith("at") && command.endsWith(")"))) {
      return;
    }
    int pos = command.indexOf("(");
    if (pos == -1) {
      return;
    }
    String classRef = command.substring(7, pos);
    // Ignore method call and inner classes
    classRef = classRef.substring(0, classRef.lastIndexOf("."));
    pos = classRef.indexOf("$");
    if (pos != -1) {
      classRef = classRef.substring(0, pos);
    }
    // Convert class reference to file reference
    classRef = classRef.replace('.', '/') + ".java";
    Browser browser = (Browser)e.getSource();
    JBProject project = (JBProject)browser.getActiveProject();
    // Locate this class on the various paths for the project
    Url url = project.findSourcePathUrl(classRef);
    if (url == null) {
      url = project.findClassPathUrl(classRef);
      if (url == null) {
        browser.getStatusView().setText("Can't find " + classRef);
        return;
      }
    }
    // And open it
    FileNode node = project.getNode(url);
    if (node == null) {
      browser.getStatusView().setText("Can't open " + classRef);
      return;
    }
    try {
      browser.openNodes(new Node[] {node}, node);
    }
    catch (Exception ex) {
      // Ignore
      ex.printStackTrace();
    }
    // Extract the line number
    int line = -1;
    pos = command.indexOf(":");
    if (pos != -1) {
      try {
        line = Integer.parseInt(command.substring(pos + 1, command.length() - 1));
      }
      catch (NumberFormatException nfe) {
        // Ignore
      }
    }
    // Try to position cursor at correct line
    if (line != -1 && node instanceof TextFileNode) {
      ((TextFileNode)node).setCaretPosition(line, 1, true);
    }
  }
}
