package wood.keith.opentools.gifeditor;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JDialog;
import javax.swing.colorchooser.AbstractColorChooserPanel;

import com.borland.primetime.actions.UpdateAction;

/**
 * Actions for drawing colours - one each for foreground and background.
 * Each one sets its colour into the editor.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 * @version  1.1  4 December 2002
 */
public class ColorAction extends UpdateAction {

  private PixelEditor editor = null;
  private boolean isForeground = false;
  private Color color = Color.red;

  /**
   * Create an action for this colour.
   *
   * @param  editor  the pixel editor to work with
   * @param  name    the colour name
   * @param  color   the actual colour to use
   */
  public ColorAction(PixelEditor editor, boolean isForeground) {
    super(isForeground ? "Foreground" : "Background");
    this.editor = editor;
    this.isForeground = isForeground;
    setColor(isForeground ? Color.black : Color.white);
    updateEditor();
  }

  /**
   * Retrieve the colour for this action.
   *
   * @return  the action's colour
   */
  public Color getColor() { return color; }

  /**
   * Save the colour for this tool and generate a swatch to show as the icon.
   *
   * @param  color  the colour for this action
   */
  public void setColor(Color color) {
    if (this.color.equals(color)) {
      return;
    }
    this.color = color;
    BufferedImage image = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
    Graphics g = image.getGraphics();
    g.setColor(color);
    g.fillRect(0, 0, 16, 16);
    if (color == PixelEditor.TRANSPARENT) {
      g.setColor(Color.black);
      g.drawLine(8, 8, 8, 8);
    }
    setSmallIcon(new ImageIcon(image));
    updateEditor();
  }

  /**
   * Ask the user for the new colour, and then display it.
   *
   * @param  event  the triggering event
   */
  public void actionPerformed(ActionEvent event) {
    JColorChooser chooser = new JColorChooser(color);
    // Allow selection of transparency
    chooser.addChooserPanel(new TransparentChooser());
    ColorTracker ok = new ColorTracker(chooser);
    JDialog dialog = JColorChooser.createDialog(editor,
      (isForeground ? "Select foreground colour" : "Select background colour"),
      true, chooser, ok, null);
    dialog.addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
        e.getWindow().hide();
      }
    });
    dialog.addComponentListener(new ComponentAdapter() {
      public void componentHidden(ComponentEvent e) {
        ((Window)e.getComponent()).dispose();
      }
    });
    dialog.show();
    if (ok.getColor() == null) {
      return;
    }
    setColor(ok.getColor());
  }

  /**
   * Transfer the colour value to the editor.
   */
  private void updateEditor() {
    if (isForeground) {
      editor.setFGColor(getColor());
    }
    else {
      editor.setBGColor(getColor());
    }
  }

  /**
   * Keep the button synchronised with the editor.
   *
   * @param  source  the source of the update request
   */
  public void update(Object source) {
    setColor(isForeground ? editor.getFGColor() : editor.getBGColor());
  }
}

/**
 * Add a panel to the colour chooser to allow selection of transparency.
 */
class TransparentChooser extends AbstractColorChooserPanel {

  private JButton transparentButton = new JButton("Make Transparent");

  public void updateChooser() {}

  protected void buildChooser() {
    add(transparentButton);
    transparentButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent event) {
        getColorSelectionModel().setSelectedColor(PixelEditor.TRANSPARENT);
      }
    });
  }

  public String getDisplayName() { return "Transparent"; }

  public Icon getSmallDisplayIcon() { return null; }

  public Icon getLargeDisplayIcon() { return null; }
}

/**
 * Remember colour selection for colour chooser.
 */
class ColorTracker implements ActionListener {

  private JColorChooser chooser;
  private Color color;

  public ColorTracker(JColorChooser c) { chooser = c; }

  public void actionPerformed(ActionEvent event) { color = chooser.getColor(); }

  public Color getColor() { return color; }
}