package wood.keith.opentools.gifeditor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Enumeration;
import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import com.borland.jbcl.layout.VerticalFlowLayout;
import com.borland.primetime.help.HelpManager;
import com.borland.primetime.help.ZipHelpBook;
import com.borland.primetime.help.ZipHelpTopic;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.StatusView;
import com.borland.primetime.node.ImageFileNode;

/**
 * GIF Editor as a content pane viewer.
 *
 * Based on JIconEditor created by Claude Duguay - Copyright (c) 1999
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 */
public class GIFEditor extends JPanel implements ActionConstants, DrawConstants,
		PropertyChangeListener, SwingConstants {
	private BorderLayout borderLayout = new BorderLayout();
	private JPanel toolbarPanel = new JPanel();
	private VerticalFlowLayout verticalFlowLayout = new VerticalFlowLayout();
	private JToolBar operationsToolbar = new JToolBar();
	private JToolBar toolsToolbar = new JToolBar();
	private JScrollPane editorScroll = new JScrollPane();
	private PixelEditor editor = new PixelEditor(null, CELL_SIZE);

	private static final int CELL_SIZE = 16;
	private static final Dimension BUTTON_SIZE = new Dimension(23, 23);
	private final ImageOps imageOps = new ImageOps();
	private StatusView status = Browser.getActiveBrowser().getStatusView();
	private ButtonGroup toolsGroup = new ButtonGroup();
	private ButtonGroup colorsGroup = new ButtonGroup();
	private Action resetAction =
		new OperationAction(editor, RESET_FILE, "Reset image") {
			public void actionPerformed(ActionEvent event) {
				this.editor.resetNode();
			}
		};
	private Action saveAction =
		new OperationAction(editor, SAVE_FILE, "Save image") {
			public void actionPerformed(ActionEvent event) {
				this.editor.saveNode();
			}
		};
	private Action zoomInAction =
		new OperationAction(editor, ZOOM_IN, "Zoom in") {
			public void actionPerformed(ActionEvent event) {
				this.editor.setCellSize(this.editor.getCellSize() * 2);
			}
		};
	private Action zoomOutAction =
		new OperationAction(editor, ZOOM_OUT, "Zoom out") {
			public void actionPerformed(ActionEvent event) {
				this.editor.setCellSize(this.editor.getCellSize() / 2);
			}
		};
	private Action helpAction =
		new OperationAction(editor, HELP, "Help") {
			public void actionPerformed(ActionEvent event) {
				HelpManager.showHelp(new ZipHelpTopic(
					new ZipHelpBook("GIFEditorDoc.jar"),
					getClass().getPackage().getName().replace('.', '/') +
					"/GIFEditor.html"));
			}
		};
	private ColorAction foregroundAction = null;
	private ColorAction backgroundAction = null;

	/**
	 * Create the new editor component and load it from the supplied node.
	 *
	 * @param  node  the file node containing the image
	 */
	public GIFEditor(ImageFileNode node) {
		try {
			jbInit();
			editor.setNode(node);
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Create the UI of the editor.
	 */
	private void jbInit() throws Exception {
		this.setLayout(borderLayout);
		toolbarPanel.setLayout(verticalFlowLayout);
		verticalFlowLayout.setHgap(0);
		verticalFlowLayout.setVgap(0);
		operationsToolbar.setName("Image operations");
		toolsToolbar.setName("Drawing tools");
		this.add(editorScroll, BorderLayout.CENTER);
		this.add(toolbarPanel, BorderLayout.NORTH);
		toolbarPanel.add(operationsToolbar, null);
		toolbarPanel.add(toolsToolbar, null);
		editorScroll.getViewport().add(editor, null);
		editor.addPropertyChangeListener(this);

		// Add operation actions
		operationsToolbar.add(
			new OperationAction(editor, ROTATE_LEFT, "Rotate anticlockwise") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.rotateImage(this.editor.getImage(), LEFT));
				}
			});
		operationsToolbar.add(
			new OperationAction(editor, ROTATE_RIGHT, "Rotate clockwise") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.rotateImage(this.editor.getImage(), RIGHT));
				}
			});
		operationsToolbar.add(
			new OperationAction(editor, FLIP_VERT, "Flip vertically") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.flipImage(this.editor.getImage(), VERTICAL));
				}
			});
		operationsToolbar.add(
			new OperationAction(editor, FLIP_HORZ, "Flip horizontally") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.flipImage(this.editor.getImage(), HORIZONTAL));
				}
			});
		operationsToolbar.addSeparator();
		operationsToolbar.add(
			new OperationAction(editor, MOVE_NORTH_WEST, "Move up and left") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), -1, -1));
				}
			});
		operationsToolbar.add(new OperationAction(editor, MOVE_NORTH, "Move up") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), 0, -1));
				}
			});
		operationsToolbar.add(
			new OperationAction(editor, MOVE_NORTH_EAST, "Move up and right") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), +1, -1));
				}
			});
		operationsToolbar.add(new OperationAction(editor, MOVE_WEST, "Move left") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), -1, 0));
				}
			});
		operationsToolbar.add(new OperationAction(editor, MOVE_EAST, "Move right") {
			public void actionPerformed(ActionEvent event) {
				this.editor.setImage(
					imageOps.moveImage(this.editor.getImage(), +1, 0));
			}
		});
		operationsToolbar.add(
			new OperationAction(editor, MOVE_SOUTH_WEST, "Move down and left") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), -1, +1));
				}
			});
		operationsToolbar.add(new OperationAction(editor, MOVE_SOUTH, "Move down") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), 0, +1));
				}
			});
		operationsToolbar.add(
			new OperationAction(editor, MOVE_SOUTH_EAST, "Move down and right") {
				public void actionPerformed(ActionEvent event) {
					this.editor.setImage(
						imageOps.moveImage(this.editor.getImage(), +1, +1));
				}
			});
		operationsToolbar.addSeparator();
		operationsToolbar.add(zoomInAction);
		operationsToolbar.add(zoomOutAction);
		operationsToolbar.addSeparator();
		operationsToolbar.add(saveAction);
		operationsToolbar.add(resetAction);
		operationsToolbar.add(new OperationAction(editor, RESIZE, "Resize image") {
			public void actionPerformed(ActionEvent event) {
				ResizeDialog resize = new ResizeDialog(this.editor.image.getWidth(),
					this.editor.image.getHeight());
				if (!resize.isCancelled()) {
					this.editor.resizeImage(resize.getWidth(), resize.getHeight());
				}
			}
		});
		operationsToolbar.addSeparator();
		operationsToolbar.add(helpAction);

		// Add tool actions
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, HAND, new Point(8, 8), "Move"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, CUT, new Point(2, 2), "Cut"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, COPY, new Point(2, 2), "Copy"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, PASTE, new Point(2, 2), "Paste"));
		toolsToolbar.addSeparator();
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, PENCIL, new Point(4, 15), "Draw"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, DROPPER, new Point(0, 15), "Pickup colour"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, PAINT, new Point(13, 14), "Fill"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, LINE, new Point(0, 0), "Line"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, OVAL, new Point(0, 0), "Oval"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, RECT, new Point(0, 0), "Rectangle"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, ROUND_RECT, new Point(0, 0), "Rounded rectangle"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, FILL_OVAL, new Point(0, 0), "Filled oval"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, FILL_RECT, new Point(0, 0), "Filled rectangle"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, FILL_ROUND_RECT, new Point(0, 0),
			"Filled rounded rectangle"));
		addToggleButton(toolsToolbar, toolsGroup,
			new ToolAction(editor, TRANSPOSE, new Point(0, 0), "Transpose colours"));

		// Add colour actions
		toolsToolbar.addSeparator();
		foregroundAction = new ColorAction(editor, true);
		JButton button = new JButton(foregroundAction);
		button.setPreferredSize(BUTTON_SIZE);
		button.setMaximumSize(BUTTON_SIZE);
		toolsToolbar.add(button);
		button = new JButton(new SwapColorsAction());
		button.setPreferredSize(BUTTON_SIZE);
		button.setMaximumSize(BUTTON_SIZE);
		toolsToolbar.add(button);
		backgroundAction = new ColorAction(editor, false);
		button = new JButton(backgroundAction);
		button.setPreferredSize(BUTTON_SIZE);
		button.setMaximumSize(BUTTON_SIZE);
		toolsToolbar.add(button);
	}

	/**
	 * Add an action as part of a group.
	 *
	 * @param  toolbar  the toolbar to add the resulting button to
	 * @param  group    the group for this action
	 * @param  action   the underlying action
	 */
	private void addToggleButton(JToolBar toolbar, ButtonGroup group,
			Action action) {
		JToggleButton button = new JToggleButton(action);
		button.setPreferredSize(BUTTON_SIZE);
		button.setMaximumSize(BUTTON_SIZE);
		button.setSelected(group.getButtonCount() == 0);
		group.add(button);
		toolbar.add(button);
	}

	/**
	 * Pass property listeners on to the editor itself.
	 *
	 * @param  propertyName  the name of the property of interest
	 * @param  listener      the listener to send change notifications to
	 */
	public synchronized void addPropertyChangeListener(String propertyName,
			PropertyChangeListener listener) {
		editor.addPropertyChangeListener(propertyName, listener);
	}

	/**
	 * Display the image coordinates in the status bar,
	 * reset colour selection when sucked up,
	 * and enable buttons when the image is modified.
	 *
	 * @param  event  the event for this property change
	 */
	public void propertyChange(PropertyChangeEvent event) {
		if (event.getPropertyName().equals(PixelEditor.COORDS_PROP)) {
			Point point = (Point)event.getNewValue();
			status.setHintText(
			 "GIF Editor " + (int)point.getX() + "," + (int)point.getY());
		}
		else if (event.getPropertyName().equals(PixelEditor.COLOR_PROP)) {
			foregroundAction.update(editor);
			backgroundAction.update(editor);
		}
		else if (event.getPropertyName().equals(PixelEditor.IMAGE_PROP)) {
			enableToolbarButtons(operationsToolbar, !editor.isReadOnly());
			zoomInAction.setEnabled(true);
			zoomOutAction.setEnabled(true);
			saveAction.setEnabled(editor.isModified());
			resetAction.setEnabled(editor.isModified());
			helpAction.setEnabled(true);
			enableToolbarButtons(toolsToolbar, !editor.isReadOnly());
		}
		else if (event.getPropertyName().equals(PixelEditor.TOOL_PROP)) {
			Enumeration enum = toolsGroup.getElements();
			while (enum.hasMoreElements()) {
				JToggleButton button = (JToggleButton)enum.nextElement();
				ToolAction action = (ToolAction)button.getAction();
				if (action.getToolName().equals(event.getNewValue())) {
					toolsGroup.setSelected(button.getModel(), true);
					action.setState(editor, true);
					button.repaint();
					break;
				}
			}
		}
	}

	/**
	 * Dis/enable all actions associated with a given toolbar.
	 *
	 * @param  toolbar  the toolbar to scan
	 * @param  value    true to enable, false to disable actions
	 */
	private void enableToolbarButtons(JToolBar toolbar, boolean value) {
		for (int index = 0; index < toolbar.getComponentCount(); index++) {
			if (toolbar.getComponent(index) instanceof AbstractButton) {
				((AbstractButton)toolbar.getComponent(index)).
					getAction().setEnabled(value);
			}
		}
	}

	/**
	 * An action to swap the background and foreground colours.
	 */
	class SwapColorsAction extends AbstractAction {

		public SwapColorsAction() {
			super("");
			putValue(Action.SMALL_ICON, new ImageIcon(ClassLoader.getSystemResource(
				"wood/keith/opentools/gifeditor/SwapColorsIcon.gif")));
			putValue(Action.SHORT_DESCRIPTION, "Swap background/foreground");
		}

		public void actionPerformed(ActionEvent event) {
			Color temp = backgroundAction.getColor();
			backgroundAction.setColor(foregroundAction.getColor());
			foregroundAction.setColor(temp);
		}
	}
}