package wood.keith.opentools.gifeditor;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import javax.swing.SwingConstants;

/**
 * Generic operations on an image.
 *
 * Based on ImageOps created by Claude Duguay - Copyright (c) 1999
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 */
public class ImageOps implements SwingConstants {

	/**
	 * Move the original image one pixel in the direction indicated.
	 *
	 * @param  image  the original image
	 * @param  x      the x offset (-1 to +1)
	 * @param  y      the y offset (-1 to +1)
	 * @return  the altered image
	 */
	protected BufferedImage moveImage(BufferedImage image, int x, int y) {
		BufferedImage buffer = new BufferedImage(
			image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_ARGB);
		Graphics g = buffer.getGraphics();
		g.drawImage(image, x, y, null);
		return buffer;
	}

	/**
	 * Flip the original image horizontally or vertically.
	 *
	 * @param  image      the original image
	 * @param  direction  HORIZONTAL or VERTICAL
	 * @return  the altered image
	 */
	protected BufferedImage flipImage(BufferedImage image, int direction) {
		int w = image.getWidth();
		int h = image.getHeight();
		BufferedImage buffer = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
		Graphics g = buffer.getGraphics();
		if (direction == HORIZONTAL) {
			g.drawImage(image, w, 0, 0, h, 0, 0, w, h, null);
		}
		else {
			g.drawImage(image, 0, h, w, 0, 0, 0, w, h, null);
		}
		return buffer;
	}

	/**
	 * Rotate the original image 90 degrees in the direction indicated.
	 *
	 * @param  image      the original image
	 * @param  direction  RIGHT (clockwise) or LEFT (anticlockwise)
	 * @return  the altered image
	 */
	protected BufferedImage rotateImage(BufferedImage image, int direction) {
		int w = image.getWidth();
		int h = image.getHeight();
		int max = Math.max(w, h);
		BufferedImage buffer =
			new BufferedImage(max, max, BufferedImage.TYPE_INT_ARGB);
		Graphics2D g = (Graphics2D)buffer.getGraphics();
		if (direction == RIGHT) {
			g.rotate(Math.PI / 2, max / 2, max / 2);
		}
		else if (direction == LEFT) {
			g.rotate(-(Math.PI / 2), max / 2, max / 2);
		}
		g.drawImage(image, 0, 0, null);
		if (w == h) {
			return buffer;
		}
		BufferedImage result = new BufferedImage(h, w, BufferedImage.TYPE_INT_ARGB);
		g = (Graphics2D)result.getGraphics();
		if (direction == RIGHT && w > h) {
			g.translate(h - w, 0);
		}
		else if (direction == LEFT && h > w) {
			g.translate(0, w - h);
		}
		g.drawImage(buffer, 0, 0, null);
		return result;
	}

	/**
	 * Fill the original image from the starting point,
	 * changing one colour to another.
	 *
	 * @param  image     the original image
	 * @param  x         the x coordinate
	 * @param  y         the y coordinate
	 * @param  oldColor  the original colour being replaced
	 * @partam newColor  the replacement colour
	 * @return  the altered image
	 */
	protected void floodFill(BufferedImage buffer, int x, int y,
			int oldColor, int newColor) {
		int w = buffer.getWidth();
		int h = buffer.getHeight();
		if (x < 0 || x >= w || y < 0 || y >= h) {
			return;
		}
		int pos = x;
		while (buffer.getRGB(pos, y) == oldColor) {
			buffer.setRGB(pos, y, newColor);
			if (y > 0 && buffer.getRGB(pos, y - 1) == oldColor) {
				floodFill(buffer, pos, y - 1, oldColor, newColor);
			}
			if ((y + 1) < h && buffer.getRGB(pos, y + 1) == oldColor) {
				floodFill(buffer, pos, y + 1, oldColor, newColor);
			}
			pos--;
			if (pos < 0) {
				break;
			}
		}
		pos = x + 1;
		if (pos >= w) {
			return;
		}
		while (buffer.getRGB(pos, y) == oldColor) {
			buffer.setRGB(pos, y, newColor);
			if (y > 0 && buffer.getRGB(pos, y - 1) == oldColor) {
				floodFill(buffer, pos, y - 1, oldColor, newColor);
			}
			if ((y + 1) < h && buffer.getRGB(pos, y + 1) == oldColor) {
				floodFill(buffer, pos, y + 1, oldColor, newColor);
			}
			pos++;
			if (pos >= w) {
				break;
			}
		}
	}
}