package wood.keith.opentools.gifeditor;

import java.awt.Image;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.image.PixelGrabber;
import java.io.IOException;

/**
 * Support for an image on the clipboard.
 * Based on example from Mastering the JFC Volume 1.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.1  11 December 2002
 */
public class ImageSelection implements ClipboardOwner, Transferable {

	public static DataFlavor ImageFlavor;
	public static DataFlavor ImageArrayFlavor;

	private DataFlavor[] flavors = {ImageFlavor, ImageArrayFlavor};
	private Image image;
	private int width, height;

	static {
		ImageFlavor = new DataFlavor(Image.class, "AWT Image");
		ImageArrayFlavor = new DataFlavor("image/gif", "GIF Image");
	}

	/**
	 * Prepare an image for transfer.
	 *
	 * @param  image   the image to copy
	 * @param  width   its width
	 * @param  height  its height
	 */
	public ImageSelection(Image image, int width, int height) {
		this.image = image;
		this.width = width;
		this.height = height;
	}

	/**
	 * Return the data flavours supported by this transfer.
	 *
	 * @return  the data flavours
	 */
	public synchronized DataFlavor[] getTransferDataFlavors() { return flavors; }

	/**
	 * Determine whether a particular data flavour is supported.
	 *
	 * @param  flavor  the requested data flavour
	 * @return  true if supported, false otherwise
	 */
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		return flavor.equals(ImageFlavor) || flavor.equals(ImageArrayFlavor);
	}

	/**
	 * Supply the content of the image in whatever flavour is requested.
	 *
	 * @param  flavor  the requested data flavour
	 * @return  the image data
	 * @throws  IOException  if a problem in reading the data
	 * @throws  UnsupportedFlavorException  if an unknown flavour is requested
	 */
	public synchronized Object getTransferData(DataFlavor flavor)
			throws IOException, UnsupportedFlavorException {
		if (flavor.equals(ImageFlavor)) {
			return image;
		}
		else if (flavor.equals(ImageArrayFlavor)) {
			return imageToArray();
		}
		else {
			throw new UnsupportedFlavorException(flavor);
		}
	}

	/**
	 * Convert the image object into an array of pixels.
	 *
	 * @return  the pixel array
	 */
	private int[] imageToArray() {
		int[] pixels = new int[width * height];
		PixelGrabber pg =
			new PixelGrabber(image, 0, 0, width, height, pixels, 0, width);
		try {
			pg.grabPixels();
		}
		catch (InterruptedException ie) {
			ie.printStackTrace();
		}
		return pixels;
	}

	public void lostOwnership(Clipboard clipboard, Transferable contents) {}
}