package wood.keith.opentools.gifeditor;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeListener;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.border.EmptyBorder;

import Acme.JPM.Encoders.GifEncoder;

import com.borland.primetime.node.ImageFileNode;
import com.borland.primetime.vfs.Buffer;
import com.borland.primetime.vfs.BufferListener;
import com.borland.primetime.vfs.BufferUpdater;

/**
 * Simple pixel editor.
 * Based on PixelEditor created by Claude Duguay - Copyright (c) 1999
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 * @version  1.1  4 December 2002
 */
public class PixelEditor extends PixelView
		implements DrawConstants, ActionConstants, MouseListener,
		MouseMotionListener, BufferListener, BufferUpdater {
	/* Properties of possible interest to external parties. */
	public static final String COLOR_PROP  = "Color";
	public static final String COORDS_PROP = "Coords";
	public static final String IMAGE_PROP  = "Image";
	public static final String TOOL_PROP   = "Tool";

	public static final Clipboard clipboard = new Clipboard("GIFEditor");

	protected static final Color TRANSPARENT = new Color(0, true);
	protected static final Color TRANSPARENT_SHOW = new Color(126, 125, 128);

	protected String activeTool = PENCIL;
	protected String lastTool = NONE;
	protected Color fgColor = Color.black;
	protected Color bgColor = TRANSPARENT;
	protected Color curColor = fgColor;
	protected BufferedImage move;
	protected Rectangle drag;
	protected ImageOps imageOps;
	protected boolean antiAliased = false;

	protected boolean modified = false;
	protected boolean readonly = false;
	protected ImageFileNode imageNode = null;
	private Point point = new Point(0, 0);

	/**
	 * Construct a new editor for the supplied image.
	 *
	 * @param  image     the image to work with
	 * @param  cellSize  the initial cell size to display at
	 */
	public PixelEditor(BufferedImage image, int cellSize) {
		super(image, cellSize);
		imageOps = new ImageOps();
		setBorder(new EmptyBorder(4, 4, 4, 4));
		addMouseListener(this);
		addMouseMotionListener(this);
	}

	/**
	 * Respond to changes in the state of the image buffer within JBuilder.
	 *
	 * @param  buffer    the buffer being changed
	 * @param  oldState  the previous settings for that buffer
	 * @param  newState  the current settings for that buffer
	 */
	public void bufferStateChanged(Buffer buffer, int oldState, int newState) {
		setReadOnly((Buffer.STATE_READONLY & newState) > 0);
	}

	public void bufferSaving(Buffer buffer) {
		// Do nothing
	}

	public void bufferLoaded(Buffer buffer) {
		// Do nothing
	}

	/**
	 * If the buffer is changed by someone other than us, then reload the image.
	 *
	 * @param  buffer   the buffer being changed
	 * @param  updater  the lazy updater for this buffer
	 */
	public void bufferChanged(Buffer buffer, BufferUpdater updater) {
		if (updater != this) {
			loadImage(buffer);
		}
	}

	public boolean isReadOnly() { return readonly; }

	public void setReadOnly(boolean value) {
		readonly = value;
		if (readonly) {
			lastTool = activeTool;
			activeTool = NONE;
			firePropertyChange(TOOL_PROP, lastTool, activeTool);
		}
	}

	public boolean isModified() { return modified; }

	public boolean isAntiAliased() { return antiAliased; }

	public void setAntiAliased(boolean value) { antiAliased = value; }

	/**
	 * Tell the node buffer that changes are available,
	 * and notify interested parties of the change.
	 * The buffer is not actually updated until necessary,
	 * via the use of a BufferUpdater.
	 *
	 * @param  value  whether or not the image has been modified
	 */
	public void setModified(boolean value) {
		modified = value;
		firePropertyChange(IMAGE_PROP, null, image);
		if (modified && imageNode != null && !isReadOnly()) {
			try {
				imageNode.getBuffer().setContent(this);
			}
			catch (IOException ioe) {
				ioe.printStackTrace();
			}
		}
	}

	/**
	 * Supply the changes to the buffer only on request.
	 *
	 * @param  buffer  the buffer being updated
	 */
	public byte[] getBufferContent(Buffer buffer) {
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			new GifEncoder(image, out).encode();
			return out.toByteArray();
		}
		catch (IOException ioe) {
			ioe.printStackTrace();
		}
		return new byte[] {0};
	}

	/**
	 * Link this editor to a JBuilder image node.
	 *
	 * @param  node  the node to attach to
	 */
	public void setNode(ImageFileNode node) {
		if (imageNode != null) {
			try {
				imageNode.getBuffer().removeBufferListener(this);
			}
			catch (IOException ex) {
				// Ignore
			}
		}
		imageNode = node;

		try {
			setReadOnly(imageNode.getBuffer().isReadOnly());
			imageNode.getBuffer().addBufferListener(this);
		}
		catch (IOException ex) {
			// Ignore
		}
		loadImage();
	}

	public Color getFGColor() { return fgColor; }

	public void setFGColor(Color color) { fgColor = color; }

	public Color getBGColor() { return bgColor; }

	public void setBGColor(Color color) { bgColor = color; }

	public BufferedImage getImage() { return image; }

	public void setImage(BufferedImage image) {
		this.image = image;
		repaint();
		setModified(true);
	}

	public String getToolName() { return activeTool; }

	public void setToolName(String toolName) {
		if (activeTool.equals(toolName)) {
			return;
		}
		lastTool = activeTool;
		activeTool = toolName;
		firePropertyChange(TOOL_PROP, lastTool, activeTool);
	}

	/* Immediately notify interested parties of the image. */
	public synchronized void addPropertyChangeListener(String propertyName,
			PropertyChangeListener listener) {
		super.addPropertyChangeListener(propertyName, listener);
		if (propertyName.equals(IMAGE_PROP)) {
			firePropertyChange(IMAGE_PROP, null, image);
		}
	}

	public void paintComponent(Graphics g) {
		super.paintComponent(g);
		if (drag != null) {
			drawSelection(g, getInsets());
		}
	}

	/**
	 * Provide feedback during drag operations.
	 *
	 * @param  g       the graphics context to draw on
	 * @param  insets  the insets for the panel
	 */
	protected void drawSelection(Graphics g, Insets insets) {
		g.setColor(curColor == TRANSPARENT ? TRANSPARENT_SHOW : curColor);
		int halfCell = cellSize / 2;
		int corner = cellSize * 3;
		Rectangle scaled = new Rectangle(insets.left + drag.x * cellSize + halfCell,
			insets.top + drag.y * cellSize + halfCell,
			drag.width * cellSize, drag.height * cellSize);
		if (!activeTool.equals(LINE)) {
      scaled = adjustDrag(scaled);
    }
		if (activeTool.equals(LINE)) {
			g.drawLine(scaled.x, scaled.y,
				scaled.x + scaled.width, scaled.y + scaled.height);
		}
		else if (activeTool.equals(FILL_OVAL)) {
			g.fillOval(scaled.x, scaled.y, scaled.width, scaled.height);
		}
		else if (activeTool.equals(OVAL) || activeTool.equals(FILL_OVAL)) {
			g.drawOval(scaled.x, scaled.y, scaled.width, scaled.height);
		}
		else if (activeTool.equals(FILL_RECT)) {
			g.fillRect(scaled.x, scaled.y, scaled.width, scaled.height);
		}
		else if (activeTool.equals(RECT) || activeTool.equals(TRANSPOSE) ||
				activeTool.equals(CUT) || activeTool.equals(COPY) ||
				activeTool.equals(PASTE)) {
			g.drawRect(scaled.x, scaled.y, scaled.width, scaled.height);
		}
		else if (activeTool.equals(FILL_ROUND_RECT)) {
			g.fillRoundRect(scaled.x, scaled.y,
				scaled.width, scaled.height, corner, corner);
		}
		else if (activeTool.equals(ROUND_RECT)) {
			g.drawRoundRect(scaled.x, scaled.y,
				scaled.width, scaled.height, corner, corner);
		}
	}

	public void mouseClicked(MouseEvent event) {}
	public void mouseEntered(MouseEvent event) {}
	public void mouseExited(MouseEvent event) {}

	/**
	 * Finish up drag operations.
	 *
	 * @param  event  the event holding the mouse details
	 */
	public void mouseReleased(MouseEvent event) {
		if (activeTool.equals(DROPPER)) {
			// Return to previous tool
			setToolName(lastTool);
		}
		else if (drag != null) {
			if (!activeTool.equals(LINE)) {
        drag = adjustDrag(drag);
      }
			Graphics2D g = (Graphics2D)image.getGraphics();
			curColor = (isForeground(event) ? fgColor : bgColor);
			// Cannot draw a transparent colour - so replace temporarily
			g.setColor(curColor == TRANSPARENT ? TRANSPARENT_SHOW : curColor);
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, (antiAliased ?
				RenderingHints.VALUE_ANTIALIAS_ON : RenderingHints.VALUE_ANTIALIAS_OFF));
			boolean modified = true;
			if (activeTool.equals(LINE)) {
				g.drawLine(drag.x, drag.y, drag.x + drag.width, drag.y + drag.height);
			}
			else if (activeTool.equals(FILL_OVAL)) {
				g.fillOval(drag.x, drag.y, drag.width + 1, drag.height + 1);
			}
			else if (activeTool.equals(OVAL)) {
				g.drawOval(drag.x, drag.y, drag.width, drag.height);
			}
			else if (activeTool.equals(FILL_RECT)) {
				g.fillRect(drag.x, drag.y, drag.width + 1, drag.height + 1);
			}
			else if (activeTool.equals(RECT)) {
				g.drawRect(drag.x, drag.y, drag.width, drag.height);
			}
			else if (activeTool.equals(FILL_ROUND_RECT)) {
				g.fillRoundRect(drag.x, drag.y, drag.width + 1, drag.height + 1, 5, 5);
			}
			else if (activeTool.equals(ROUND_RECT)) {
				g.drawRoundRect(drag.x, drag.y, drag.width, drag.height, 4, 4);
			}
			else if (activeTool.equals(TRANSPOSE)) {
				for (int x = 0; x < drag.width; x++) {
					for (int y = 0; y < drag.height; y++) {
						if (image.getRGB(drag.x + x, drag.y + y) == fgColor.getRGB()) {
							image.setRGB(drag.x + x, drag.y + y, bgColor.getRGB());
						}
					}
				}
			}
			else if (activeTool.equals(CUT) || activeTool.equals(COPY)) {
				move = new BufferedImage(drag.width + 1, drag.height + 1,
					BufferedImage.TYPE_INT_ARGB);
				Graphics2D g2 = (Graphics2D)move.getGraphics();
				g2.drawImage(image, 0, 0, drag.width + 1, drag.height + 1,
					drag.x, drag.y, drag.x + drag.width + 1, drag.y + drag.height + 1, this);
				ImageSelection selection =
					new ImageSelection(move, drag.width + 1, drag.height + 1);
				clipboard.setContents(selection, selection);
				if (activeTool.equals(CUT)) {
          curColor = bgColor;
					g.setColor(bgColor == TRANSPARENT ? TRANSPARENT_SHOW : bgColor);
					g.fillRect(drag.x, drag.y, drag.width + 1, drag.height + 1);
				}
        else {
          modified = false;
        }
			}
			else if (activeTool.equals(PASTE)) {
				Transferable contents = clipboard.getContents(this);
				if (contents != null &&
						contents.isDataFlavorSupported(ImageSelection.ImageFlavor)) {
					try {
						Image newImage =
							(Image)contents.getTransferData(ImageSelection.ImageFlavor);
						g.drawImage(newImage,
							drag.x, drag.y, drag.x + drag.width + 1, drag.y + drag.height + 1,
							0, 0, Math.min(image.getWidth(), drag.width + 1),
							Math.min(image.getHeight(), drag.height + 1), this);
					}
					catch (IOException ioe) {
						ioe.printStackTrace();
					}
					catch (UnsupportedFlavorException ufe) {
						ufe.printStackTrace();
					}
				}
			}
			else {
				modified = false;
			}
			if (modified) {
				// Replace transparent display colour if necessary
				checkForTransparency();
				setModified(true);
			}
		}
		move = null;
		drag = null;
		repaint();
	}

  /**
   * Adjust drag rectangle to cater for reverse drags.
   *
   * @param  drag  the original drag rectangle
   * @return  the adjusted rectangle
   */
  private Rectangle adjustDrag(Rectangle drag) {
    Rectangle adjusted = (Rectangle)drag.clone();
    if (adjusted.width < 0) {
      adjusted.x = adjusted.x + adjusted.width;
      adjusted.width = -adjusted.width;
    }
    if (adjusted.height < 0) {
      adjusted.y = adjusted.y + adjusted.height;
      adjusted.height = -adjusted.height;
    }
    return adjusted;
  }

	/**
	 * Replace occurrences of the transparent display colour with actual
	 * transparency within the current drawing rectangle.
	 */
	private void checkForTransparency() {
		if (curColor != TRANSPARENT) {
			return;
		}
		for (int x = 0; x < drag.width + 1; x++) {
      if (drag.x + x >= 0 && drag.x + x < image.getWidth()) {
  			for (int y = 0; y < drag.height + 1; y++) {
          if (drag.y + y >= 0 && drag.y + y < image.getHeight()) {
            if (image.getRGB(drag.x + x, drag.y + y) == TRANSPARENT_SHOW.getRGB()) {
              image.setRGB(drag.x + x, drag.y + y, TRANSPARENT.getRGB());
            }
          }
        }
			}
		}
	}

	/**
	 * Start a drag or move operation, colour/erase/sample an individual pixel,
	 * or invoke a fill operation.
	 *
	 * @param  event  the event holding the mouse details
	 */
	public void mousePressed(MouseEvent event) {
		Insets insets = getInsets();
		int x = (event.getX() - insets.left) / cellSize;
		int y = (event.getY() - insets.top) / cellSize;
		if (x < 0 || x >= image.getWidth() || y < 0 || y >= image.getHeight()) {
			return;
		}
		curColor = (isForeground(event) ? fgColor : bgColor);
		if (activeTool.equals(PENCIL)) {
			image.setRGB(x, y, curColor.getRGB());
			setModified(true);
		}
		else if (activeTool.equals(DROPPER)) {
			// Pick up new colour
			Color newColor = (image.getRGB(x, y) == TRANSPARENT.getRGB() ?
				TRANSPARENT : new Color(image.getRGB(x, y)));
			if (isForeground(event)) {
				fgColor = newColor;
			}
			else {
				bgColor = newColor;
			}
			firePropertyChange(COLOR_PROP, curColor, newColor);
		}
		else if (activeTool.equals(PAINT)) {
			// Fill the current pixel and surrounding ones of the same colour
			imageOps.floodFill(image, x, y, image.getRGB(x, y), curColor.getRGB());
			setModified(true);
		}
		else if (activeTool.equals(PENCIL) || activeTool.equals(LINE) ||
				activeTool.equals(OVAL) || activeTool.equals(FILL_OVAL) ||
				activeTool.equals(RECT) || activeTool.equals(FILL_RECT) ||
				activeTool.equals(ROUND_RECT) || activeTool.equals(FILL_ROUND_RECT) ||
				activeTool.equals(TRANSPOSE) || activeTool.equals(CUT) ||
				activeTool.equals(COPY) || activeTool.equals(PASTE)) {
			// Prepare for a drag operation
			drag = new Rectangle(x, y, 0, 0);
		}
		else if (activeTool.equals(HAND)) {
			// Prepare for a move operation
			move = new BufferedImage(image.getWidth(), image.getHeight(),
				BufferedImage.TYPE_INT_ARGB);
			Graphics g = move.getGraphics();
			g.drawImage(image, 0, 0, this);
			drag = new Rectangle(x, y, 0, 0);
		}
		repaint();
	}

	private boolean isForeground(MouseEvent event) {
		return ((event.getModifiers() & MouseEvent.BUTTON1_MASK) != 0);
	}
	/**
	 * Update the current coordinates.
	 *
	 * @param  event  the event holding the mouse details
	 */
	public void mouseMoved(MouseEvent event) {
		Insets insets = getInsets();
		int x = (event.getX() - insets.left) / cellSize;
		int y = (event.getY() - insets.top) / cellSize;
		point.setLocation(x, y);
		if (!(x < 0 || x >= image.getWidth() || y < 0 || y >= image.getHeight())) {
			firePropertyChange(COORDS_PROP, null, point);
		}
	}

	/**
	 * Provide feedback during a drag or move operation.
	 *
	 * @param  event  the event holding the mouse details
	 */
	public void mouseDragged(MouseEvent event) {
		mouseMoved(event);
		curColor = (isForeground(event) ? fgColor : bgColor);
		int x = (int)point.getX();
		int y = (int)point.getY();
		if (move != null) {
			image = new BufferedImage(image.getWidth(), image.getHeight(),
				BufferedImage.TYPE_INT_ARGB);
			Graphics g = image.getGraphics();
			g.drawImage(move, x - drag.x, y - drag.y, this);
			setModified(true);
		}
		if (drag != null) {
			drag.setSize(x - drag.x, y - drag.y);
			repaint();
		}

		if (x < 0 || x >= image.getWidth() || y < 0 || y >= image.getHeight()) {
			return;
		}
		if (activeTool.equals(PENCIL)) {
			// Colouring individual pixels
			image.setRGB(x, y, curColor.getRGB());
			setModified(true);
			repaint();
		}
	}

	/**
	 * Resize the image to the new dimensions.
	 *
	 * @param  width   the new width of the image (pixels)
	 * @param  height  the new height of the image (pixels)
	 */
	protected void resizeImage(int width, int height) {
		BufferedImage buffer = new BufferedImage(width, height,
			BufferedImage.TYPE_INT_ARGB);
		Graphics g = buffer.getGraphics();
		g.drawImage(image, 0, 0, null);
		image = buffer;
		repaint();
		setModified(true);
	}

	/**
	 * Save any changes out to the original file.
	 */
	protected void saveNode() {
		try {
			imageNode.save();
		}
		catch (IOException ex) {
			JOptionPane.showMessageDialog(null, "Error during save:\n" +
				ex.getMessage(), "GIF Editor", JOptionPane.ERROR_MESSAGE);
		}
	}

	/**
	 * Discard any changes and return the image to its previous state.
	 */
	protected void resetNode() {
		try {
			imageNode.revert();
		}
		catch (IOException ioe) {
			ioe.printStackTrace();
		}
		loadImage();
	}

	/**
	 * Retrieve the image from its file node.
	 */
	public void loadImage() {
		try {
			loadImage(imageNode.getBuffer());
		}
		catch (IOException ioe) {
			ioe.printStackTrace();
		}
	}

	/**
	 * Retrieve the image from the specified buffer.
	 *
	 * @param  buffer  the buffer containing the image
	 */
	public void loadImage(Buffer buffer) {
		Image img = new ImageIcon(buffer.getBufferContent(buffer)).getImage();
		image = new BufferedImage(
			(img.getWidth(null) == -1 ? 16 : img.getWidth(null)),
			(img.getHeight(null) == -1 ? 16 : img.getHeight(null)),
			BufferedImage.TYPE_INT_ARGB);
		Graphics g = image.getGraphics();
		g.drawImage(img, 0, 0, null);
		repaint();
		setModified(false);
	}
}