package wood.keith.opentools.gifeditor;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.image.BufferedImage;
import javax.swing.JPanel;

/**
 * Viewer for pixel-based image in large size.
 *
 * Based on PixelView created by Claude Duguay - Copyright (c) 1999
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 */
public class PixelView extends JPanel {

  protected int cellSize;
  protected BufferedImage image;

  /**
   * Create a new pixel grid display.
   *
   * @param  image     the image to display in the grid
   * @param  cellSize  the size of each cell
   */
  public PixelView(BufferedImage image, int cellSize) {
    this.image = (image != null ? image :
      new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB));
    this.cellSize = cellSize;
  }

  public int getCellSize() { return cellSize; }

  /**
   * Change the cell size used to display each pixel.
   *
   * @param  value  the new cell size in pixels
   */
  public void setCellSize(int value) {
    if (value != cellSize && value >= 4 && value <= 128) {
      cellSize = value;
      repaint();
    }
  }

  /**
   * The preferred size of the grid allows for the cell sizes and grid lines,
   * as well as any insets.
   */
  public Dimension getPreferredSize() {
    Insets insets = getInsets();
    int width = image.getWidth();
    int height = image.getHeight();
    return new Dimension(insets.left + insets.right + width * cellSize + 1,
      insets.top + insets.bottom + height * cellSize + 1);
  }

  /**
   * Draw the entire grid and cells.
   *
   * @param  g  the graphics context to draw to
   */
  public void paintComponent(Graphics g) {
    g.setColor(getBackground());
    g.fillRect(0, 0, getSize().width, getSize().height);

    Insets insets = getInsets();
    int width = image.getWidth();
    int height = image.getHeight();

    // Draw grid
    g.setColor(Color.lightGray);
    int xOffset = 0;
    int yOffset = insets.top + height * cellSize;
    for (int x = 0; x <= width; x++) {
      xOffset = insets.left + x * cellSize;
      g.drawLine(xOffset, insets.top, xOffset, yOffset);
    }
    xOffset = insets.left + width * cellSize;
    for (int y = 0; y <= height; y++) {
      yOffset = insets.top + y * cellSize;
      g.drawLine(insets.left, yOffset, xOffset, yOffset);
    }
    // Draw cells
    for (int x = 0; x < width; x++) {
      for (int y = 0; y < height; y++) {
        drawCell(g, insets, x, y, image.getRGB(x, y));
      }
    }
  }

  /**
   * Draw an individual cell in its colour (may be transparent).
   *
   * @param  g       the graphics context to draw to
   * @param  insets  the insets for the grid
   * @param  x       the x-coordinate for the current cell
   * @param  y       the y-coordinate for the current cell
   * @param  rgb     the colour of this cell (0 if transparent)
   */
  protected void drawCell(Graphics g, Insets insets, int x, int y, int rgb) {
    int alpha = (rgb >> 24) & 0xff;
    int xOffset = insets.left + x * cellSize;
    int yOffset = insets.top + y * cellSize;
    if (!g.hitClip(xOffset + 1, yOffset + 1, cellSize - 1, cellSize - 1)) {
      return;
    }
    g.setColor(alpha == 0 ? getBackground() : new Color(rgb));
    g.fillRect(xOffset + 1, yOffset + 1, cellSize - 1, cellSize - 1);
    if (alpha == 0) {
      g.setColor(Color.black);
      g.drawLine(xOffset + (cellSize / 2), yOffset + (cellSize / 2),
        xOffset + (cellSize / 2), yOffset + (cellSize / 2));
    }
  }
}