package wood.keith.opentools.vfs.http;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.util.ArrayList;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListModel;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import com.borland.primetime.help.HelpTopic;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.ui.SearchTree;
import com.borland.primetime.vfs.Filesystem;
import com.borland.primetime.vfs.Url;
import com.borland.primetime.vfs.VFS;
import com.borland.primetime.vfs.ui.UrlChooserPage;
import com.borland.primetime.vfs.ui.UrlExplorerIcons;
import com.borland.primetime.vfs.ui.UrlFilter;

/**
 * A Url chooser page for exploring a remote (HTTP) filesystem.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  21 March 2004
 */
public class HttpFSChooserPage extends UrlChooserPage {

  public static final Icon ROOT_ICON =
      new ImageIcon(ClassLoader.getSystemResource(
      "wood/keith/opentools/vfs/http/HttpFilesystemS.gif"));

  private int _mode;
  private Url _rootUrl = null;

  private GridBagLayout _gridBagLayout = new GridBagLayout();
  private JPanel _buttonPanel = new JPanel();
  private FlowLayout _flowLayout = new FlowLayout();
  private JButton _upButton = new JButton();
  private JButton _newButton = new JButton();
  private JButton _deleteButton = new JButton();
  private JScrollPane _dirScroll = new JScrollPane();
  private DefaultTreeModel _dirModel = null;
  private SearchTree _dirTree = new SearchTree();
  private JScrollPane _fileScroll = new JScrollPane();
  private DefaultListModel _fileModel = new DefaultListModel();
  private JList _fileList = new JList(_fileModel);
  private JLabel _nameLabel = new JLabel();
  private JTextField _nameText = new JTextField();
  private JLabel _typeLabel = new JLabel();
  private JComboBox _typeCombo = new JComboBox();

  /**
   * Initialisation.
   *
   * @param  mode  MODE_DIRECTORY or MODE_FILE
   */
  public HttpFSChooserPage(int mode) {
    this();
    _mode = mode;
    initialise();
  }

  /**
   * Initialise the page.
   */
  public HttpFSChooserPage() {
    try {
      jbInit();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * UI setup.
   *
   * @throws  Exception
   */
  private void jbInit() throws Exception {
    this.setLayout(_gridBagLayout);
    _nameLabel.setDisplayedMnemonic('N');
    _nameLabel.setLabelFor(_nameText);
    _nameLabel.setText("File name");
    _nameText.setMinimumSize(new Dimension(4, 21));
    _nameText.setPreferredSize(new Dimension(63, 21));
    _nameText.setText("");
    _typeLabel.setDisplayedMnemonic('T');
    _typeLabel.setLabelFor(_typeCombo);
    _typeLabel.setText("File type");
    _typeCombo.setRenderer(new UrlFilterListRenderer());
    _typeCombo.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent event) {
        _typeCombo_actionPerformed(event);
      }
    });
    _buttonPanel.setLayout(_flowLayout);
    _buttonPanel.setMinimumSize(new Dimension(83, 25));
    _buttonPanel.setPreferredSize(new Dimension(83, 25));
    _flowLayout.setAlignment(FlowLayout.RIGHT);
    _upButton.setBorder(null);
    _upButton.setMaximumSize(new Dimension(21, 21));
    _upButton.setMinimumSize(new Dimension(21, 21));
    _upButton.setPreferredSize(new Dimension(21, 21));
    _upButton.setToolTipText("Move up a folder");
    _upButton.setMargin(new Insets(0, 0, 0, 0));
    _upButton.setText("");
    _upButton.setIcon(UrlExplorerIcons.ICON_UP_FOLDER);
    _upButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent event) {
        _upButton_actionPerformed(event);
      }
    });
    _newButton.setBorder(null);
    _newButton.setMaximumSize(new Dimension(21, 21));
    _newButton.setMinimumSize(new Dimension(21, 21));
    _newButton.setPreferredSize(new Dimension(21, 21));
    _newButton.setToolTipText("Create a new directory");
    _newButton.setMargin(new Insets(0, 0, 0, 0));
    _newButton.setText("");
    _newButton.setIcon(UrlExplorerIcons.ICON_NEW_FOLDER);
    _newButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent event) {
        _newButton_actionPerformed(event);
      }
    });
    _deleteButton.setBorder(null);
    _deleteButton.setMaximumSize(new Dimension(21, 21));
    _deleteButton.setMinimumSize(new Dimension(21, 21));
    _deleteButton.setPreferredSize(new Dimension(21, 21));
    _deleteButton.setToolTipText("Delete directory or file");
    _deleteButton.setMargin(new Insets(0, 0, 0, 0));
    _deleteButton.setText("");
    _deleteButton.setIcon(UrlExplorerIcons.ICON_DELETE);
    _deleteButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent event) {
        _deleteButton_actionPerformed(event);
      }
    });
    _dirTree.setShowsRootHandles(true);
    _dirTree.setRootVisible(true);
    _dirTree.setCellRenderer(new UrlRenderer());
    _dirTree.addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent event) {
        _dirTree_valueChanged(event);
      }
    });
    _dirTree.addMouseListener(new MouseAdapter() {
      public void mouseClicked(MouseEvent event) {
        _dirTree_mouseClicked(event);
      }
    });
    _fileList.setCellRenderer(new UrlListRenderer());
    _fileList.addListSelectionListener(new ListSelectionListener() {
      public void valueChanged(ListSelectionEvent event) {
        _fileList_valueChanged(event);
      }
    });
    _fileList.addMouseListener(new MouseAdapter() {
      public void mouseClicked(MouseEvent event) {
        _fileList_mouseClicked(event);
      }
    });
    _dirScroll.setPreferredSize(new Dimension(76, 76));
    _fileScroll.setPreferredSize(new Dimension(76, 76));
    this.add(_buttonPanel, new GridBagConstraints(0, 0, GridBagConstraints.REMAINDER, 1, 0.0, 0.0
        , GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(0, 0, 0, 0), 0, 0));
    this.add(_dirScroll, new GridBagConstraints(0, 1, 2, 1, 1.0, 1.0
        , GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(2, 4, 2, 2), 0, 0));
    this.add(_fileScroll, new GridBagConstraints(2, 1, 1, 1, 1.0, 1.0
        , GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(2, 2, 2, 2), 0, 0));
    this.add(_nameLabel, new GridBagConstraints(0, 2, 1, 1, 0.0, 0.0
        , GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(2, 4, 2, 2), 0, 0));
    this.add(_nameText, new GridBagConstraints(1, 2, 2, 1, 0.0, 0.0
        , GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(2, 2, 2, 2), 0, 0));
    this.add(_typeLabel, new GridBagConstraints(0, 3, 1, 1, 0.0, 0.0
        , GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(2, 4, 2, 2), 0, 0));
    this.add(_typeCombo, new GridBagConstraints(1, 3, 2, 1, 0.0, 0.0
        , GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(2, 2, 2, 2), 0, 0));
    _dirScroll.getViewport().add(_dirTree, null);
    _fileScroll.getViewport().add(_fileList, null);
    _buttonPanel.add(_upButton, null);
    _buttonPanel.add(_newButton, null);
    _buttonPanel.add(_deleteButton, null);
  }

  /**
   * Fill controls with default data.
   */
  private void initialise() {
    if (_mode == MODE_DIR) {
      // Choose a directory only
      _nameLabel.setDisplayedMnemonic('D');
      _nameLabel.setText("Directory");
      _typeLabel.setVisible(false);
      _typeCombo.setVisible(false);
      _fileScroll.setVisible(false);
    }
    _rootUrl = new Url(HttpFilesystem.PROTOCOL, HttpFilesystem.VFS_HOST,
        HttpFilesystem.VFS_HOST + "/");
    UrlTreeNode rootNode = new UrlTreeNode(_rootUrl);
    _dirModel = new DefaultTreeModel(rootNode);
    _dirTree.setModel(_dirModel);
    _dirTree.setSelectionPath(_dirTree.getPathForRow(0));
  }

  /**
   * Move up one level in the directory hierarchy.
   *
   * @param  event  the triggering event
   */
  private void _upButton_actionPerformed(ActionEvent event) {
    TreePath path = _dirTree.getSelectionPath();
    if (path != null && path.getParentPath() != null) {
      _dirTree.setSelectionPath(path.getParentPath());
    }
  }

  /**
   * Create a new directory beneath the current selected one.
   *
   * @param  event  the triggering event
   */
  private void _newButton_actionPerformed(ActionEvent event) {
    String dirName = JOptionPane.showInputDialog(this, "Directory name",
        HttpFSChooserPageFactory.PAGE_NAME, JOptionPane.QUESTION_MESSAGE);
    if (dirName != null && dirName.length() > 0) {
      Url url = (Url)((UrlTreeNode)_dirTree.getSelectionPath().
          getLastPathComponent()).getUserObject();
      try {
        HttpFilesystem.FILESYSTEM.createDirectory(url.getRelativeUrl(dirName));
        initialise();
      }
      catch (IOException ioe) {
        JOptionPane.showMessageDialog(this, "Error during directory create\n" +
            ioe.getClass().getName() + "\n" + ioe.getMessage(),
            HttpFSChooserPageFactory.PAGE_NAME, JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /**
   * Delete the current file (or directory if no file selected).
   *
   * @param  event  the triggering event
   */
  private void _deleteButton_actionPerformed(ActionEvent event) {
    Url url = (Url)_fileList.getSelectedValue();
    boolean isDirectory = false;
    if (url == null) {
      url = (Url)((UrlTreeNode)_dirTree.getSelectionPath().
          getLastPathComponent()).getUserObject();
      isDirectory = true;
    }
    try {
      if (JOptionPane.showConfirmDialog(this,
          "Delete " + url.getFullName() + "?",
          HttpFSChooserPageFactory.PAGE_NAME, JOptionPane.YES_NO_OPTION) ==
          JOptionPane.YES_OPTION) {
        HttpFilesystem.FILESYSTEM.delete(url);
        if (isDirectory) {
          initialise();
        }
        else {
          updateFileList();
        }
      }
    }
    catch (IOException ioe) {
      JOptionPane.showMessageDialog(this, "Error during deletion\n" +
          ioe.getClass().getName() + "\n" + ioe.getMessage(),
          HttpFSChooserPageFactory.PAGE_NAME, JOptionPane.ERROR_MESSAGE);
    }
  }

  /**
   * Update the UI based on the selection of a directory.
   *
   * @param  event  the triggering event
   */
  private void _dirTree_valueChanged(TreeSelectionEvent event) {
    if (_mode != MODE_DIR) {
      updateFileList();
    }
  }

  /**
   * Double click on a directory to select it and close the dialog.
   * But only when in directory mode.
   *
   * @param  event  the triggering event
   */
  private void _dirTree_mouseClicked(MouseEvent event) {
    if (event.getClickCount() == 2 && _mode == MODE_DIR) {
      okPressed();
    }
  }

  /**
   * Update the UI based on the selection of a file.
   *
   * @param  event  the triggering event
   */
  private void _fileList_valueChanged(ListSelectionEvent event) {
    Url url = (Url)_fileList.getSelectedValue();
    _nameText.setText(
        _fileList.getSelectedIndices().length != 1 ? "" : url.getName());
  }

  /**
   * Double click on a file to select it and close the dialog.
   *
   * @param  event  the triggering event
   */
  private void _fileList_mouseClicked(MouseEvent event) {
    if (event.getClickCount() == 2) {
      okPressed();
    }
  }

  /**
   * Update the UI based on the selection of a type filter.
   *
   * @param  event  the triggering event
   */
  private void _typeCombo_actionPerformed(ActionEvent event) {
    updateFileList();
  }

  /**
   * Repopulate the list of files for the currently selected directory.
   */
  private void updateFileList() {
    _fileModel.removeAllElements();
    if (_dirTree.getSelectionPath() == null) {
      return;
    }
    Url url = (Url)((UrlTreeNode)_dirTree.getSelectionPath().
        getLastPathComponent()).getUserObject();
    Url[] children = HttpFilesystem.FILESYSTEM.getChildren(
        url, null, Filesystem.TYPE_FILE);
    UrlFilter filter = (UrlFilter)_typeCombo.getSelectedItem();
    for (int index = 0; index < children.length; index++) {
      if (filter == null || filter.accept(children[index])) {
        _fileModel.addElement(children[index]);
      }
    }
  }

  /**
   * Respond to the user clicking OK in the host dialog.
   */
  public void okPressed() {
    Url dirUrl = (Url)((UrlTreeNode)_dirTree.getSelectionPath().
        getLastPathComponent()).getUserObject();
    ArrayList urls = new ArrayList();
    if (_mode == MODE_DIR) {
      // Select the current directory
      if (_nameText.getText().length() > 0) {
        urls.add(dirUrl.getRelativeUrl(_nameText.getText()));
      }
      else {
        urls.add(dirUrl);
      }
    }
    else {
      // Or the current file(s)
      if (_nameText.getText().length() > 0) {
        urls.add(dirUrl.getRelativeUrl(_nameText.getText()));
      }
      else {
        Object[] files = _fileList.getSelectedValues();
        for (int index = 0; index < files.length; index++) {
          urls.add((Url)files[index]);
        }
      }
    }
    if (urls.size() > 0) {
      closeChooser((Url[])urls.toArray(new Url[urls.size()]));
    }
  }

  /**
   * Set the selection mode for the page.
   *
   * @param  enabled  true if multiple files may be selected, false if only one
   */
  public void setMultiSelectionEnabled(boolean enabled) {
    _fileList.getSelectionModel().setSelectionMode(enabled ?
        ListSelectionModel.MULTIPLE_INTERVAL_SELECTION :
        ListSelectionModel.SINGLE_SELECTION);
  }

  /**
   * Establish the current Url.
   *
   * @param  url  the current Url
   */
  public void setUrl(Url url) {
    if (!url.getProtocol().equals(HttpFilesystem.PROTOCOL)) {
      return;
    }
    // Set the directory in the tree
    Url dirUrl = url;
    if (!VFS.isDirectory(url)) {
      dirUrl = url.getParent();
    }
    TreePath path = null;
    for (int index = 0; index < _dirTree.getRowCount(); index++) {
      if (((UrlTreeNode)_dirTree.getPathForRow(index).getLastPathComponent()).
          getUserObject().equals(url)) {
        path = _dirTree.getPathForRow(index);
        break;
      }
    }
    if (path != null) {
      _dirTree.setSelectionPath(path);
      // And the file in the list
      if (_mode != MODE_DIR && !VFS.isDirectory(url)) {
        _fileList.setSelectedValue(url, true);
      }
    }
  }

  /**
   * Establish the files to be applied to the Urls.
   *
   * @param  filters  the list of filters
   */
  public void setUrlFilters(UrlFilter[] filters) {
    for (int index = 0; index < filters.length; index++) {
      _typeCombo.addItem(filters[index]);
    }
    _typeCombo.setSelectedIndex(0);
  }

  /**
   * Provide a reference to help for this page.
   *
   * @return  the help reference, or null for none
   */
  public HelpTopic getHelpTopic() {
    return null;
  }

  /**
   * A tree node that encapsulates a Url.
   */
  private class UrlTreeNode extends DefaultMutableTreeNode {

    /**
     * Create a new tree node.
     * Automatically find and add any children.
     *
     * @param  url  the Url that this node wraps
     */
    public UrlTreeNode(Url url) {
      super(url);
      addChildren();
    }

    /**
     * Find and add all the children for this Url.
     */
    public void addChildren() {
      Url[] children = HttpFilesystem.FILESYSTEM.getChildren(
          (Url)getUserObject(), null, Filesystem.TYPE_DIRECTORY);
      for (int index = 0; index < children.length; index++) {
        add(new UrlTreeNode(children[index]));
      }
    }

    /**
     * Provide the file name of the Url as the name of this node.
     * Used for SearchTree for searching.
     *
     * @return  the file name of the Url
     */
    public String toString() {
      return ((Url)getUserObject()).getName();
    }
  }

  /**
   * A tree cell renderer for Urls.
   */
  private class UrlRenderer extends DefaultTreeCellRenderer {

    public Component getTreeCellRendererComponent(JTree tree, Object value,
        boolean selected, boolean expanded, boolean leaf, int row,
        boolean hasFocus) {
      if (!(value instanceof UrlTreeNode)) {
        super.getTreeCellRendererComponent(
            tree, "", selected, expanded, leaf, row, hasFocus);
      }
      else {
        Url url = (Url)((UrlTreeNode)value).getUserObject();
        super.getTreeCellRendererComponent(
            tree, url.getName(), selected, expanded, leaf, row, hasFocus);
        setIcon(url == _rootUrl ? ROOT_ICON : UrlExplorerIcons.ICON_FOLDER);
      }
      return this;
    }
  }

  /**
   * A list cell renderer for Urls.
   */
  private class UrlListRenderer extends DefaultListCellRenderer {

    public Component getListCellRendererComponent(JList list, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
      Url url = (Url)value;
      super.getListCellRendererComponent(
          list, url.getName(), index, isSelected, cellHasFocus);
      FileNode fileNode = Browser.getActiveBrowser().getActiveProject().
          createFileNode(url.getFileExtension());
      setIcon(fileNode.getDisplayIcon());
      return this;
    }
  }

  /**
   * A list cell renderer for UrlFilters.
   */
  private class UrlFilterListRenderer extends DefaultListCellRenderer {

    public Component getListCellRendererComponent(JList list, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
      return super.getListCellRendererComponent(list,
          ((UrlFilter)value).getDescription(), index, isSelected, cellHasFocus);
    }
  }
}
