package wood.keith.opentools.vfs.http;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLDecoder;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.Collection;
import java.util.HashMap;

/**
 * A servlet to handle requests from the HTTP VFS.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  19 November 2001
 * @version  1.1  12 May 2002
 */
public class HttpFSServlet extends HttpServlet {

  /** The property name for the base directory to serve up. */
  public static final String BASE_DIR      = "wood.keith.httpvfs.basedir";
  public static final String PARAM_ACT     = "action";
  public static final String PARAM_MOD     = "modified";
  public static final String PARAM_TYPE    = "type";
  public static final String ACT_CHILDREN  = "children";
  public static final String ACT_DELETE    = "delete";
  public static final String ACT_DIRECTORY = "isDirectory";
  public static final String ACT_EXISTS    = "exists";
  public static final String ACT_GET       = "get";
  public static final String ACT_LASTMOD   = "lastMod";
  public static final String ACT_MKDIR     = "createDirectory";
  public static final String ACT_NEWMOD    = "newMod";
  public static final String ACT_READONLY  = "isReadOnly";
  public static final byte   BACKUP_NONE   = 0;
  public static final byte   BACKUP_SINGLE = 1;

  // From Filesystem
  public static final int TYPE_BOTH = 3;
  public static final int TYPE_FILE = 2;
  public static final int TYPE_DIRECTORY = 1;
  // From Buffer
  public static final long MODIFIED_NEVER = -9223372036854775808L;

  private String baseDirectory;

  /**
   * Set the base directory from configuration parameters.
   *
   * @param  config  the configuration environment for the servlet
   * @throws  ServletException
   */
  public void init(ServletConfig config) throws ServletException {
    super.init(config);
    baseDirectory = config.getInitParameter(BASE_DIR);
    if (baseDirectory == null) {
      baseDirectory = new File(System.getProperty("user.dir")).getAbsolutePath();
    }
  }

  /**
   * Handle an incoming request, based on its "action" value.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  protected void doGet(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    String action = req.getParameter(PARAM_ACT);
    if (action == null || action.equals(ACT_GET)) {
      handleGet(req, resp);
    }
    else if (action.equals(ACT_CHILDREN)) {
      handleChildren(req, resp);
    }
    else if (action.equals(ACT_DELETE)) {
      handleDelete(req, resp);
    }
    else if (action.equals(ACT_DIRECTORY)) {
      handleIsDirectory(req, resp);
    }
    else if (action.equals(ACT_EXISTS)) {
      handleExists(req, resp);
    }
    else if (action.equals(ACT_LASTMOD)) {
      handleLastModified(req, resp);
    }
    else if (action.equals(ACT_MKDIR)) {
      handleCreateDirectory(req, resp);
    }
    else if (action.equals(ACT_NEWMOD)) {
      handleNewModified(req, resp);
    }
    else if (action.equals(ACT_READONLY)) {
      handleIsReadOnly(req, resp);
    }
    else {
      resp.setStatus(resp.SC_BAD_REQUEST);
    }
  }

  /**
   * Handle an incoming request, to save the file.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  protected void doPost(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    handlePut(req, resp);
  }

  /**
   * Extract the relative file name from the request and create the actual file.
   * The first path element is discarded since it is only used by JBuilder.
   *
   * @param  req  the incoming request
   * @return  the file corresponding to the incoming request
   * @throws  IOException
   */
  private File getFile(HttpServletRequest req) throws IOException {
    String filename = URLDecoder.decode(req.getPathInfo());
    int pos = filename.indexOf("/", 1);
    if (pos == -1) {
      filename = "/";
    }
    else {
      filename = filename.substring(pos);
    }
    if (filename.indexOf("/..") != -1) {
      throw new IOException("Cannot step out of the served subdirectory");
    }
    return new File(baseDirectory + filename);
  }

  /**
   * Find the children for the nominated file.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleChildren(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    int baseLen = baseDirectory.length();
    int type = Integer.parseInt(req.getParameter(PARAM_TYPE));
    StringBuffer buffer = new StringBuffer("");
    File[] files = getFile(req).listFiles();
    if (files != null) {
      for (int index = 0; index < files.length; index++) {
        if ((type == TYPE_BOTH) ||
            (files[index].isDirectory() && type == TYPE_DIRECTORY) ||
            (!files[index].isDirectory() && type == TYPE_FILE)) {
          buffer.append(files[index].getAbsolutePath().substring(baseLen).
              replace(File.separatorChar, '/') +
              (files[index].isDirectory() ? "/" : "") + "\n");
        }
      }
    }
    setResponse(resp, buffer.toString());
  }

  /**
   * Create a new directory. Write to the response
   * a boolean value indicating its success or not.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleCreateDirectory(HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    setResponse(resp, String.valueOf(getFile(req).mkdir()));
  }

  /**
   * Delete the specified file. Write to the response a boolean
   * value indicating success or failure.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleDelete(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    setResponse(resp, String.valueOf(getFile(req).delete()));
  }

  /**
   * Check that the specified file exists. Write to the response
   * a boolean value indicating its presence or not.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleExists(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    setResponse(resp, String.valueOf(getFile(req).exists()));
  }

  /**
   * Retrieve the contents of the specified file. Write them to the response.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleGet(HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    File file = getFile(req);
    copyStream(new FileInputStream(file), resp.getOutputStream());
    resp.setContentLength((int)file.length());
  }

  /**
   * Check if the specified file is a directory. Write to the response
   * a boolean value indicating whether or not it is a directory.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleIsDirectory(HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    setResponse(resp, String.valueOf(getFile(req).isDirectory()));
  }

  /**
   * Check if the specified file is read only. Write to the response
   * a boolean value indicating whether or not it is read only.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleIsReadOnly(HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    File file = getFile(req);
    setResponse(resp, String.valueOf(file.exists() && !file.canWrite()));
  }

  /**
   * Find the last modified date/time for the specified file.
   * Write the value to the response.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleLastModified(HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    File file = getFile(req);
    setResponse(resp, String.valueOf(file.exists() ?
        file.lastModified() : MODIFIED_NEVER));
  }

  /**
   * Update the last modified date/time for the specified file. Write to the
   * response a boolean value indicating whether or not it was successful.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   * @throws  ServletException
   */
  private void handleNewModified(HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    try {
      long lastModified = Long.parseLong(req.getParameter(PARAM_MOD));
      setResponse(resp,
          String.valueOf(getFile(req).setLastModified(lastModified)));
    }
    catch (NumberFormatException ex) {
      setResponse(resp, String.valueOf(false));
    }
  }

  /**
   * Update a file with new contents.
   *
   * @param  req   the incoming request
   * @param  resp  the outgoing response
   * @throws  IOException
   */
  private void handlePut(HttpServletRequest req, HttpServletResponse resp)
      throws IOException {
    // The first byte indicates the backup status
    byte backupLevel = (byte)req.getInputStream().read();
    try {
      File file = getFile(req);
      if (backupLevel != BACKUP_NONE && file.exists()) {
        // Copy the existing file to a backup
          file.renameTo(new File(file.getAbsolutePath() + "~"));
      }
      // Save the new content
      FileOutputStream fos = new FileOutputStream(file);
      copyStream(req.getInputStream(), fos);
      fos.close();
      setResponse(resp, Boolean.TRUE.toString());
    }
    catch (Exception ex) {
      setResponse(resp, ex.getMessage());
    }
  }

  /**
   * Prepare the response, setting its length as well.
   *
   * @param  resp    the outgoing response
   * @param  output  the text content of the response
   * @throws  IOException
   */
  private void setResponse(HttpServletResponse resp, String output)
      throws IOException {
    resp.setContentLength(output.length());
    resp.getOutputStream().print(output);
  }

  /**
   * Copy from one stream to another.
   *
   * @param  in   the source stream
   * @param  out  the destination stream
   * @throws  IOException  if a problem during copying
   */
  private void copyStream(InputStream in, OutputStream out) throws IOException {
    byte[] buffer = new byte[4096];
    int count = 0;
    while ((count = in.read(buffer)) != -1) {
      out.write(buffer, 0, count);
    }
  }

  /**
 * Composition object
 * @todo  check the constructor
 */

  List _list = new ArrayList();

  /**
 *  Appends the specified element to the end of this list.
 *  @param o element to be appended to this list.
 *  @return <tt>true</tt> (as per the general contract of Collection.add).
 */

  public boolean add(Object o) {
    return _list.add(o);
  }

  /**
 *  Appends all of the elements in the specified Collection to the end of
 *  this list, in the order that they are returned by the
 *  specified Collection's Iterator.  The behavior of this operation is
 *  undefined if the specified Collection is modified while the operation
 *  is in progress.  (This implies that the behavior of this call is
 *  undefined if the specified Collection is this list, and this
 *  list is nonempty.)
 *  @param c the elements to be inserted into this list.
 *  @return <tt>true</tt> if this list changed as a result of the call.
 *  @throws    IndexOutOfBoundsException if index out of range <tt>(index
 * 		  &lt; 0 || index &gt; size())</tt>.
 *  @throws    NullPointerException if the specified collection is null.
 */

  public boolean addAll(Collection c) {
    return _list.addAll(c);
  }

  /**
 * Composition object
 * @todo  check the constructor
 */

  HashMap _hashMap = new HashMap();

  public void clear() {
    _hashMap.clear();
  }

  public boolean containsKey(Object key) {
    return _hashMap.containsKey(key);
  }
}
