package wood.keith.opentools.vfs.http;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.StringTokenizer;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.editor.EditorPropertyGroup;
import com.borland.primetime.util.RegularExpression;
import com.borland.primetime.vfs.AbstractFilesystem;
import com.borland.primetime.vfs.Buffer;
import com.borland.primetime.vfs.Url;

/**
 * A Virtual File System that operates over the Internet.
 * Url's for this system have the form: httpvfs://&lt;host>/&lt;host>/&lt;file>.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  19 November 2001
 * @version  1.1  11 May 2002
 */
public class HttpFilesystem extends AbstractFilesystem {

  public static final String VERSION = "1.1";
  public static final HttpFilesystem FILESYSTEM = new HttpFilesystem();
  public static final String PROTOCOL = "httpvfs";

  private static final String IS_TRUE = String.valueOf(true);

  /**
   * Register the new file system.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    Url.registerFilesystem(FILESYSTEM);
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded HTTP File System v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au");
    }
  }

  /** The internal protocol used to retrieve files. */
  private static final String VFS_PROTOCOL = "http://";
  /** The default host for the remote retrieval. */
  static String VFS_HOST                   = "localhost:8080";
  /** The virtual path that identifies VFS requests. */
  private static final String VFS_ROOT     = "/httpvfs/vfs/";
  /** The name of the system property to change the remote host. */
  private static final String VFS_PROP     = "wood.keith.httpvfs.host";

  /**
   * Load the VFS remote host name.
   */
  private HttpFilesystem() {
    VFS_HOST = System.getProperty(VFS_PROP, VFS_HOST);
  }

  /**
   * Transform a local Url (httpvfs://&lt;host>/&lt;host>/&lt;file>) into the
   * appropriate remote URL (http://&lt;host>/vfs/&lt;host>/&lt;file>?action=&lt;action>.
   *
   * @param  url     the local Url
   * @param  action  the type of request or null for no action
   * @return  the corresponding remote URL
   */
  private URL makeURL(Url url, String action) {
    try {
      StringBuffer buffer = new StringBuffer(VFS_PROTOCOL).
          append(url.getHost()).append(VFS_ROOT).append(url.getFile());
      if (action != null) {
        buffer.append("?").append(HttpFSServlet.PARAM_ACT).
            append("=").append(action);
      }
      return new URL(buffer.toString());
    }
    catch (MalformedURLException ex) {
      throw new IllegalArgumentException(url.toString());
    }
  }

  /**
   * Retrieve the response from a remote call and transfer it into a string.
   *
   * @param  url     the local Url
   * @param  action  the type of request
   * @return  the contents of the response as a string
   */
  private String getResponse(Url url, String action) {
    StringWriter out = new StringWriter(80);
    try {
      char[] buffer = new char [80];
      InputStream ins = makeURL(url, action).openStream();
      if (ins == null) {
        return "";
      }
      BufferedReader in = new BufferedReader(new InputStreamReader(ins));
      int count = 0;
      while ((count = in.read(buffer)) != -1) {
        out.write(buffer, 0, count);
      }
      out.close();
    }
    catch (IOException ex) {
      ex.printStackTrace();
    }
    return out.toString();
  }

  /**
   * Delete the nominated file.
   *
   * @param  url  the local Url
   * @return  true if the directory was created, or false if not
   * @throws  IOException
   */
  public boolean createDirectory(Url url) throws IOException {
//    System.out.println("createDirectory " + url);
    return getResponse(url, HttpFSServlet.ACT_MKDIR).equals(IS_TRUE);
  }

  /**
   * Delete the nominated file.
   *
   * @param  url  the local Url
   * @throws  IOException
   */
  public void delete(Url url) throws IOException {
    delete(url, false);
  }

  /**
   * Delete the nominated file.
   *
   * @param  url         the local Url
   * @param  makeBackup  true to make a backup before deleting,
   *                     false to just delete
   * @throws  IOException  if the file cannot be deleted
   */
  public void delete(Url url, boolean makeBackup) throws IOException {
//    System.out.println("delete " + url);
    if (!getResponse(url, HttpFSServlet.ACT_DELETE).equals(IS_TRUE)) {
      throw new IOException("Couldn't delete file " + url.getFile());
    }
  }

  /**
   * Check that the nominated file exists.
   *
   * @param  url  the local Url
   * @return  true if this file exists, and false otherwise
   */
  public boolean exists(Url url) {
//    System.out.println("exists " + url);
    return getResponse(url, HttpFSServlet.ACT_EXISTS).equals(IS_TRUE);
  }

  /**
   * Find all the children of the specified directory.
   *
   * @param  url       the local Url
   * @param  patterns  an array of filename patterns to match against,
   *                   may be null
   * @param  type      a value indicating files only, directories only,
   *                   or either (see Filesystem literals)
   * @return  an array of matching Url's that are children of the original file
   */
  public Url[] getChildren(Url url, RegularExpression[] patterns, int type) {
//    System.out.println("getChildren " + url + " " + type);
    // Retrieve the list of children separated by linefeeds, based on type
    StringTokenizer responses = new StringTokenizer(
        getResponse(url, HttpFSServlet.ACT_CHILDREN + "&" +
        HttpFSServlet.PARAM_TYPE + "=" + type), "\n\r");
    ArrayList urls = new ArrayList();
    while (responses.hasMoreTokens()) {
      String filename = responses.nextToken();
      if (filename == null) {
        continue;
      }
      // No patterns always matches as do directories (based on marker)
      boolean isDirectory = (filename.charAt(filename.length() - 1) == '/');
      boolean match = (patterns == null || isDirectory);
      if (!match) {
        // Attempt to match each filename with one of the patterns
        for (int index = 0; index < patterns.length; index++) {
          if (patterns[index].exactMatch(filename)) {
            match = true;
            break;
          }
        }
      }
      if (match) {
        // Add the new Url (removing directory marker)
        Url child = new Url(getProtocol(), url.getHost(),
            url.getHost() + "/" + (isDirectory ?
            filename.substring(0, filename.length() - 1): filename));
        // Save directory flag
        child.setFilesystemObject(new Boolean(isDirectory));
        urls.add(child);
//        System.out.println("adding " + child + " " + child.getFilesystemObject());
      }
    }
    return (Url[])urls.toArray(new Url[urls.size()]);
  }

  /**
   * Retrieve the contents of the specified file.
   *
   * @param  url  the local Url
   * @return  an input stream that contains the file's contents
   * @throws  IOException
   */
  public InputStream getInputStream(Url url) throws IOException {
//    System.out.println("getInputStream " + url);
    return makeURL(url, HttpFSServlet.ACT_GET).openStream();
  }

  /**
   * Find the date/time when the file was last modified.
   *
   * @param  url  the local Url
   * @return  the timestamp of the file, or 0 if an error occurs
   */
  public long getLastModified(Url url) {
//    System.out.println("getLastModified " + url);
    try {
      return Long.parseLong(getResponse(url, HttpFSServlet.ACT_LASTMOD));
    }
    catch (NumberFormatException ex) {
      return Buffer.MODIFIED_NEVER;
    }
  }

  /**
   * Write out the contents of the specified file.
   * Make a single level backup before overwriting.
   *
   * @param  url  the local Url
   * @return  an output stream for writing the file's contents
   * @throws  IOException
   */
  public OutputStream getOutputStream(Url url) throws IOException {
//    System.out.println("getOutputStream " + url);
    return new RemoteOutputStream(url, HttpFSServlet.BACKUP_SINGLE);
  }

  /**
   * Write out the contents of the specified file.
   *
   * @param  url         the local Url
   * @param  makeBackup  true to make a multi-level backup copy of the file
   *                     before overwriting, false for no backup
   * @return  an output stream for writing the file's contents
   * @throws  IOException
   */
  public OutputStream getOutputStream(Url url, boolean makeBackup)
      throws IOException {
//    System.out.println("getOutputStream " + url);
    return new RemoteOutputStream(url, (makeBackup ?
        EditorPropertyGroup.BACKUP_LEVEL.getInteger() :
        HttpFSServlet.BACKUP_NONE));
  }

  /**
   * Return the protocol identifier for this filesystem.
   *
   * @return  the protocol
   */
  public String getProtocol() {
    return PROTOCOL;
  }

  /**
   * Determine if the specified file is a directory.
   * Use the cached flag on the Url if possible.
   *
   * @param  url  the local Url
   * @return  true if this file is a directory, false otherwise
   */
  public boolean isDirectory(Url url) {
//    System.out.println("isDirectory " + url + " " + url.getFilesystemObject());
    if (url.getFilesystemObject() != null) {
      return ((Boolean)url.getFilesystemObject()).booleanValue();
    }
    else {
      return getResponse(url, HttpFSServlet.ACT_DIRECTORY).equals(IS_TRUE);
    }
  }

  /**
   * Determine if the specified file is read only.
   *
   * @param  url  the local Url
   * @return  true if this file is read only, false otherwise
   */
  public boolean isReadOnly(Url url) {
//    System.out.println("isReadOnly " + url);
    return getResponse(url, HttpFSServlet.ACT_READONLY).equals(IS_TRUE);
  }

  /**
   * Update the last modified stamp for the file.
   *
   * @param  url           is the local Url
   * @param  lastModified  is the new timestamp
   * @return  true if successful, false otherwise
   */
  public boolean setLastModified(Url url, long lastModified) {
//    System.out.println("setLastModified " + url + " " + lastModified);
    return getResponse(url, HttpFSServlet.ACT_NEWMOD + "&" +
        HttpFSServlet.PARAM_MOD + "=" + lastModified).equals(IS_TRUE);
  }

  /**
   * An intermediary stream between JBuilder and the remote file system.
   */
  private class RemoteOutputStream extends OutputStream {

    private Url url;
    private ByteArrayOutputStream buffer = new ByteArrayOutputStream();

    /**
     * Create a new output stream for remote saving of a file.
     *
     * @param  url          the local Url
     * @param  backupLevel  one of the HttpFSServlet BACKUP_* values or a number
     *                      indicating the required backup level
     */
    public RemoteOutputStream(Url url, int backupLevel) {
      this.url = url;
      buffer.write(backupLevel);
    }

    /**
     * On closing the stream, the contents are transmitted to the remote
     * file system via a HTTP POST.
     *
     * @throws  IOException  if the remote save failed
     */
    public void close() throws IOException {
      super.close();
      URLConnection conn = makeURL(url, null).openConnection();
      conn.setAllowUserInteraction(false);
      conn.setDoInput(true);
      conn.setDoOutput(true);
      conn.setUseCaches(false);
      OutputStream out = conn.getOutputStream();
      out.write(buffer.toByteArray(), 0, buffer.size());
      out.close();
      InputStream in = conn.getInputStream();
      byte[] respBuffer = new byte[100];
      if (in.read(respBuffer) != HttpFilesystem.IS_TRUE.length()) {
        throw new IOException("Save failed " + new String(respBuffer));
      }
    }

    /**
     * Save the output for later use.
     *
     * @param  b  the code for the character being written
     * @throws  IOException
     */
    public void write(int b) throws IOException {
      buffer.write(b);
    }
  }
}
