package wood.keith.opentools.icontips;

import java.awt.Point;
import javax.swing.tree.TreeNode;

import com.borland.jbuilder.node.idl.Adapter;
import com.borland.jbuilder.node.idl.IDLStructureElement;
import com.borland.primetime.editor.EditorManager;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.node.Node;
import com.borland.primetime.ui.MergeTreeNode;

/**
 * Listener for mouse movements over the structure view tree component.
 * Intended for IDL views only, it displays the entire node in the status bar.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.2  3 December 2003
 */
public class IconTipsIDLListener extends IconTipsListener {

  private static final int MEMBER_TYPE_LOW = 15;
  private static final int MEMBER_TYPE_HIGH = 17;

  /**
   * Attach this listener to the tree in the structure view.
   *
   * @param  browser  the current browser
   * @param  node     the current node
   */
  public IconTipsIDLListener(Browser browser, Node node) {
    super(browser, node);
  }

  /**
   * Retrieve the tool tip text for this node in the tree.
   *
   * @param  treeNode  the current node in the structure tree
   * @param  fileNode  the current file node
   * @return  the tool tip text for this tree node
   */
  protected String getText(TreeNode treeNode, FileNode fileNode) {
    StringBuffer tip = new StringBuffer();
    // And set the basic type
    IDLStructureElement element =
        (IDLStructureElement)((MergeTreeNode)treeNode).getUserObject();
    addType(element.type, tip);
    try {
      // And its location within the file
      Point location = Adapter.getLocation(treeNode);
      // Load the content
      StringBuffer buffer =
          new StringBuffer(new String(fileNode.getBuffer().getContent()));
      for (int index = buffer.length() - 1; index >= 0; index--) {
        // Remove extraneous carriage returns
        if (buffer.charAt(index) == '\r') {
          buffer.deleteCharAt(index);
        }
      }
      String content = buffer.toString();
      // And extract the node's text (offset points to end)
      int offset = EditorManager.getEditor(fileNode).
          calcCaretPosition((int)location.getX(), (int)location.getY());
      if (offset > -1) {
        // Offset may point to start of next element
        int end = 0;
        while (offset + end < content.length() &&
            content.charAt(offset + end) != '{' &&
            content.charAt(offset + end) != '}' &&
            content.charAt(offset + end) != ';' &&
            (element.type < MEMBER_TYPE_LOW ||
            element.type > MEMBER_TYPE_HIGH ||
            content.charAt(offset + end) != ',')) {
          end++;
        }
        tip.append(normalise(
            content.substring(offset, offset + end).trim()));
      }
      else {
        tip.append(element.name);
      }
    }
    catch (Exception ex) {
      tip.append(element.name);
    }
    return tip.toString();
  }

  /**
   * Add in a description of the icon.
   *
   * @param  type  the type of element
   * @param  tip   the buffer to add to
   */
  private void addType(int type, StringBuffer tip) {
    switch (type) {
      case 0:  tip.append(" [errors] "); break;
      case 1:  tip.append(" [error] "); break;
      case 18:
      case 19: tip.append(" [include] "); break;
      case 3:  tip.append(" [module] "); break;
      case 4:  tip.append(" [interface] "); break;
      case 5:  tip.append(" [value] "); break;
      case 6:  tip.append(" [alias] "); break;
      case 7:  tip.append(" [const] "); break;
      case 8:  tip.append(" [struct] "); break;
      case 9:  tip.append(" [exception] "); break;
      case 10: tip.append(" [enum] "); break;
      case 11: tip.append(" [union] "); break;
      case 12: tip.append(" [attribute] "); break;
      case 13: tip.append(" [operation] "); break;
      case 14: tip.append(" [initializer] "); break;
      case 15:
      case 16:
      case 17: tip.append(" [member] "); break;
      case 20: tip.append(" [parent interface] "); break;
      case 21: tip.append(" [parent value] "); break;
      case 22: tip.append(" [parent structure] "); break;
      case 23: tip.append(" [support value] "); break;
      case 2:
      default: tip.append(" [unknown] "); break;
    }
  }
}
