package wood.keith.opentools.icontips;

import java.io.IOException;
import javax.swing.tree.TreeNode;

import com.borland.jbuilder.node.ClassFileNode;
import com.borland.jbuilder.node.JavaFileNode;
import com.borland.jbuilder.node.java.JavaStructureElement;
import com.borland.jbuilder.node.java.JavaStructureNode;
import com.borland.primetime.editor.EditorManager;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.node.Node;

/**
 * Listener for mouse movements over the structure view tree component.
 * Intended for Java views only, it decodes the icons for each node
 * and displays the result in the status bar.
 * Show full signature for structure elements if possible in v1.1.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  1 December 2000
 * @version  1.1  7 May 2001
 * @version  1.2  3 December 2003
 */
public class IconTipsJavaListener extends IconTipsListener {

  /**
   * Attach this listener to the tree in the structure view.
   *
   * @param  browser  the current browser
   * @param  node     the current node
   */
  public IconTipsJavaListener(Browser browser, Node node) {
    super(browser, node);
  }

  /**
   * Retrieve the tool tip text for this node in the tree.
   *
   * @param  treeNode  the current node in the structure tree
   * @param  fileNode  the current file node
   * @return  the tool tip text for this tree node
   */
  protected String getText(TreeNode treeNode, FileNode fileNode) {
    StringBuffer tip = new StringBuffer();
    if (treeNode instanceof JavaStructureNode) {
      // Retrieve the associated user object (should be a JavaStructureElement)
      Object element = ((JavaStructureNode)treeNode).getUserObject();
      if (element instanceof JavaStructureElement) {
        // Decode the icon (from type and subType)
        JavaStructureElement jelement = (JavaStructureElement)element;
        switch (jelement.type) {
          case JavaStructureNode.TYPE_CLASS:
            addVisibility(jelement, "class", tip);
            break;
          case JavaStructureNode.TYPE_CONSTRUCTOR:
            addVisibility(jelement, "constructor", tip);
            break;
          case JavaStructureNode.TYPE_FIELD:
            addVisibility(jelement, "field", tip);
            break;
          case JavaStructureNode.TYPE_IMPLEMENT_INTERFACE:
            tip.append(" [interface] implements ").append(jelement.name);
            break;
          case  JavaStructureNode.TYPE_IMPORT:
            tip.append(" [class] import ").append(jelement.name);
            break;
          case JavaStructureNode.TYPE_INHERIT_CLASS:
            tip.append(" [class] extends ").append(jelement.name);
            break;
          case JavaStructureNode.TYPE_INTERFACE:
            addVisibility(jelement, "interface", tip);
            break;
          case JavaStructureNode.TYPE_METHOD:
            addVisibility(jelement, "method", tip);
            break;
          case JavaStructureNode.TYPE_PACKAGE:
            tip.append(" [class] package ").append(jelement.name);
            break;
        }
        if (tip.length() > 0) {
          tip.delete(0, 1);
        }
      }
    }
    return tip.toString();
  }

  /**
   * Try to extract full signature from source.
   * Otherwise, add visibility keywords for nodes that need them.
   *
   * @param  jelement  the node we are currently over
   * @param  type      the type of node encountered
   * @param  tip       the tip text being constructed
   */
  private void addVisibility(JavaStructureElement jelement,
      String type, StringBuffer tip) {
    tip.append(" [").append(type).append("]");
    try {
      Node node = Browser.getActiveBrowser().getActiveNode();
      if (node instanceof JavaFileNode || node instanceof ClassFileNode) {
        // Extract full signature from source file
        StringBuffer buffer = new StringBuffer(new String(
          node instanceof JavaFileNode ?
          ((JavaFileNode)node).getBuffer().getContent() :
          ((ClassFileNode)node).getStubSource().getBytes()));
        for (int index = buffer.length() - 1; index >= 0; index--) {
          // Remove extraneous carriage returns
          if (buffer.charAt(index) == '\r') {
            buffer.deleteCharAt(index);
          }
        }
        String content = buffer.toString();
        int offset = (jelement.offset != -1 ? jelement.offset :
            EditorManager.getEditor(node).
            calcCaretPosition(jelement.line, jelement.column));
        int end = 0;
        while (offset + end < content.length() &&
            content.charAt(offset + end) != '{' &&
            content.charAt(offset + end) != ';') {
          end++;
        }
        if (end == 0) {
          tip.append(" static initializer");
        }
        else {
          tip.append(" ").append(normalise(
              content.substring(offset, offset + end).trim()));
        }
      }
      else {
        // Revert to decoding of flags in the structure element
        throw new IOException();
      }
    }
    catch (IOException ex) {
      addDefaultVisibility(jelement, tip);
    }
    catch (StringIndexOutOfBoundsException ex) {
      addDefaultVisibility(jelement, tip);
    }
  }

  /**
   * Add visibility keywords for nodes that need them.
   *
   * @param  jelement  the node we are currently over
   * @param  tip       the tip text being constructed
   */
  private void addDefaultVisibility(JavaStructureElement jelement,
      StringBuffer tip) {
    // Decode flags in the structure element
    int visibility =
      jelement.subType & JavaStructureNode.SUBTYPE_VISIBILITY_MASK;
    switch (visibility) {
      case JavaStructureNode.SUBTYPE_PRIVATE:
        tip.append(" private");
        break;
      case JavaStructureNode.SUBTYPE_PROTECTED:
        tip.append(" protected");
        break;
      case JavaStructureNode.SUBTYPE_PUBLIC:
        tip.append(" public");
        break;
      case JavaStructureNode.SUBTYPE_DEFAULT:
        tip.append(" package");
        break;
    }
    if ((jelement.subType & JavaStructureNode.SUBTYPE_STATIC) ==
        JavaStructureNode.SUBTYPE_STATIC) {
      tip.append(" static");
    }
    tip.append(" ").append(jelement.name);
  }
}
