package wood.keith.opentools.icontips;

import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.NodeViewer;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.node.Node;
import com.borland.primetime.util.VetoException;
import com.borland.primetime.viewer.Res;

/**
 * Listener for mouse movements over the structure view tree component.
 * Displays more details for structure nodes in the status bar.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.2  3 December 2003
 */
public abstract class IconTipsListener extends MouseMotionAdapter {

  private static final String SOURCE_TAB = Res.getString(49);

  // The current browser
  protected Browser _browser;
  // The current node
  protected FileNode _node;
  // The tree view of nodes
  protected JTree _tree;

  /**
   * Attach this listener to the tree in the structure view.
   *
   * @param  tree    the tree view of the structure
   * @param  status  the status bar for this browser
   */
  public IconTipsListener(Browser browser, Node node) {
    super();
    _browser = browser;
    _node = (FileNode)node;
    try {
      _tree = getStructureTree(browser, node);
      _tree.addMouseMotionListener(this);
    }
    catch (VetoException ex) {
      _tree = null;
    }
  }

  /**
   * Tidy up - remove ourselves from the tree notifications.
   */
  public void detach() {
    if (_tree != null) {
      _tree.removeMouseMotionListener(this);
    }
  }

  /**
   * When the mouse moves over a node, display its text into the status bar.
   *
   * @param  e  the mouse movement event
   */
  public void mouseMoved(MouseEvent e) {
    String tip = null;
    TreePath path = _tree.getPathForLocation(e.getX(), e.getY());
    if (path != null) {
      // Find the node we are over
      TreeNode treeNode = (TreeNode)path.getLastPathComponent();
      tip = getText(treeNode, _node);
    }
    // And show it's tip - setting it as a hint means
    // it only displays while over the node
    _browser.getStatusView().setHintText(tip);
    _tree.setToolTipText(tip);
  }

  /**
   * Retrieve the tool tip text for this node in the tree.
   *
   * @param  treeNode  the current node in the structure tree
   * @param  fileNode  the current file node
   * @return  the tool tip text for this tree node
   */
  protected abstract String getText(TreeNode treeNode, FileNode fileNode);

  /**
   * Replace consecutive whitespace characters with single spaces.
   *
   * @param  value  the text to be normalised
   * @returns  the normalised text
   */
  protected String normalise(String value) {
    StringBuffer buffer = new StringBuffer(value);
    for (int index = buffer.length() - 1; index > 1; index--) {
      if (Character.isWhitespace(buffer.charAt(index)) &&
          Character.isWhitespace(buffer.charAt(index - 1))) {
        buffer.replace(index - 1, index + 1, " ");
      }
    }
    return buffer.toString();
  }

  /**
   * Locate the tree component within the structure pane.
   *
   * @param  browser  the current browser
   * @return  the structure tree
   */
  protected JTree getStructureTree(Browser browser, Node node)
      throws VetoException {
    NodeViewer[] viewers = browser.getViewers(node);
    for (int index = 0; index < viewers.length; index++) {
      if (viewers[index].getViewerTitle().equals(SOURCE_TAB)) {
        // Get the structure view component
        JComponent comp = viewers[index].getStructureComponent();
        // Get the tree view (assumed to be in a scroll pane in the component)
        return (JTree)((JScrollPane)comp.getComponent(0)).
          getViewport().getComponent(0);
      }
    }
    throw new VetoException(
        "Missing " + SOURCE_TAB + " tab for " + node.getDisplayName());
  }
}
