/*
 * Voyager JBoss OpenTool for JBuilder
 *
 * Copyright (c) 2002 Marcus Redeker, Gedoplan GmbH - Bielefeld, Germany
 *    marcus.redeker@gedoplan.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Cyrille Morvan's excellent 'Kelly JOnAS OpenTool' sources provided
 * much of the inspiration and some of the code for this OpenTool.
 *
*/
package de.gedoplan.opentools.jboss;

import com.borland.jbuilder.enterprise.ejb.*;
import java.util.ArrayList;
import com.borland.primetime.node.Project;
import java.util.Map;
import com.borland.primetime.vfs.Url;
import com.borland.jbuilder.runtime.RunJavaProcessTracker;
import java.awt.Component;
import com.borland.primetime.properties.MultipleNodeProperty;
import com.borland.primetime.properties.GlobalBooleanProperty;
import java.io.File;
import com.borland.primetime.PrimeTime;
import com.borland.jbuilder.info.JBuilderInfo;
import com.borland.jbuilder.ide.SetupManager;
import com.borland.jbuilder.wizard.ejbclient.EjbTestClientState;
import com.borland.jbuilder.jot.*;
import java.util.List;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.vfs.VFS;
import com.borland.jbuilder.node.JBProject;
import com.borland.primetime.runtime.RunConfiguration;
import com.borland.primetime.vfs.ui.UrlChooser;
import com.borland.jbuilder.enterprise.ejb.CommandLineEjbDeployer;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.MessageCategory;
import com.borland.primetime.ide.Message;
import java.io.FilenameFilter;
import com.borland.jbuilder.enterprise.ear.EarGrpUtils;
import com.borland.jbuilder.node.EarGrpFileNode;
import com.borland.jbuilder.node.JarFileNode;
import java.net.URL;
import java.net.URLConnection;
import java.lang.reflect.*;

/**
 * This is the main class of the JBoss OpenTool. It extends JBuilder's abstract
 * AppServer class which is responsible for providing all needed information
 * like classpath, main class, JVM properties, ...
 *
 * @author <a href="mailto:marcus.redeker@gedoplan.de">Marcus Redeker (Gedoplan GmbH, Germany)</a>
 * @version 1.2
 */
public class JBossAppServer24 extends AppServer
{
  final static String JBOSS_DEFAULT_NAME    = JBossResource.get("SERVER_NAME");
  final static String JBOSS_DEFAULT_VERSION24 = JBossResource.get("SERVER_VERSION24");
  protected boolean isJB6 = true;

  public JBossAppServer24()
  {
  }


  public static void initOpenTool(byte major, byte minor)
  {
    // Check OpenTools version number
    if (major != PrimeTime.CURRENT_MAJOR_VERSION)
    {
      return;
    }

    String aCompleteName = JBossResource.COMMERCIAL_NAME + " JBoss OpenTool " + JBossResource.VOYAGER_VERSION;
    if(JBuilderInfo.isEntEnabled())
    {
      JBossAppServer24 temp = new JBossAppServer24();
      AppServerManager.registerAppServer(temp);

      if ( (new Byte(minor)).intValue() < 3)   //this is not JBuilder 6 or higher
      {
        temp.isJB6 = false;
      }

      if( PrimeTime.isVerbose() )
      {
        System.out.println("#####################################################################");
        System.out.println("# " + aCompleteName + " for JBuilder 6 - Major: " + major + " - Minor: " + minor);
        System.out.println("# @author Marcus Redeker <mr@gedoplan.de>");
        System.out.println("# @see " + JBossResource.get("URL"));
        if( temp.isSetup())
        {
          System.out.println("# @home " + JBossPropertyGroup.INSTALL_PATH_24.getValue() );
        }
        else
        {
          System.out.println("# Not setup");
        }
        System.out.println("#####################################################################");
      }
    }
    else
    {
      System.out.println("ERROR : you should have JBuilder Enterprise to run " + aCompleteName );
    }
  }

  public boolean isJB6()
  {
    return isJB6;
  }

  public int getEjbCompliance()
  {
    return super.EJB_2_0;
  }

  public boolean isSetup()
  {
    boolean flag = true;
    String s = JBossPropertyGroup.INSTALL_PATH_24.getValue();
    if(s == null || s.equals(""))
    {
      flag = false;
    }
    if(flag)
    {
      String s1 = JBossPropertyGroup.CONFIGURATION_NAME_24.getValue();
      if(s1 == null || s1.equals(""))
      {
        flag = false;
      }
    }
    return flag;
  }

  public String getDefaultVersion()
  {
    return JBOSS_DEFAULT_VERSION24;
  }

  public String getShortName()
  {
    return AppServer.makeFullName(JBOSS_DEFAULT_NAME, JBOSS_DEFAULT_VERSION24);
  }

  public String customizeParameters(String currentParameters, ArrayList parm2, Project parm3, Map parm4)
  {
    return currentParameters;
  }

  public String getDefaultName()
  {
    return JBOSS_DEFAULT_NAME;
  }

  public boolean supportsPoolNames()
  {
    return false;
  }

  public String getDefaultVmParameters()
  {
    String s = getDefaultHomeDirectory().getFile();
    if(!s.endsWith("/"))
    {
        s = s + "/";
    }
    StringBuffer stringbuffer = new StringBuffer();
    stringbuffer.append("-ms64m -mx64m");
    stringbuffer.append(" -Djavax.xml.parsers.DocumentBuilderFactory=org.apache.crimson.jaxp.DocumentBuilderFactoryImpl");
    stringbuffer.append(" -Djavax.xml.parsers.SAXParserFactory=org.apache.crimson.jaxp.SAXParserFactoryImpl");
    stringbuffer.append(" -Duser.dir=");
    stringbuffer.append(JBossAppServer24.doQuote(getDefaultUserDirectory().getFile()));
    return stringbuffer.toString();
  }

  public String customizeVmParameters(String currentParameters, ArrayList parm2, Project parm3, Map parm4)
  {
    copyJarsToDeployLocation(parm2);
    return currentParameters;
  }

  /**
   * This is the name which was in my JBoss zip-file which came with Tomcat
   * @return an URL
   */
  public Url getDefaultHomeDirectory()
  {
    return new Url("file","/JBoss/jboss/");
  }

  /**
   * JBoss 2.4.x does not fully support EJB 2.0 yet but I put it in already for MessageDrivenBeans;-)
   * @return an int
   */
  public int[] supportedEjbSpecifications()
  {
    return new int[]{super.EJB_1_0, super.EJB_1_1, super.EJB_2_0};
  }

  public String getDefaultParameters()
  {
    return "";
  }

  public boolean shutdown(Project parm1, Map parm2, RunJavaProcessTracker parm3)
  {
    return false;

//    String COMMAND = "/InvokeAction//DefaultDomain%3Atype%3DShutdown/action=shutdown?action=shutdown";
//    try
//    {
//      URL url = new URL("http", "localhost", 8082, COMMAND);
//      url.getContent();
//    }
//    catch (Exception ex)
//    {
//      //ex.printStackTrace();
//      //In all my test a "Connection refused" exception was throws but the server shutdown anyway
//      return true;
//    }
//    return true;
  }

  public boolean supportsCmp1xRelations()
  {
    return false;
  }

  public String getDefaultNecessaryParameters()
  {
    return "";
  }

  public String getDefaultServerName()
  {
     return JBOSS_DEFAULT_NAME;
  }

  public Url[] getDefaultClassPath()
  {
    Url aurl[] = new Url[3];
    String s = getDefaultHomeDirectory().getFile();
    if(!s.endsWith("/"))
        s = s + "/";

    String fileSeperator = System.getProperty("file.separator");
    String javaHome = System.getProperty("java.home");
    String toolsHome = javaHome.substring(0,javaHome.lastIndexOf(fileSeperator));

    aurl[0] = new Url("zip", ((new StringBuffer("[")).append(s).append("bin/run.jar]")).toString());
    aurl[1] = new Url("zip", ((new StringBuffer("[")).append(s).append("lib/crimson.jar]")).toString());
    aurl[2] = new Url("zip", ((new StringBuffer("[")).append(toolsHome).append("/lib/tools.jar]")).toString());

    return aurl;
  }

  public boolean supportsPrimitivePrimaryKeys()
  {
    return true;
  }

  public String getDefaultClassName()
  {
    return "org.jboss.Main";
  }

  public void updateSetup(Component parm1)
  {
    JBossSetup24 jbossSetup = (JBossSetup24)SetupManager.getSetup(getShortName(), AppServerSetup.APPSERVER_SETUP_NAME);
    if(jbossSetup != null)
    {
      jbossSetup.saveSettings(getHomeDirectory().getFile());
    }
  }

  public MultipleNodeProperty getAddProjectClassPathOnDeployNodeProperty()
  {
    return JBossPropertyGroup.ADD_PROJECT_CLASSPATH_ON_DEPLOY_NODE;
  }

  public boolean supportsMinimalJars()
  {
    return true;
  }

  public GlobalBooleanProperty getAddProjectClassPathOnDeployGlobalProperty()
  {
    return JBossPropertyGroup.ADD_PROJECT_CLASSPATH_ON_DEPLOY_GLOBAL;
  }

  public String getDefaultNecessaryVmParameters()
  {
    return "";
  }

  /**
   * Check whether we have a /bin/run.jar in the specified jboss directory.
   * @param path
   * @return a string with the error message or null if everything is ok
   */
  public String isValidSetupDirectory(String path)
  {
    String s1 = null;
    JBossSetup24 jbossSetup = (JBossSetup24)SetupManager.getSetup(getShortName(), AppServerSetup.APPSERVER_SETUP_NAME);
    if(jbossSetup != null)
    {
      s1 = jbossSetup.validateInstallDirectory(path);
    }
    return s1;
  }

  public Url getDefaultUserDirectory()
  {
      return new Url("file", getDefaultHomeDirectory().getFile()+"/bin/");
  }

  public static String doQuote(String s)
  {
      String s1 = s;
      if(s.indexOf(' ') > -1)
      {
          s1 = (new StringBuffer("\"").append(s).append("\"")).toString();
      }
      return s1;
  }

  public Url[] getClientLibraryClassPath(String s)
  {
    if(s == null || s.equals(""))
    {
      s = getHomeDirectory().getFile();
    }

    File clientdir = new File(s, "client");
    String[] filenames = clientdir.list(new FilenameFilter()
      {
        public boolean accept(File file, String s)
        {
          return s.endsWith(".jar") || s.endsWith(".zip");
        }
      });

    Url urls[] = new Url[filenames.length];

    for (int i = 0; i < filenames.length; i++)
    {
      urls[i] = new Url("zip", "[" + clientdir.toString() + "/" + filenames[i] + "]");
    }

    return urls;
  }

  public String createClientGetInitialContext(EjbTestClientState clientState, JotClassSource classSource)
  {
    StringBuffer code = new StringBuffer();
    code.append("{\n");
    code.append("    java.util.Hashtable JNDIParm = new java.util.Hashtable();\n");
    code.append("    JNDIParm.put(Context.PROVIDER_URL, \"localhost\");\n");
    code.append("    JNDIParm.put(Context.INITIAL_CONTEXT_FACTORY, \"org.jnp.interfaces.NamingContextFactory\");\n");
    code.append("    return new InitialContext(JNDIParm);\n");
    code.append("  }\n");

    JotMethodSource newMethod = classSource.addMethod(null, false, "javax.naming.Context", "getJBossInitialContext");
    newMethod.addThrowSpecifier(null, false, "NamingException");
    newMethod.setModifiers(Modifier.PRIVATE);
    newMethod.getCodeBlock().setText(code.toString());

    return "getJBossInitialContext";
  }

  /**
   * I just use this as an entry point to get the event when a user selected the deploy options with
   * a right-click on a deployable node (-jar, -war, -ear). I then trigger my own deployment mechanismn
   * since JBoss does not have a java tool like Weblogic to deploy, undeploy and list applications. I will
   * just copy the files to the JBoss deploy folder, list the folder contents or delete files out of that
   * folder.
   */
  public RunConfiguration getDeployRunConfiguration(ArrayList nodesToDeploy, int deployOptions, JBProject jbproject)
  {
    JBossEJBDeployer24 jbossDeployer = (JBossEJBDeployer24)getEjbDeployer();
    if(jbossDeployer == null)
    {
      return null;
    }

    switch (deployOptions)
    {
      case 1: jbossDeployer.deploy(nodesToDeploy, jbproject);
        break;
      case 2: jbossDeployer.redeploy(nodesToDeploy, jbproject);
        break;
      case 4: jbossDeployer.undeploy(nodesToDeploy, jbproject);
        break;
      case 8: jbossDeployer.listDeployments(nodesToDeploy, jbproject);
        break;
    }

    return null;
  }

  private void copyJarsToDeployLocation(ArrayList arraylist)
  {
    String homeDir = getHomeDirectory().getFile();
    if(!homeDir.endsWith("/"))
    {
      homeDir = homeDir + "/";
    }
    String deployDir = homeDir + "deploy/";
    for(int i = 0; i < arraylist.size(); i++)
    {
      Url url = new Url("file", (String)arraylist.get(i));
      Url url1 = new Url("file", deployDir + url.getName());
      try
      {
        VFS.copy(url, url1);
      }
      catch(Exception exception)
      {
        exception.printStackTrace();
      }
    }
  }

  public void applyDeployUrlFilters(UrlChooser urlchooser)
  {
    super.applyDeployUrlFilters(urlchooser);
    applyEarOrWarDeployUrlFilters(urlchooser);
  }

  public List getAllDeployableNodes(Project project)
  {
    List list = super.getAllDeployableNodes(project);
    if(list == null)
      return null;
    else
      return getEarOrWarAllDeployableNodes(project, list);
  }

  public List getAllDeployableNodeNames(Project project, boolean flag)
  {
    List list = super.getAllDeployableNodeNames(project, flag);
    return getEarOrWarAllDeployableNodeNames(project, flag, list);
  }

  public FileNode getDeployableNode(Node node)
  {
    FileNode filenode = super.getDeployableNode(node);
    if(filenode == null)
    {
      filenode = getEarOrWarDeployableNode(node);
    }
    return filenode;
  }

  public boolean isNodeOrCompanionNodeDeployable(Node node)
  {
    return super.isNodeOrCompanionNodeDeployable(node) || isEarOrWarNodeOrCompanionNodeDeployable(node);
  }

  public Node getCompanionNode(Node node)
  {
    Node node1 = super.getCompanionNode(node);
    if(node1 == null)
    {
      node1 = getEarOrWarCompanionNode(node);
    }
    return node1;
  }
}