/*
 * Voyager JBoss OpenTool for JBuilder
 *
 * Copyright (c) 2002 Marcus Redeker, Gedoplan GmbH - Bielefeld, Germany
 *    marcus.redeker@gedoplan.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Cyrille Morvan's excellent 'Kelly JOnAS OpenTool' sources provided
 * much of the inspiration and some of the code for this OpenTool.
 *
*/
package de.gedoplan.opentools.jboss;

import com.borland.jbuilder.build.*;
import com.borland.jbuilder.enterprise.ejb.*;
import com.borland.jbuilder.node.*;
import com.borland.primetime.ide.*;
import java.util.*;
import java.io.ByteArrayOutputStream;
import com.borland.enterprise.xml.api.IEjbJar;
import com.borland.enterprise.xml.api.IEnterpriseBean;
import com.borland.enterprise.xml.api.IAccessibleBean;
import com.borland.enterprise.xml.api.ISessionBean;
import com.borland.enterprise.xml.api.IEntityBean;
import com.borland.enterprise.xml.api.IMessageDrivenBean;
import com.borland.enterprise.xml.api.IEjbRef;
import com.borland.enterprise.xml.api.IProperty;
import com.borland.enterprise.xml.api.enum.EnumPersistenceType;
import com.borland.enterprise.xml.api.IResourceRef;
import com.borland.enterprise.xml.api.IResourceEnvRef;
import java.text.DateFormat;
import com.borland.primetime.util.AssertionException;
import com.borland.primetime.util.Strings;
import com.borland.enterprise.xml.api.ICmpField;
import com.borland.enterprise.xml.api.ICmpFieldMap;
import com.borland.enterprise.xml.api.IFinder;
import com.borland.enterprise.xml.api.enum.*;
import javax.swing.*;
import com.borland.enterprise.xml.api.*;


/**
 * This class is used by the targeting class to create the JBoss deployment
 * descriptos. Right now it is able to create the jboss.xml and the jaws.xml.
 * Some extra properties for CMP entity beans are displayed in the DD wizard
 * on the JBoss properties tab.
 *
 * @author <a href="mailto:marcus.redeker@gedoplan.de">Marcus Redeker (Gedoplan GmbH, Germany)</a>
 * @author <a href="mailto:m0nstermind@users.sourceforge.net">Oleg Anasatsyev</a>
 * @author <a href="mailto:fred.sauer@allen-sauer.com">Frederik Sauer</a>
 * @version 2.0
 */
public class JBossDescriptorConversion24 extends AbstractDescriptorConversion
{
  protected static MessageCategory deploymentDescriptorCategory;

  protected static String[] generalProperties = new String[]{"security-proxy"};

  protected static String[] mdbProperties = new String[]{"mdb-user", "mdb-passwd", "mdb-subscriber-id"};

  protected static String[] cmpProperties24 = new String[]{"datasource", "type-mapping", "create-table", "remove-table",
                                                  "tuned-updates", "read-only", "time-out", "select-for-update"
                                                  };

  static
  {
    deploymentDescriptorCategory = new MessageCategory(JBossResource.get("DEPLOYMENT_DESCRIPTOR"));
    deploymentDescriptorCategory.setIcon(new ImageIcon(JBossDescriptorConversion24.class.getResource("/de/gedoplan/opentools/jboss/images/jboss_logo.gif")));
  }

  public JBossDescriptorConversion24(DeploymentDescriptor adeploymentdescriptor[], EJBGRPFileNode ejbgrpfilenode, BuildReport buildreport)
  {
    super(adeploymentdescriptor, ejbgrpfilenode, buildreport);
  }

  protected DeploymentDescriptor toDescriptor(XMLNode xmlnode, String doctype, String fileName) throws Exception
  {
    ByteArrayOutputStream bytearrayoutputstream = new ByteArrayOutputStream();
    try
    {
        XMLPrinter xmlprinter = new XMLPrinter(bytearrayoutputstream, "UTF-8");
        xmlprinter.println(doctype);
        xmlprinter.print(xmlnode);
    }
    finally
    {
        bytearrayoutputstream.close();
    }
    DeploymentDescriptor deploymentdescriptor = new DeploymentDescriptor(fileName);
    deploymentdescriptor.setBytes(bytearrayoutputstream.toByteArray());
    deploymentdescriptor.setTimestamp(System.currentTimeMillis());
    return deploymentdescriptor;
  }

  public void verifyDeploymentDescriptors(MessageView messageview, MessageCategory messagecategory) throws Exception
  {
    DateFormat dateformat = DateFormat.getDateTimeInstance(2, 1);
    try
    {
        super.report = new BuildReport();
        messageview.addMessage(messagecategory, "");
        String s = dateformat.format(new Date());
        s = Strings.format("Maybe error??", s);
        messageview.addMessage(messagecategory, s);
        convert(false, true);
        BuildMessage abuildmessage[] = super.report.getAllMessages();
        if(abuildmessage != null)
        {
            for(int k = 0; k < abuildmessage.length; k++)
            {
                if(abuildmessage[k].isError())
                {
                    s = Strings.format("Error", abuildmessage[k].getMessage(), String.valueOf(abuildmessage[k].getLineNumber()));
                    continue;
                }
                if(abuildmessage[k].isWarning())
                {
                    s = Strings.format("Warning", abuildmessage[k].getMessage(), String.valueOf(abuildmessage[k].getLineNumber()));
                    messageview.addMessage(messagecategory, s);
                }
            }

        }
        s = Strings.format("no error", String.valueOf(0), String.valueOf(0));
        messageview.addMessage(messagecategory, s);
    }
    catch(Exception exception)
    {
        throw new AssertionException(exception.getMessage());
    }
  }

/**
 * This method is triggered by the inherited convert method which is called from the method updateDeploymentDescriptors()
 * from the JBossTargeting class. I do not have a property page to create different container configurations for a EJBGRP
 * yet so the per bean property 'cofiguration-name' is not yet supported either.
 *
 * @param ddList an ArrayList to which the generated deployment descriptors are added
 * @throws Exception if an exception occurs somewhere during creation of the xml files
 */
  public void generate(ArrayList ddList) throws java.lang.Exception
  {
    MessageView msgView = getMsgView();
    if (msgView != null)
    {
      msgView.removeTab(deploymentDescriptorCategory);
      //msgView.clearMessages(deploymentDescriptorCategory);
    }

    XMLNode rootNode = new XMLNode(null, "jboss");

    // Write security-domain out of authorization domain defined in ejbgrp DD Editor
    String auth_domain=super.iEjbJar.getAuthorizationDomain();
    if (auth_domain!=null && auth_domain.length()>0)
    {
      // there is auth domain in DD editor - need to write it to jboss.xml
      if (!auth_domain.startsWith("java:/"))
      {
        auth_domain="java:/jaas"+ ( auth_domain.startsWith("/") ? auth_domain : "/"+auth_domain ); // make it conformant to EJB spec
      }
      XMLNode sec_domain_node=new XMLNode(rootNode,"security-domain");
      sec_domain_node.setText(auth_domain);
    }

    XMLNode enterpriseBeansNode = new XMLNode(rootNode, "enterprise-beans");
    Collection collection = super.iEjbJar.getEjbList();
    ArrayList cmpEntityBeans = new ArrayList();
    String datasource = null;
    String datasourceMapping = null;
    String typeMapping = null;
    boolean datasourceUnique = true;
    boolean datasourceMappingUnique = true;

    for (Iterator i = collection.iterator(); i.hasNext(); )
    {
      IEnterpriseBean iEjb = (IEnterpriseBean)i.next();
      XMLNode beanNode = null;
      String securityProxy = null;
      for (Iterator j = iEjb.getPropertyList().iterator(); j.hasNext(); )
      {
        IProperty item = (IProperty)j.next();
        String propName = item.getPropName();
        String propValue = item.getPropValue();

        if (propName.equals("security-proxy"))
        {
          securityProxy = propValue;
        }
        else if (propName.equals("datasource"))
        {
          if (propValue != null && datasourceUnique)
          {
            if (datasource == null)
            {
              datasource = propValue;
            }
            else if (!datasource.equals(propValue))
            {
              datasourceUnique = false;
            }
          }
        }
        else if (propName.equals("datasource-mapping"))
        {
          if (propValue != null && datasourceMappingUnique)
          {
            if (datasourceMapping == null)
            {
              datasourceMapping = propValue;
            }
            else if (!datasourceMapping.equals(propValue))
            {
              datasourceMappingUnique = false;
            }
          }
        }
        else if (propName.equals("type-mapping"))
        {
          typeMapping = propValue;
        }
      }

      if (iEjb instanceof ISessionBean)
      {
        beanNode = new XMLNode(enterpriseBeansNode, "session");
        generateJNDINames(beanNode, iEjb);
        generateSecurityProxy(beanNode, securityProxy);
        generateEJBReferences(beanNode, iEjb);
        generateEJBLocalReferences(beanNode, iEjb);
        generateResourceReferences(beanNode, iEjb);
        generateResourceEnvReferences(beanNode, iEjb);
      }
      else if (iEjb instanceof IEntityBean)
      {
        beanNode = new XMLNode(enterpriseBeansNode, "entity");
        generateJNDINames(beanNode, iEjb);
        generateSecurityProxy(beanNode, securityProxy);
        generateEJBReferences(beanNode, iEjb);
        generateEJBLocalReferences(beanNode, iEjb);
        generateResourceReferences(beanNode, iEjb);
        generateResourceEnvReferences(beanNode, iEjb);

        // Collect all the CMP entityBeans to create the jaws.xml later
        IEntityBean ientitybean = (IEntityBean)iEjb;
        if(ientitybean.getPersistenceTypeEnum() == EnumPersistenceType.CONTAINER)
        {
          cmpEntityBeans.add(ientitybean);
        }
      }
      else if (iEjb instanceof IMessageDrivenBean)
      {
        IMessageDrivenBean imessagebean = (IMessageDrivenBean) iEjb;
        beanNode = new XMLNode(enterpriseBeansNode, "message-driven");
        generateJNDINames(beanNode, iEjb);

        String destJndiName = imessagebean.getMessageDrivenDestinationName();
        if(destJndiName != null && destJndiName.length() > 0)
        {
          new XMLNode(beanNode, "destination-jndi-name", destJndiName.trim());
        }

        if ( imessagebean.getSubscriptionDurabilityEnum() == EnumSubscriptionDurability.DURABLE)
        {
          Map props = new HashMap();
          for (Iterator l = imessagebean.getPropertyList().iterator(); l.hasNext(); )
          {
            IProperty item = (IProperty)l.next();
            props.put(item.getPropName(), item.getPropValue());
          }
          for (int m = 0; m < mdbProperties.length; m++)
          {
            if (props.containsKey(mdbProperties[m]))
              new XMLNode(beanNode, mdbProperties[m], (String)props.get(mdbProperties[m]));
          }
        }

        generateSecurityProxy(beanNode, securityProxy);
        generateEJBReferences(beanNode, iEjb);
        generateEJBLocalReferences(beanNode, iEjb);
        generateResourceReferences(beanNode, iEjb);
        generateResourceEnvReferences(beanNode, iEjb);
      }

    }

    String docType = getJBossXMLDoctype();
    DeploymentDescriptor dd = toDescriptor(rootNode, docType, "jboss.xml");
    ddList.add(dd);

    if (!cmpEntityBeans.isEmpty())
    {
      if (!datasourceUnique) datasource = null;
      if (!datasourceMappingUnique) datasourceMapping = null;
//        generateJawsDeploymentDescriptor(cmpEntityBeans, ddList, datasource, typeMapping);
      generateJbossCmpDeploymentDescriptor(cmpEntityBeans, ddList, datasource, typeMapping, datasourceMapping, msgView);
    }
  }

  private void generateEJBReferences(XMLNode beanNode, IEnterpriseBean iEjb)
  {
    Collection ejbRefCollection = iEjb.getEjbRefList();
    if (ejbRefCollection != null && !ejbRefCollection.isEmpty())
    {
      IEjbRef iejbref;
      XMLNode xmlnode;
      String refJndiName;
      for (Iterator i = ejbRefCollection.iterator(); i.hasNext(); )
      {
        iejbref = (IEjbRef)i.next();
        xmlnode = new XMLNode(beanNode, "ejb-ref");
        new XMLNode(xmlnode, "ejb-ref-name", iejbref.getEjbRefName());
        refJndiName = iejbref.getJndiName();
        if(refJndiName == null || refJndiName.length() == 0)
        {
          refJndiName = iejbref.findJndiNameOfLink();
        }
        if(refJndiName == null || refJndiName.length() == 0)
        {
          addWarningMessage(JBossResource.get("MISSING_EJB_REF_JNDI_NAME") + iejbref.getEjbRefName());
        }
        else
        {
          new XMLNode(xmlnode, "jndi-name", refJndiName);
        }
      }
    }
  }

  private void generateEJBLocalReferences(XMLNode beanNode, IEnterpriseBean iEjb)
  {
    Collection ejbLocalRefCollection = iEjb.getEjbLocalRefList();
    if (ejbLocalRefCollection != null && !ejbLocalRefCollection.isEmpty())
    {
      IEjbLocalRef iejbLocalRef;
      XMLNode xmlnode;
      String refJndiName;
      for (Iterator i = ejbLocalRefCollection.iterator(); i.hasNext(); )
      {
        iejbLocalRef = (IEjbLocalRef)i.next();

        xmlnode = new XMLNode(beanNode, "ejb-local-ref");
        new XMLNode(xmlnode, "ejb-ref-name", iejbLocalRef.getEjbRefName());
        refJndiName = iejbLocalRef.getJndiName();
        if(refJndiName == null || refJndiName.length() == 0)
        {
          refJndiName = iejbLocalRef.findJndiNameOfLink();
        }
        if(refJndiName == null || refJndiName.length() == 0)
        {
          addWarningMessage(JBossResource.get("MISSING_EJB_LOCAL_REF_JNDI_NAME") + iejbLocalRef.getEjbRefName());
        }
        else
        {
          new XMLNode(xmlnode, "local-jndi-name", refJndiName);
        }
      }
    }
  }

  private void generateResourceReferences(XMLNode beanNode, IEnterpriseBean iEjb)
  {
    Collection resRefCollection = iEjb.getResourceRefList();
    if (resRefCollection != null && !resRefCollection.isEmpty())
    {
      IResourceRef iresref;
      XMLNode xmlnode;
      String refJndiName;
      for (Iterator i = resRefCollection.iterator(); i.hasNext(); )
      {
        // Eventually we have to check want kind of resource: iresref.getResType()
        iresref = (IResourceRef)i.next();
        xmlnode = new XMLNode(beanNode, "resource-ref");
        new XMLNode(xmlnode, "res-ref-name", iresref.getResRefName());
        refJndiName = iresref.getJndiName();
        if(refJndiName == null || refJndiName.length() == 0)
        {
          addWarningMessage(JBossResource.get("MISSING_RES_REF_JNDI_NAME") + iresref.getResRefName());
        }
        else
        {
          int idx = refJndiName.indexOf("java:");
          if( idx !=-1 )
          {
            new XMLNode(xmlnode, "jndi-name", refJndiName );
          }
          else
          {
            new XMLNode(xmlnode, "jndi-name", refJndiName.startsWith("/") ? "java:" + refJndiName : "java:/" + refJndiName );
          }
        }
      }
    }
  }


  private void generateResourceEnvReferences(XMLNode beanNode, IEnterpriseBean iEjb)
  {
    Collection resEnvRefCollection = iEjb.getResourceEnvRefList();
    if (resEnvRefCollection != null && !resEnvRefCollection.isEmpty())
    {
      IResourceEnvRef iresenvref;
      XMLNode xmlnode;
      String refJndiName;
      for (Iterator i = resEnvRefCollection.iterator(); i.hasNext(); )
      {
        iresenvref = (IResourceEnvRef)i.next();
        xmlnode = new XMLNode(beanNode, "resource-env-ref");
        new XMLNode(xmlnode, "resource-env-ref-name", iresenvref.getResourceEnvRefName());
        refJndiName = iresenvref.getJndiName();
        if(refJndiName == null || refJndiName.length() == 0)
        {
          addWarningMessage(JBossResource.get("MISSING_RES_ENV_REF_JNDI_NAME") + iresenvref.getResourceEnvRefName());
        }
        else
        {
          new XMLNode(xmlnode, "jndi-name", refJndiName);
        }
      }
    }
  }


  private void generateJNDINames(XMLNode beanNode, IEnterpriseBean iEjb)
  {
    String ejbName = iEjb.getEjbName();
    if(ejbName != null && ejbName.length() > 0)
    {
      new XMLNode(beanNode, "ejb-name", ejbName);
    }

    if (iEjb instanceof IAccessibleBean)
    {
      IAccessibleBean iAccessibleBean = (IAccessibleBean)iEjb;
      String jndiHomeName = iAccessibleBean.getJndiHomeName();
      if(jndiHomeName != null && jndiHomeName.length() > 0)
      {
        new XMLNode(beanNode, "jndi-name", jndiHomeName);
      }

      String jndiLocalHomeName = iAccessibleBean.getJndiLocalHomeName();
      if(jndiLocalHomeName != null && jndiLocalHomeName.length() > 0)
      {
        new XMLNode(beanNode, "local-jndi-name", jndiLocalHomeName);
      }
    }
  }

  private void generateSecurityProxy(XMLNode beanNode, String securityProxy)
  {
    if (securityProxy != null && securityProxy.trim().length() > 0)
    {
      new XMLNode(beanNode, "security-proxy", securityProxy.trim());
    }
  }

  protected void generateJbossCmpDeploymentDescriptor(ArrayList cmpEntityBeans, ArrayList ddList, String datasource, String typeMapping, String datasourceMapping, MessageView msgview) throws Exception
  {
    XMLNode rootNode = new XMLNode(null, "jaws");
    new XMLNode(rootNode, "datasource", datasource);
    new XMLNode(rootNode, "type-mapping", typeMapping);
    XMLNode enterpriseBeansNode = new XMLNode(rootNode, "enterprise-beans");
    XMLNode beanNode = null;
    for (Iterator i = cmpEntityBeans.iterator(); i.hasNext(); )
    {
      IEntityBean entityBean = (IEntityBean)i.next();
      Map props = new HashMap();
      for (Iterator l = entityBean.getPropertyList().iterator(); l.hasNext(); )
      {
        IProperty item = (IProperty)l.next();
        props.put(item.getPropName(), item.getPropValue());
      }

      beanNode = new XMLNode(enterpriseBeansNode, "entity");
      new XMLNode(beanNode, "ejb-name", entityBean.getEjbName());

      Collection cmpFieldList = entityBean.getCmpFieldList();
      if (cmpFieldList != null)
      {
        XMLNode cmpFieldNode = null;
        for (Iterator j = cmpFieldList.iterator(); j.hasNext(); )
        {
          ICmpField cmpField = (ICmpField)j.next();
          cmpFieldNode = new XMLNode(beanNode, "cmp-field");
          new XMLNode(cmpFieldNode, "field-name", cmpField.getFieldName());
          new XMLNode(cmpFieldNode, "column-name", stripQuotes(cmpField.getColumnName()));
        }
      }

      Collection finderList = entityBean.getFinderList();
      if (finderList != null)
      {
        XMLNode finderNode = null;
        for (Iterator k = finderList.iterator(); k.hasNext(); )
        {
          IFinder finder = (IFinder)k.next();
          if(finder.getMethodSignature().indexOf("findByPrimaryKey(") == -1)
          {
            finderNode = new XMLNode(beanNode, "finder");
            String methodName = finder.getMethodSignature();
            String whereClause = finder.getWhereClause();
            if(methodName == null)
            {
              new XMLNode(finderNode, "name", "");
              new XMLNode(finderNode, "query", "");
              new XMLNode(finderNode, "order", "");
            }
            else
            {
              ArrayList arraylist = new ArrayList();
              int bracketPos = methodName.indexOf('(');
              String newMethodName = bracketPos <= -1 ? methodName : methodName.substring(0, bracketPos);
              new XMLNode(finderNode, "name", newMethodName);
              new XMLNode(finderNode, "query", whereClause);
              new XMLNode(finderNode, "order", "");
            }
          }
        }
      }

      if (props.containsKey("read-only"))
        new XMLNode(beanNode, "read-only", (String)props.get("read-only"));

      new XMLNode(beanNode, "table-name", stripQuotes(entityBean.getTableName()));

      if (props.containsKey("tuned-updates"))
        new XMLNode(beanNode, "tuned-updates", (String)props.get("tuned-updates"));

      if (props.containsKey("create-table"))
        new XMLNode(beanNode, "create-table", (String)props.get("create-table"));

      if (props.containsKey("remove-table"))
        new XMLNode(beanNode, "remove-table", (String)props.get("remove-table"));

      if (props.containsKey("select-for-update"))
        new XMLNode(beanNode, "select-for-update", (String)props.get("select-for-update"));

      if (props.containsKey("time-out"))
        new XMLNode(beanNode, "time-out", (String)props.get("time-out"));
    }

    String docType = getCMPXMLDoctype();
    DeploymentDescriptor dd = toDescriptor(rootNode, docType, "jaws.xml");
    ddList.add(dd);
  }

  protected String getCMPXMLDoctype()
  {
    String docType = "<!DOCTYPE jaws PUBLIC '-//JBoss//DTD JAWS 2.4//EN' 'http://www.jboss.org/j2ee/dtd/jaws_2_4.dtd'>";
    return docType;
  }

  protected String getJBossXMLDoctype()
  {
    String docType = "<!DOCTYPE jboss PUBLIC '-//JBoss//DTD JBOSS 2.4//EN' 'http://www.jboss.org/j2ee/dtd/jboss_2_4.dtd'>";
    return docType;
  }

  protected String stripQuotes(String s)
  {
    if(s != null && s.length() > 2 && s.startsWith("\"") && s.endsWith("\""))
        s = s.substring(1, s.length() - 1);
    return s;
  }

  public static EjbPropertyElement[] getPropertyNamesToSurface(EjbReference ejbReference)
  {
    if (ejbReference == null)
    {
      return new EjbPropertyElement[0];
    }

    IEnterpriseBean bean = ejbReference.getBean();
    ArrayList arraylist = new ArrayList();

    //general properties
    for (int i = 0; i < generalProperties.length; i++)
    {
      arraylist.add(new EjbPropertyElement(generalProperties[i]));
    }

    if ((bean instanceof IMessageDrivenBean) && (((IMessageDrivenBean)bean).getSubscriptionDurabilityEnum() == EnumSubscriptionDurability.DURABLE) )
    {
      // mdb properties
      for (int i = 0; i < mdbProperties.length; i++)
      {
        arraylist.add(new EjbPropertyElement(mdbProperties[i]));
      }
    }

    if ((bean instanceof IEntityBean) && (((IEntityBean)bean).getPersistenceTypeEnum() == EnumPersistenceType.CONTAINER) )
    {
      // cmp properties
      for (int i = 0; i < cmpProperties24.length; i++)
      {
        arraylist.add(new EjbPropertyElement(cmpProperties24[i]));
      }
    }

    return (EjbPropertyElement[])arraylist.toArray(new EjbPropertyElement[arraylist.size()]);
  }


  protected MessageView getMsgView()
  {
    if (Browser.getActiveBrowser() != null)
    {
      return Browser.getActiveBrowser().getMessageView();
    }

    return null;
  }

  protected StatusView getStatusView()
  {
    return Browser.getActiveBrowser().getStatusView();
  }
}