package wood.keith.opentools.javadoc;

import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;

import com.borland.jbuilder.node.JavaFileNode;
import com.borland.jbuilder.node.JBProject;
import com.borland.jbuilder.node.PackageNode;
import com.borland.primetime.Command;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.Project;
import com.borland.primetime.vfs.InvalidUrlException;
import com.borland.primetime.vfs.Url;

/**
 * Example of a command line processor OpenTool.
 * It runs javadoc for a nominated project.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 November 2001
 */
public class JavadocCommand implements Command {

  private static final String VERSION = "1.0";
  private static final String LEADER  = "Javadoc Command: ";
  private static final String DESCR   =
      "Run javadoc on the nominated project";
  private static final String USAGE   =
      "  Runs the javadoc utility on the nominated project.\n" +
      "  If you specify options, replace initial hyphens (-) with " +
      "underscores (_),\n" +
      "  and double quotes (\") with single ones (').\n\n" +
      "    -javadoc [javadoc options]... project.jpx\n\n" +
      "  For example:\n\n" +
      "    -javadoc _header 'My new project' project.jpx";

  /**
   * Register the new command with JBuilder.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    PrimeTime.registerCommand("javadoc", new JavadocCommand());
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded javadoc command v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  public JavadocCommand() {
  }

  /**
   * Return the text for the -help command.
   *
   * @return  a basic description of the command
   */
  public String getCommandDescription() { return DESCR; }

  /**
   * Print out more detailed help on this command.
   *
   * @param  out  the stream to write the help to
   */
  public void printCommandHelp(PrintStream out) { out.println(USAGE); }

  /**
   * Does this command accept any arguments
   *
   * @return  true, since it does accept parameters.
   */
  public boolean takesCommandArguments() { return true; }

  /**
   * Check parameters and then run javadoc utility accordingly.
   * First parameters are options to be supplied to the javadoc utility.
   * Last parameter is the full name of a JBuilder project file.
   *
   * @param  params  an array of parameter values for this command
   */
  public void invokeCommand(String[] params) {
    if (params.length == 0) {
      // Must have one or more parameters
      System.err.println(LEADER + USAGE);
      System.exit(1);
    }
    // Initial parameter(s) are javadoc options
    ArrayList command = new ArrayList();
    for (int index = 0; index < params.length - 1; index++) {
      if (params[index].charAt(0) == '_') {
        command.add("-" + params[index].substring(1));
      }
      else {
        command.add(params[index].replace('\'', '"'));
      }
    }
    // Last parameter is the project file name
    File file = new File(params[params.length - 1]);
    if (!file.exists()) {
      System.err.println("The file '" + file.getAbsolutePath() +
          "' does not exist");
      System.exit(1);
    }
    try {
      Project project = Project.getProject(new Url(file));
      if (!(project instanceof JBProject)) {
        throw new InvalidUrlException();
      }
      runJavadoc(command, (JBProject)project);
    }
    catch (InvalidUrlException ex) {
      System.err.println("The file '" + file.getAbsolutePath() +
          "' is not a valid JBuilder project");
      System.exit(1);
    }
    // Suppress normal JBuilder startup
    PrimeTime.registerCommand(null, null);
  }

  /**
   * Prepare the command string and execute it.
   *
   * @param  command  contains any options for the javadoc utility
   * @param  project  is the reference to the JBuilder project
   */
  private void runJavadoc(ArrayList command, JBProject project) {
    if (command.size() == 0) {
      // Document protected and higher visibilities
      command.add("-protected");
      // Add the author's name
      String author = project.getProperty("Author");
      if (author != null && author.length() > 0) {
        command.add("-author");
      }
      // Add the version
      String version = project.getProperty("Version");
      if (version != null && version.length() > 0) {
        command.add("-version");
      }
      // Add the document title
      String title = project.getProperty("Title");
      if (title != null && title.length() > 0) {
        command.add("-doctitle");
        command.add("\"" + title + "\"");
      }
    }

    // Add Javadoc command
    command.add(0, getJavadocPath(project));
    // Write the output to this directory
    Url[] urls = project.getPaths().getDocPath();
    if (urls != null && urls.length > 0) {
      command.add("-d");
      command.add(urls[0].getFileObject().getAbsolutePath());
    }
    // Find imported classes here
    command.add("-classpath");
    command.add(getClassPath(project));
    // Document all source files
    addSourcePaths(command, getSourceBase(project), project);
    // Display the actual command line
    printCommand(command);
    try {
      // Execute the command
      Process process =
          Runtime.getRuntime().exec((String[])command.toArray(new String[0]));
      // And redirect its output to the console
      InputStream in = process.getInputStream();
      byte[] buffer = new byte[1024];
      int count = 0;
      while ((count = in.read(buffer)) != -1) {
        System.out.write(buffer, 0, count);
      }
    }
    catch (IOException ioe) {
      System.err.println("Error during processing");
      ioe.printStackTrace();
    }
  }

  /**
   * Find the path to the javadoc utility.
   *
   * @param  project  is the project being documented
   * @return  the full path to the javadoc utility
   */
  private String getJavadocPath(JBProject project) {
    File javadocPath =
        project.getPaths().getJDKPathSet().getHomePath().getFileObject();
    if (javadocPath == null) {
      System.err.println("Unable to locate project's Java VM");
      System.exit(1);
    }
    return javadocPath.getAbsolutePath() + File.separatorChar + "bin" +
        File.separatorChar + "javadoc";
  }

  /**
   * Find the class path for this project.
   *
   * @param  project  is the project being documented
   * @return  the full class path
   */
  private String getClassPath(JBProject project) {
    StringBuffer classPath = new StringBuffer(".");
    Url[] urls = project.getPaths().getFullClassPath();
    if (urls != null) {
      for (int index = 0; index < urls.length; index++) {
        classPath.append(File.pathSeparator).
            append(urls[index].getFileObject().getAbsolutePath());
      }
    }
    return classPath.toString();
  }

  /**
   * Find the base path for source files.
   *
   * @param  project  is the project being documented
   * @return  the full path to the base directory for the source files
   */
  private String getSourceBase(JBProject project) {
    Url[] paths = project.getPaths().getSourcePath();
    if (paths == null || paths.length == 0) {
      System.err.println("Unable to locate project's source path");
      System.exit(1);
    }
    File file = paths[0].getFileObject();
    if (file == null) {
      System.err.println("Unable to locate project's source path");
      System.exit(1);
    }
    return file.getAbsolutePath() + File.separatorChar;
  }

  /**
   * Accumulate all the paths to Java source files to be documented.
   * Starting from the project node, step down through the node hierarchy,
   * adding source paths as you go.
   *
   * @param  command     is the command to add to
   * @param  sourceBase  is the path to the base source directory
   * @param  node        is the current node to check out
   */
  private void addSourcePaths(ArrayList command, String sourceBase, Node node) {
    Node[] nodes = node.getChildren();
    for (int index = 0; index < nodes.length; index++) {
      if (nodes[index] instanceof JavaFileNode) {
        // Individual source file
        File file = ((JavaFileNode)nodes[index]).getUrl().getFileObject();
        if (file != null) {
          command.add(file.getAbsolutePath());
        }
      }
      else if (nodes[index] instanceof PackageNode) {
        // Package of source files - convert to directory under source base
        command.add(sourceBase + ((PackageNode)nodes[index]).getName().
            replace('.', File.separatorChar) + File.separatorChar + "*.java");
      }
      addSourcePaths(command, sourceBase, nodes[index]);
    }
  }

  /**
   * Display the completed Javadoc command line.
   *
   * @param  command  contains the command and its options
   */
  private void printCommand(ArrayList command) {
    StringBuffer buffer = new StringBuffer(">");
    for (int index = 0; index < command.size(); index++) {
      buffer.append(" ").append((String)command.get(index));
    }
    System.out.println();
    System.out.println(buffer.toString());
    System.out.println();
  }
}
