package wood.keith.opentools.layouts;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager2;
import java.io.Serializable;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;

/**
 * An extended border layout that adds locations in each corner.
 * Components are sized to the maximum of their preferred dimensions
 * around the edges of the layout, with the centre taking whatever is left.
 * For example, the maximum width of the northwest, west, and southwest
 * components is used as the western width. Similarly, the maximum height
 * of the northwest, north, and northeast components is used as the northern
 * height.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 */
public class BorderCornerLayout extends AbstractLayout
    implements LayoutManager2, Serializable {
  public static final String CENTER    = "Center";
  public static final String NORTH     = "North";
  public static final String SOUTH     = "South";
  public static final String EAST      = "East";
  public static final String WEST      = "West";
  public static final String NORTHEAST = "NorthEast";
  public static final String NORTHWEST = "NorthWest";
  public static final String SOUTHEAST = "SouthEast";
  public static final String SOUTHWEST = "SouthWest";
  static final String[] AREAS = new String[] {
    BorderCornerLayout.NORTHWEST, BorderCornerLayout.NORTH,
    BorderCornerLayout.NORTHEAST, BorderCornerLayout.WEST,
    BorderCornerLayout.CENTER, BorderCornerLayout.EAST,
    BorderCornerLayout.SOUTHWEST, BorderCornerLayout.SOUTH,
    BorderCornerLayout.SOUTHEAST
  };

  protected Component center, north, south, east, west,
    northeast, northwest, southeast, southwest;
  protected boolean cornersContribute = true;

  public BorderCornerLayout() {
    this(0, 0);
  }

  public BorderCornerLayout(int hgap, int vgap) {
    setHgap(hgap);
    setVgap(vgap);
  }

  public boolean getCornersContribute() { return cornersContribute; }

  public void setCornersContribute(boolean value) { cornersContribute = value; }

  /**
   * Save a reference to each added component based on its constraint.
   *
   * @param  comp        the component being added
   * @param  constraint  one of the values listed above
   */
  public void addLayoutComponent(Component comp, Object constraint) {
    if (constraint == CENTER || constraint == null) {
      center = comp;
    }
    if (constraint == NORTH) {
      north = comp;
    }
    if (constraint == SOUTH) {
      south = comp;
    }
    if (constraint == WEST) {
      west = comp;
    }
    if (constraint == EAST) {
      east = comp;
    }
    if (constraint == NORTHEAST) {
      northeast = comp;
    }
    if (constraint == NORTHWEST) {
      northwest = comp;
    }
    if (constraint == SOUTHEAST) {
      southeast = comp;
    }
    if (constraint == SOUTHWEST) {
      southwest = comp;
    }
  }

  /**
   * Drop the reference to the noiminated component.
   *
   * @param  comp  the component being removed
   */
  public void removeLayoutComponent(Component comp)
  {
    if (center == comp) {
      center = null;
    }
    if (north == comp) {
      north = null;
    }
    if (south == comp) {
      south = null;
    }
    if (west == comp) {
      west = null;
    }
    if (east == comp) {
      east = null;
    }
    if (northeast == comp) {
      northeast = null;
    }
    if (northwest == comp) {
      northwest = null;
    }
    if (southeast == comp) {
      southeast = null;
    }
    if (southwest == comp) {
      southwest = null;
    }
  }

  /**
   * Return the preferred or minimum dimensions of the component.
   *
   * @param  comp       the component whose dimensions are returned
   * @param  preferred  true for the preferred dimensions,
   *                    false for the minimum dimensions
   * @return  the dimensions as requested
   */
  private Dimension getPrefOrMinSize(Component comp, boolean preferred) {
    return (preferred ? comp.getPreferredSize() : comp.getMinimumSize());
  }

  /**
   * Retrieve the preferred/minimum height of the supplied component.
   *
   * @param  comp       the component being inspected (may be null)
   * @param  preferred  true for the preferred dimensions,
   *                    false for the minimum dimensions
   * @return  the requested height of the component, or zero if null
   */
  private int getVertSize(Component comp, boolean preferred) {
    return (comp == null ? 0 :
      (int)getPrefOrMinSize(comp, preferred).getHeight());
  }

  /**
   * Retrieve the preferred/minimum width of the supplied component.
   *
   * @param  comp       the component being inspected (may be null)
   * @param  preferred  true for the preferred dimensions,
   *                    false for the minimum dimensions
   * @return  the requested width of the component, or zero if null
   */
  private int getHorizSize(Component comp, boolean preferred) {
    return (comp == null ? 0 :
      (int)getPrefOrMinSize(comp, preferred).getWidth());
  }

  /**
   * Calculate the preferred/minimum dimensions of all the components.
   *
   * @param  container  the component using the layout
   * @param  preferred  true for the preferred dimensions,
   *                    false for the minimum dimensions
   * @return  the requested dimensions for all the components
   */
  private Dimension prefOrMinLayoutSize(Container container, boolean preferred) {
    Insets insets = container.getInsets();
    // Find the maximum heights/widths for each section
    int northHeight = getVertSize(north, preferred);
    int southHeight = getVertSize(south, preferred);
    int eastWidth = getHorizSize(east, preferred);
    int westWidth = getHorizSize(west, preferred);
    int centerHeight = (int)Math.max(getVertSize(center, preferred),
      Math.max(getVertSize(west, preferred), getVertSize(east, preferred)));
    int centerWidth = (int)Math.max(getHorizSize(center, preferred),
      Math.max(getHorizSize(north, preferred), getHorizSize(south, preferred)));
    if (cornersContribute || northHeight == 0) {
      northHeight = (int)Math.max(northHeight, Math.max(
        getVertSize(northwest, preferred), getVertSize(northeast, preferred)));
    }
    if (cornersContribute || southHeight == 0) {
      southHeight = (int)Math.max(southHeight, Math.max(
        getVertSize(southwest, preferred), getVertSize(southeast, preferred)));
    }
    if (cornersContribute || eastWidth == 0) {
      eastWidth = (int)Math.max(eastWidth, Math.max(
        getHorizSize(northeast, preferred), getHorizSize(southeast, preferred)));
    }
    if (cornersContribute || westWidth == 0) {
      westWidth = (int)Math.max(westWidth, Math.max(
        getHorizSize(northwest, preferred), getHorizSize(southwest, preferred)));
    }

    // And return their totals
    int height = northHeight + centerHeight + southHeight;
    int width = westWidth + centerWidth + eastWidth;
    return new Dimension(insets.left + insets.right + width +
      (westWidth == 0 ? 0 : hgap) + (eastWidth == 0 ? 0 : hgap),
      insets.top + insets.bottom + height +
      (northHeight == 0 ? 0 : vgap) + (southHeight == 0 ? 0 : vgap));
  }

  /**
   * Retrieve the preferred dimensions of the container using this layout.
   *
   * @param  container  the container using the layout
   * @return  the preferred dimensions of all its components
   */
  public Dimension preferredLayoutSize(Container container) {
    return prefOrMinLayoutSize(container, true);
  }

  /**
   * Retrieve the minimum dimensions of the container using this layout.
   *
   * @param  container  the container using the layout
   * @return  the minimum dimensions of all its components
   */
  public Dimension minimumLayoutSize(Container container) {
    return prefOrMinLayoutSize(container, false);
  }

  /**
   * Actually position the components managed by this layout.
   *
   * @param  container  the container using the layout
   */
  public void layoutContainer(Container container) {
    Dimension size = container.getSize();
    Insets insets = container.getInsets();

    // Find the maximum heights/widths for each section
    int northHeight = getVertSize(north, true);
    int southHeight = getVertSize(south, true);
    int eastWidth = getHorizSize(east, true);
    int westWidth = getHorizSize(west, true);
    if (cornersContribute || northHeight == 0) {
      northHeight = (int)Math.max(northHeight, Math.max(
        getVertSize(northwest, true), getVertSize(northeast, true)));
    }
    if (cornersContribute || southHeight == 0) {
      southHeight = (int)Math.max(southHeight, Math.max(
        getVertSize(southwest, true), getVertSize(southeast, true)));
    }
    if (cornersContribute || eastWidth == 0) {
      eastWidth = (int)Math.max(eastWidth, Math.max(
        getHorizSize(northeast, true), getHorizSize(southeast, true)));
    }
    if (cornersContribute || westWidth == 0) {
      westWidth = (int)Math.max(westWidth, Math.max(
        getHorizSize(northwest, true), getHorizSize(southwest, true)));
    }

    // Compute the starting offsets for each section
    int x0 = insets.left;
    int x1 = x0 + westWidth + (westWidth == 0 ? 0 : hgap);
    int x2 = size.width - insets.right - eastWidth;
    int y0 = insets.top;
    int y1 = y0 + northHeight + (northHeight == 0 ? 0 : vgap);
    int y2 = size.height - insets.bottom - southHeight;
    int w = x2 - x1 - (eastWidth == 0 ? 0 : hgap);
    int h = y2 - y1 - (southHeight == 0 ? 0 : vgap);

    // And apply them
    if (northwest != null) {
      northwest.setBounds(x0, y0, westWidth, northHeight);
    }
    if (north != null) {
      north.setBounds(x1, y0, w, northHeight);
    }
    if (northeast != null) {
      northeast.setBounds(x2, y0, eastWidth, northHeight);
    }
    if (west != null) {
      west.setBounds(x0, y1, westWidth, h);
    }
    if (center != null) {
      center.setBounds(x1, y1, w, h);
    }
    if (east != null) {
      east.setBounds(x2, y1, eastWidth, h);
    }
    if (southwest != null) {
      southwest.setBounds(x0, y2, westWidth, southHeight);
    }
    if (south != null) {
      south.setBounds(x1, y2, w, southHeight);
    }
    if (southeast != null) {
      southeast.setBounds(x2, y2, eastWidth, southHeight);
    }
  }

  /**
   * Test harness for the layout.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("BorderCornerLayout Demonstration");
    frame.getContentPane().setLayout(new BorderLayout());

    BorderCornerLayout layout = new BorderCornerLayout(9, 4);
//    layout.setCornersContribute(false);
    JPanel panel = new JPanel(layout);
    JButton swButton = new JButton("SouthWest");
    swButton.setPreferredSize(new Dimension(150, 90));
    panel.setBorder(BorderFactory.createEmptyBorder(9, 9, 9, 9));
    panel.add(BorderCornerLayout.CENTER, new JButton("Center"));
    panel.add(BorderCornerLayout.NORTH, new JButton("North"));
    panel.add(BorderCornerLayout.SOUTH, new JButton("South"));
    panel.add(BorderCornerLayout.WEST, new JButton("West"));
    panel.add(BorderCornerLayout.EAST, new JButton("East"));
    panel.add(BorderCornerLayout.NORTHEAST, new JButton("NorthEast"));
    panel.add(BorderCornerLayout.NORTHWEST, new JButton("NorthWest"));
    panel.add(BorderCornerLayout.SOUTHEAST, new JButton("SouthEast"));
    panel.add(BorderCornerLayout.SOUTHWEST, swButton);

    frame.getContentPane().add(panel);
    frame.setSize(500, 300);
    frame.setVisible(true);
  }
}
