package wood.keith.opentools.layouts;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Point;
import java.beans.PropertyEditor;
import java.util.Enumeration;
import java.util.HashMap;
import javax.swing.JComponent;

import com.borland.jbuilder.designer.ui.BasicLayoutAssistant;
import com.borland.jbuilder.designer.ui.DesignView;
import com.borland.jbuilder.designer.ui.ModelNode;
import com.borland.jbuilder.designer.ui.SelectNib;
import com.borland.jbuilder.designer.ui.UIDesigner;
import com.borland.jbuilder.designer.ui.opt.SelectBoxes;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.actions.ActionGroup;

/**
 * An assistant to help in the visual construction of a BorderCornerLayout.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  30 August 2001
 */
public class BorderCornerLayoutAssistant extends BasicLayoutAssistant {

  private static final String VERSION = "1.0";

  /**
   * Register the assistant, which also makes the associated layout
   * available in the drop-down box in the designer.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    UIDesigner.registerAssistant(BorderCornerLayoutAssistant.class,
      "wood.keith.opentools.layouts.BorderCornerLayout", true);
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded BorderCornerLayoutAssistant v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  private BorderCornerLayoutPropertyEditor propertyEditor = null;

  public BorderCornerLayoutAssistant() {
  }

  /**
   * Convert a point into one of the areas within this layout.
   *
   * @param  node   the node for the container that uses this layout
   * @param  point  the point to convert into an area
   * @return  the constraint name for this area
   */
  private String findArea(ModelNode node, Point point) {
    Dimension size = node.getLiveComponent().getSize();
    if (point == null) {
      // No specific drop point, then try to find a spare spot
      HashMap used = new HashMap(9);
      Enumeration children = node.children();
      while (children.hasMoreElements()) {
        Object child = children.nextElement();
        if (child instanceof ModelNode) {
          String area = ((ModelNode)child).getConstraints().getValueSource();
          area = area.substring(area.indexOf('.') + 1);
          used.put(area, area);
        }
      }
      for (int index = 0; index < BorderCornerLayout.AREAS.length; index++) {
        if (used.get(BorderCornerLayout.AREAS[index].toUpperCase()) == null) {
          return BorderCornerLayout.AREAS[index];
        }
      }
      // Default to centre
      return BorderCornerLayout.CENTER;
    }
    int limitW = (int)(size.getWidth() / 5);
    int limitH = (int)(size.getHeight() / 5);
    int areaIndex = getSection(point.getX(), size.getWidth(), limitW) +
      getSection(point.getY(), size.getHeight(), limitH) * 3;
    return BorderCornerLayout.AREAS[areaIndex];
  }

  /**
   * Retrieve the type of constraints used by this layout.
   *
   * @return  the constraint type
   */
  public String getConstraintsType() {
    return "java.lang.String";
  }

  /**
   * Retrieve the class name of the layout for display.
   *
   * @param  node  the node for the container using the layout
   * @return  the layout's class name (without any path info)
   */
  private String getLayoutClassName(ModelNode node) {
    String name = node.getSubcomponent().getAsContainer().getLayout().
      getClass().getName();
    int index = name.lastIndexOf('.');
    return (index == -1 ? name : name.substring(index + 1));
  }

  /**
   * Retrieve the property editor for this layout's constraints.
   *
   * @return  the appropriate property editor
   */
  public PropertyEditor getPropertyEditor() {
    if (propertyEditor == null) {
      propertyEditor = new BorderCornerLayoutPropertyEditor();
    }
    return propertyEditor;
  }

  /**
   * Convert a positional value into a section number (0..2).
   *
   * @param  pos    the position within the range
   * @param  max    the maximum position (minimum is zero)
   * @param  limit  the inset for the two edge sections
   * @return  0 if in the left inset, 1 if in the middle, 2 if in the right inset
   */
  private int getSection(double pos, double max, int limit) {
    return (pos < limit ? 0 : (pos > max - limit ? 2 : 1));
  }

  /**
   * Add items to the popup menu. Nothing to add.
   */
  public void prepareActionGroup(ActionGroup actionGroup) {
  }

  /**
   * Add a component to the container using this layout (including
   * drag-and-drop) with the appropriate layout constraint.
   * If an existing component is at the new position,
   * swap it to the old position.
   *
   * @param  addNode    the node for the component being added
   * @param  point      the point at which to add the component
   * @param  dimension  null if the component was just dropped, or
   *                    the dragged size of the component being added
   */
  public void prepareAddComponent(ModelNode addNode, Point point,
      Dimension dimension) {
    ModelNode contNode = (ModelNode)addNode.getParent();
    // Get area dropped into
    String area =
      "BorderCornerLayout." + findArea(contNode, point).toUpperCase();
    // Locate any existing occupant
    ModelNode occupant = null;
    Enumeration children = contNode.children();
    while (children.hasMoreElements()) {
      Object child = children.nextElement();
      if (child instanceof ModelNode) {
        ModelNode childNode = (ModelNode)child;
        if (childNode.getConstraints().getValueSource().equals(area)) {
          occupant = childNode;
          break;
        }
      }
    }
    // Set the new area for the dropped component
    String oldArea = addNode.getConstraints().getValueSource();
    addNode.getConstraints().setValueSource(area);
    if (occupant != null) {
      // And swap an existing occupant to its original position
      occupant.getConstraints().setValueSource(oldArea);
    }
  }

  /**
   * Provide feedback to the user when about to add a component.
   * Show the container name and its prospective constraint.
   *
   * @param  addNode    the node for the component being added
   * @param  contNode   the node for the container using the layout
   * @param  point      the point at which to add the component
   * @param  dimension  null if the component was just dropped, or
   *                    the dragged size of the component being added
   * @return  the feedback string
   */
  public String prepareAddStatus(ModelNode addNode, ModelNode contNode,
      Point point, Dimension dimension) {
    return contNode.getName() + " (" + getLayoutClassName(contNode) +
      "): " + findArea(contNode, point);
  }

  /**
   * Initialisation when setting this layout as the manager to use.
   *
   * @param  contNode  the node having its layout set
   */
  public void prepareChangeLayout(ModelNode contNode) {
    contNode.getModel().getComponentSource().getSourceFile().
      addImport("wood.keith.opentools.layouts.*");
    super.prepareChangeLayout(contNode);
  }

  /**
   * Provide feedback to the user when hovering over components.
   * Show the current component name and its constraint, or,
   * if no component, the container name and prospective constraint.
   *
   * @param  curNode   the node for the component the mouse is over
   * @param  contNode  the node for the container using the layout
   * @param  point     the point at which the mouse is located
   * @return  the feedback string
   */
  public String prepareMouseMoveStatus(ModelNode curNode, ModelNode contNode,
      Point point) {
    return (curNode != null ?
      curNode.getName() + ": " + curNode.getConstraints().getValueText() :
      contNode.getName() + " (" + getLayoutClassName(contNode) + ")");
  }

  /**
   * Provide feedback to the user when dragging a component.
   * Show the container name and its prospective constraint.
   * Also display a rectangle to show the corresponding location.
   *
   * @param  curNode      the node for the component being moved
   * @param  contNode     the node for the container using the layout
   * @param  location     the point to which to move the component
   * @param  selectBoxes  reference for drawing the feedback rectangle
   * @param  offset       the point within the component where the mouse clicked
   * @return  the feedback string - displayed on the status line
   */
  public String prepareMoveStatus(ModelNode curNode, ModelNode contNode,
      Point location, SelectBoxes selectBoxes, Point offset) {
    Container container = contNode.getSubcomponent().getAsContainer();
    Dimension size = container.getSize();
    Point absLocation = DesignView.componentAbsLocation(container);
    int limitW = (int)(size.getWidth() / 5);
    int limitH = (int)(size.getHeight() / 5);
    int areaIndex = getSection(location.getX(), size.getWidth(), limitW) +
      getSection(location.getY(), size.getHeight(), limitH) * 3;
    // Display a positioning outline
    Point point = new Point((int)absLocation.getX() + (areaIndex % 3 == 0 ? 0 :
      (areaIndex % 3 == 1 ? limitW : (int)size.getWidth() - limitW)),
      (int)absLocation.getY() + ((int)(areaIndex / 3) == 0 ? 0 :
      ((int)(areaIndex / 3) == 1 ? limitH : (int)size.getHeight() - limitH)));
    Dimension showSize = new Dimension(
      (areaIndex % 3 == 1 ? (int)size.getWidth() - 2 * limitW : limitW),
      ((int)(areaIndex / 3) == 1 ? (int)size.getHeight() - 2 * limitH : limitH));
    selectBoxes.show(0, point, showSize, 2);
    // Return the feedback value for the status bar
    return contNode.getName() + " (" + getLayoutClassName(contNode) + "): " +
      BorderCornerLayout.AREAS[areaIndex];
  }

  /**
   * Resize the component's preferred dimensions.
   *
   * @param  node       the node for the component selected
   * @param  selectNib  the resizing handle being dragged
   */
  public void prepareResizeComponent(ModelNode node, SelectNib selectNib) {
    JComponent component = (JComponent)node.getLiveComponent();
    component.setPreferredSize(selectNib.getRectangleDimension());
  }

  /**
   * Provide feedback for resizing a component's preferred dimensions.
   *
   * @param  node       the node for the component being resized
   * @param  point      the location of the mouse
   * @param  dimension  the new dimensions of the component
   * @return  the feedback string - displayed on the status line
   */
  public String prepareResizeStatus(ModelNode node, Point point,
      Dimension dimension) {
    return node.getName() + ": " +
      (int)dimension.getWidth() + "x" + (int)dimension.getHeight();
  }

  /**
   * Set up when user selects a component.
   * Provide a resizing handle if JComponent selected.
   *
   * @param  node        the node for the component selected
   * @param  designView  the designer
   */
  public void prepareSelectComponent(ModelNode node, DesignView designView) {
    super.prepareSelectComponent(node, designView);
    if (node.getLiveComponent() instanceof JComponent) {
      // Establish a resizing handle for JComponents - #3 is bottom right
      SelectNib[] nibs = designView.assureNibs(4);
      nibs[3].setBackground(Color.black);
      nibs[3].setLayoutAssistant(this);
      nibs[3].setSelectable(true);
      nibs[3].target = node;
    }
  }

  /**
   * Provide feedback for resizing a component's preferred dimensions.
   * Also draw an outline showing the new size.
   *
   * @param  point       the current location of the mouse
   * @param  designView  the designer
   * @param  selectNib   the resizing handle being dragged
   * @return  the feedback string - displayed on the status line
   */
  public String resizeAction(Point point, DesignView designView,
      SelectNib selectNib) {
    Point nibPoint = selectNib.getRectangleLocation();
    Dimension dimension = selectNib.getRectangleDimension();
    // Update rectangle size based on nib type and location
    DesignView.adjustPositionForNib(nibPoint, dimension, point, selectNib.type);
    // Draw the outline for the new size
    designView.getTempComponent().show(0, nibPoint, dimension, 2);
    return prepareResizeStatus((ModelNode)selectNib.target, point, dimension);
  }
}
