package wood.keith.opentools.otrunner;

import java.io.File;
import java.util.Map;

import com.borland.jbuilder.node.JBProject;
import com.borland.jbuilder.runtime.JavaRunner;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.Project;
import com.borland.primetime.properties.MapArrayProperty;
import com.borland.primetime.properties.MapProperty;
import com.borland.primetime.properties.PropertyManager;
import com.borland.primetime.properties.PropertyPage;
import com.borland.primetime.properties.PropertyPageFactory;
import com.borland.primetime.runtime.RuntimeManager;

/**
 * A runtime environment for OpenTools (i.e. a new JBuilder instance).
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  18 May 2002
 */
public class OTRunner extends JavaRunner {

  public static final String CATEGORY = "runtime";
  public static final MapProperty VM_PARAMS = new MapProperty(
    CATEGORY, "opentools.vmparameters", "-classic -Xverify:none -Xms8m -Xmx128m");
  public static final MapProperty JB_PARAMS = new MapProperty(
    CATEGORY, "opentools.parameters", "-verbose");
  public static final MapArrayProperty OTHER_DIRS = new MapArrayProperty(
    CATEGORY, "opentools.otherDirs");

  private static final String VERSION = "1.0";

  /**
   * Register the OpenTools runtime.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    RuntimeManager.registerRunner(new OTRunner());
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded OpenTools runtime v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  /**
   * Validate the runtime environment.
   */
  public boolean isValid(Browser browser, Project project, Map propertyMap,
      boolean debug) {
    return true;
  }

  /**
   * Return a factory for an appropriate property page.
   */
  public PropertyPageFactory getPageFactory(Project project, Map propertyMap) {
    if (project instanceof JBProject) {
      final Project aProject = project;
      final Map aPropertyMap = propertyMap;
      return new PropertyPageFactory("OpenTool", "Run an OpenTool") {
        public PropertyPage createPropertyPage() {
          return new OTRunnerPropertyPage(aProject, aPropertyMap);
        }
      };
    }
    else {
      return null;
    }
  }

  /**
   * Update the classpath for this runtime - add all JARs and Zips
   * from JBuilder/lib, and any other specified directories.
   */
  public String getClassPath(JBProject project, Map propertyMap) {
    String classPath = super.getClassPath(project, propertyMap);
    StringBuffer buffer = new StringBuffer(classPath);
    addToClassPath(buffer,
      new File(PropertyManager.getInstallRootUrl().getFileObject(), "lib"));
    String[] otherDirs = OTHER_DIRS.getValues(propertyMap);
    for (int index = 0; index < otherDirs.length; index++) {
      File dir = new File(otherDirs[index]);
      if (dir.exists() && dir.isDirectory()) {
        addToClassPath(buffer, dir);
      }
    }
    return buffer.toString();
  }

  /**
   * Add all JARs and Zips from a directory to the classpath.
   *
   * @param  buffer     is the current classpath to add to
   * @param  directory  is the location to scan
   * @return  all the
   */
  private void addToClassPath(StringBuffer buffer, File directory) {
    String classPath = buffer.toString();
    File[] files = directory.listFiles();
    for (int index = 0; index < files.length; index++) {
      if (files[index].getName().toLowerCase().endsWith("jar") ||
          files[index].getName().toLowerCase().endsWith("zip")) {
        if (classPath.indexOf(files[index].getAbsolutePath()) == -1) {
          buffer.append(File.pathSeparatorChar).
            append(files[index].getAbsolutePath());
        }
      }
    }
  }

  /**
   * Return the main class to run - JBuilder itself.
   */
  public String getMainClassName(JBProject project, Map propertyMap) {
    return "com.borland.jbuilder.JBuilder";
  }

  /**
   * Provide the parameters for JBuilder.
   */
  public String getParameters(JBProject project, Map propertyMap) {
    return JB_PARAMS.getValue(propertyMap);
  }

  /**
   * Supply the parameters for the JVM.
   */
  public String getVMParameters(JBProject project, Map propertyMap) {
    return VM_PARAMS.getValue(propertyMap);
  }
}
