package wood.keith.opentools.otrunner;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.Map;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import com.borland.primetime.properties.GlobalArrayProperty;
import com.borland.primetime.properties.PropertyPage;
import com.borland.primetime.help.HelpTopic;
import com.borland.primetime.help.ZipHelpBook;
import com.borland.primetime.help.ZipHelpTopic;
import com.borland.primetime.node.Project;
import com.borland.primetime.ui.ListPanel;

/**
 * A property page for the OpenTools runtime environment.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  18 May 2002
 */
public class OTRunnerPropertyPage extends PropertyPage implements ChangeListener {

  public static final String CATEGORY = "wood.keith.otrunner";
  public static final GlobalArrayProperty DEFAULT_DIRS =
    new GlobalArrayProperty(CATEGORY, "defaultDirs");

  private GridBagLayout gridBagLayout = new GridBagLayout();
  private JLabel jLabel1 = new JLabel();
  private JTextField vmparamsField = new JTextField();
  private JLabel jLabel2 = new JLabel();
  private JTextField jbparamsField = new JTextField();
  private JLabel jLabel3 = new JLabel();
  private JButton loadButton = new JButton();
  private JButton saveButton = new JButton();
  private DirectoryListPanel dirList = new DirectoryListPanel();

  private Map propertyMap = null;

  public OTRunnerPropertyPage(Project project, Map propertyMap) {
    this.propertyMap = propertyMap;
    try {
      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * UI initialisation.
   */
  private void jbInit() throws Exception {
    this.setLayout(gridBagLayout);
    jLabel1.setDisplayedMnemonic('V');
    jLabel1.setLabelFor(vmparamsField);
    jLabel1.setText("VM parameters:");
    vmparamsField.setToolTipText("Parameters for the JVM");
    jLabel2.setDisplayedMnemonic('J');
    jLabel2.setLabelFor(jbparamsField);
    jLabel2.setText("JBuilder parameters:");
    jbparamsField.setToolTipText("Parameters for JBuilder itself");
    jLabel3.setDisplayedMnemonic('O');
    jLabel3.setLabelFor(dirList);
    jLabel3.setText("Other directories:");
    loadButton.setMinimumSize(new Dimension(53, 21));
    loadButton.setPreferredSize(new Dimension(53, 21));
    loadButton.setToolTipText("Reload directory list from defaults");
    loadButton.setActionCommand("Load");
    loadButton.setMargin(new Insets(2, 4, 2, 4));
    loadButton.setMnemonic('L');
    loadButton.setText("Load");
    loadButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        loadButton_actionPerformed(e);
      }
    });
    loadButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        loadButton_actionPerformed(e);
      }
    });
    saveButton.setMinimumSize(new Dimension(53, 21));
    saveButton.setPreferredSize(new Dimension(53, 21));
    saveButton.setToolTipText("Save directory list as defaults");
    saveButton.setActionCommand("Save");
    saveButton.setMargin(new Insets(2, 4, 2, 4));
    saveButton.setMnemonic('S');
    saveButton.setText("Save");
    saveButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        saveButton_actionPerformed(e);
      }
    });
    dirList.addChangeListener(this);
    this.add(jLabel1, new GridBagConstraints(0, 0, 2, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(0, 4, 0, 4), 0, 0));
    this.add(vmparamsField, new GridBagConstraints(0, 1, 3, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 4, 0, 4), 0, 0));
    this.add(jLabel2, new GridBagConstraints(0, 2, 2, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(4, 4, 0, 4), 0, 0));
    this.add(jbparamsField, new GridBagConstraints(0, 3, 3, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 4, 0, 4), 0, 0));
    this.add(jLabel3, new GridBagConstraints(0, 4, 1, 1, 1.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(4, 4, 4, 4), 0, 0));
    this.add(dirList, new GridBagConstraints(0, 5, 3, 1, 1.0, 1.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
    this.add(loadButton, new GridBagConstraints(1, 4, 1, 1, 0.0, 0.0
            ,GridBagConstraints.NORTH, GridBagConstraints.HORIZONTAL, new Insets(4, 0, 0, 0), 0, 0));
    this.add(saveButton, new GridBagConstraints(2, 4, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(4, 4, 0, 4), 0, 0));
    disEnableButtons();
  }

  /**
   * Disable/enable buttons depending on current state.
   */
  private void disEnableButtons() {
    loadButton.setEnabled(DEFAULT_DIRS.getValues().length > 0);
    saveButton.setEnabled(dirList.getList().size() > 0);
  }

  /**
   * Respond to changes in the directory list.
   */
  public void stateChanged(ChangeEvent e) {
    disEnableButtons();
  }

  /**
   * Load the list of default directories.
   */
  void loadButton_actionPerformed(ActionEvent e) {
    loadDirectories(DEFAULT_DIRS.getValues());
    disEnableButtons();
  }

  /**
   * Save the list of default directories.
   */
  void saveButton_actionPerformed(ActionEvent e) {
    DEFAULT_DIRS.setValues(saveDirectories());
  }

  /**
   * Load the properties into the UI.
   */
  public void readProperties() {
    vmparamsField.setText(OTRunner.VM_PARAMS.getValue(propertyMap));
    jbparamsField.setText(OTRunner.JB_PARAMS.getValue(propertyMap));
    loadDirectories(OTRunner.OTHER_DIRS.getValues(propertyMap));
    disEnableButtons();
  }

  /**
   * Load the list of directories into the UI.
   *
   * @param  dirs  the list of directories to load
   */
  public void loadDirectories(String[] dirs) {
    dirList.clear();
    for (int index = 0; index < dirs.length; index++) {
      dirList.addListElement(dirs[index]);
    }
  }

  /**
   * Save the properties from the UI.
   */
  public void writeProperties() {
    OTRunner.VM_PARAMS.setValue(propertyMap, vmparamsField.getText());
    OTRunner.JB_PARAMS.setValue(propertyMap, jbparamsField.getText());
    OTRunner.OTHER_DIRS.setValues(propertyMap, saveDirectories());
  }

  /**
   * Save the list of directories from the UI.
   *
   * @return  the list of directories to save
   */
  public String[] saveDirectories() {
    String[] dirs = new String[dirList.getList().size()];
    for (int index = 0; index < dirs.length; index++) {
      dirs[index] = (String)dirList.getList().get(index);
    }
    return dirs;
  }

  /**
   * Provide help for this page.
   */
  public HelpTopic getHelpTopic() {
    return new ZipHelpTopic(new ZipHelpBook("OTRunnerDoc.jar"),
      "wood/keith/opentools/otrunner/OTRunner.html");
  }
}
