package wood.keith.opentools.otnode;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import com.borland.jbuilder.jot.JotClass;
import com.borland.jbuilder.jot.JotExpression;
import com.borland.jbuilder.jot.JotExpressionStatement;
import com.borland.jbuilder.jot.JotFile;
import com.borland.jbuilder.jot.JotMethodSource;
import com.borland.jbuilder.jot.JotPackages;
import com.borland.jbuilder.jot.JotPrimitiveClass;
import com.borland.jbuilder.jot.JotStatement;
import com.borland.jbuilder.node.JBProject;
import com.borland.jbuilder.node.JavaFileNode;
import com.borland.jbuilder.node.PackageNode;
import com.borland.jbuilder.paths.ProjectPathSet;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.help.HelpManager;
import com.borland.primetime.help.ZipHelpBook;
import com.borland.primetime.help.ZipHelpTopic;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.node.Node;
import com.borland.primetime.personality.Personality;
import com.borland.primetime.util.Streams;
import com.borland.primetime.util.VetoException;
import com.borland.primetime.vfs.InvalidUrlException;
import com.borland.primetime.vfs.Url;
import com.borland.primetime.wizard.BasicWizard;
import com.borland.primetime.wizard.Wizard;
import com.borland.primetime.wizard.WizardAction;
import com.borland.primetime.wizard.WizardHost;
import com.borland.primetime.wizard.WizardManager;
import com.borland.primetime.wizard.WizardPage;

/**
 * This class defines the "OpenTools Manifest" wizard at the highest level.
 * It constructs the classes.opentools file based on classes in the project.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  20 May 2003
 * @version  1.1  22 December 2003
 */
public class OTManifestWizard extends BasicWizard {

  private static final String TITLE = "OpenTools Manifest";

  private static Integer _archiverIndex;
  private static Integer _buildIndex;
  private static Integer _coreIndex;
  private static Integer _designerIndex;
  private static Integer _docletIndex;
  private static Integer _editorIndex;
  private static Integer _nodeIndex;
  private static Integer _serversIndex;
  private static Integer _testIndex;
  private static Integer _uiIndex;
  private static Integer _urlChooserIndex;
  private static Integer _vfsIndex;
  private static Integer _wizardIndex;

  // Internal variables
  private OTManifestWizardPage1 _wizardPage1 = null;
  private Map _tools = null;

  /**
   * Register the "OpenTools Manifest" tool.
   *
   * Provides the needed OpenTools interface required to register the
   * WizardAction which defines and creates this wizard.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    WizardManager.registerWizardAction(
      OTManifestWizard.WIZARD_Classes_OpenTools);
    _archiverIndex = findCategory("Archiver");
    _buildIndex = findCategory("Build");
    _coreIndex = findCategory("Core");
    _designerIndex = findCategory("Designer");
    _docletIndex = findCategory("Doclet");
    _editorIndex = findCategory("Editor");
    _nodeIndex = findCategory("Node");
    _serversIndex = findCategory("Servers");
    _testIndex = findCategory("TestRunner");
    _uiIndex = findCategory("UI");
    _urlChooserIndex = findCategory("UrlChooser");
    _vfsIndex = findCategory("VFS");
    _wizardIndex = findCategory("Wizard");
  }

  /**
   * Locate the index for a particular OpenTools category entry.
   *
   * @param  name  the category name
   * @return  the corresponding index as an Integer
   */
  private static Integer findCategory(String name) {
    return new Integer(Arrays.binarySearch(OTScanner.KEYWORDS, name));
  }

  /**
   * Definition of "OpenTools Manifest" WizardAction.
   *
   * This is an OpenTools registered action defining where the wizard will
   * appear in the IDE and a factory for creating each instance of this wizard.
   */
  public static final WizardAction WIZARD_Classes_OpenTools = new WizardAction(
      "OpenTools Manifest", '\0',
      "Generate an OpenTools manifest file for this project",
      new ImageIcon(ClassLoader.getSystemResource(
      OTManifestWizard.class.getName().replace('.', '/') + ".gif")), null,
      false, null) {

    // Override method so wizard is disabled if no active project.
    public void update(Object source) {
      Browser browser = Browser.findBrowser(source);
      setEnabled((browser != null) && (browser.getActiveProject() != null));
    }

    protected Wizard createWizard() {
      return new OTManifestWizard();
    }

    /**
     * This action only applies to the OpenTools personality.
     *
     * @return  the list of applicable personalities
     */
    public Personality[] getPersonalities() {
      return OTPersonality.PERSONALITIES;
    }
  };

  /**
   * Wizard startup.
   *
   * "OpenTools Manifest" wizard is to become visible,
   * create and order the wizard pages to be displayed.
   *
   * @param  host  the WizardHost that owns this wizard instance.
   * @return  the initial WizardPage to show.
   */
  public WizardPage invokeWizard(WizardHost host) {
    Browser browser = host.getBrowser();
    JBProject project = (JBProject)browser.getActiveProject();
    _tools = scanProjectForTools(project);
    setWizardTitle(TITLE);
    _wizardPage1 = new OTManifestWizardPage1(this, _tools);
    addWizardPage(_wizardPage1);
    return super.invokeWizard(host);
  }

  /**
   * Perform scan.
   * Invokes the scan of the Java source in this project for OpenTools candidates.
   *
   * @param  project  the current project
   * @return  a list of class names representing the OpenTool candidates found,
   *          and their suggested OpenTools category
   */
  protected Map scanProjectForTools(JBProject project) {
    Map tools = new TreeMap();
    scanNodesForTools(project, project.getChildren(), tools);
    return tools;
  }

  /**
   * Perform scan.
   * Invokes the scan of the Java source in this project for OpenTools candidates.
   *
   * @param  project   the current project
   * @param  nodes     the set of nodes to scan
   * @param  tools     the set of tool classes found (updated)
   */
  protected void scanNodesForTools(JBProject project, Node[] nodes,
      Map tools) {
    for (int index = 0; index < nodes.length; index++) {
      if (nodes[index] instanceof JavaFileNode) {
        scanFileForTools(project, (JavaFileNode)nodes[index], tools);
      }
      if (nodes[index] instanceof PackageNode &&
          nodes[index].getParent() == project) {
        // Scan into packages, but only to one level
        scanNodesForTools(project,
          ((PackageNode)nodes[index]).getDisplayChildren(), tools);
      }
    }
  }

  /**
   * Scans a file looking for classes with
   * public static void initOpenTools(byte, byte) method.
   *
   * @param  project   the current JBuilder project
   * @param  fileNode  the node whose content is to be scanned
   * @param  tools     the set of tool classes found (updated)
   */
  protected void scanFileForTools(JBProject project,
      JavaFileNode fileNode, Map tools) {
    JotPackages jpackages = project.getJotPackages();
    JotFile jfile = jpackages.getFile(fileNode.getUrl());
    JotClass[] jclasses = jfile.getClasses();
    for (int index = 0; index < jclasses.length; index++) {
      JotMethodSource jmethod = (JotMethodSource)jclasses[index].
        getDeclaredMethod("initOpenTool", new JotClass[]
        {JotPrimitiveClass.byteType, JotPrimitiveClass.byteType});
      if (jmethod != null && jmethod.getReturnType().getName().equals("void") &&
          Modifier.isPublic(jmethod.getModifiers()) &&
          Modifier.isStatic(jmethod.getModifiers())) {
        tools.put(jclasses[index].getName(),
          suggestCategory(jmethod.getStatements()));
      }
    }
    jpackages.release(jfile);
  }

  /**
   * Suggest an OpenTool category for the class based
   * on the contents of the initOpenTool method.
   *
   * @param  jmethod  the initOpenTool method
   * @return  the OpenTool category, or null if no suggestion
   */
  protected Integer suggestCategory(JotStatement[] jstatements) {
    for (int index = 0; index < jstatements.length; index++) {
      if (jstatements[index] instanceof JotExpressionStatement) {
        JotExpression jexpr =
          ((JotExpressionStatement)jstatements[index]).getExpression();
        if (jexpr.getMethodCall() != null) {
          String methodName = jexpr.getMethodCall().getFullName();
          if (methodName.indexOf("ArchiveWizard.registerArchiver") > -1) {
            return _archiverIndex;
          }
          if (methodName.indexOf("Browser.addStaticBrowserListener") > -1) {
            return _uiIndex;
          }
          if (methodName.indexOf("Browser.registerNodeViewerFactory") > -1) {
            return _uiIndex;
          }
          if (methodName.indexOf("BuilderManager.registerBuilder") > -1) {
            return _buildIndex;
          }
          if (methodName.indexOf("DesignerManager.registerDesigner") > -1) {
            return _designerIndex;
          }
          if (methodName.indexOf("EditorManager.registerContextActionProvider") > -1) {
            return _editorIndex;
          }
          if (methodName.indexOf("EditorManager.registerEditorKit") > -1) {
            return _editorIndex;
          }
          if (methodName.indexOf("JavadocWizard.registerDoclet") > -1) {
            return _docletIndex;
          }
          if (methodName.indexOf("JBuilderMenu.") > -1 && methodName.endsWith("add")) {
            return _uiIndex;
          }
          if (methodName.indexOf("PrimeTime.registerCommand") > -1) {
            return _coreIndex;
          }
          if (methodName.indexOf("PropertyManager.registerPropertyGroup") > -1) {
            return _uiIndex;
          }
          if (methodName.indexOf("registerFileNodeClass") > -1) {
            return _nodeIndex;
          }
          if (methodName.indexOf("RuntimeManager.registerRunner") > -1) {
            return _uiIndex;
          }
          if (methodName.indexOf("ServerManager.registerServer") > -1) {
            return _serversIndex;
          }
          if (methodName.indexOf("TestRunner.registerTestRunner") > -1) {
            return _testIndex;
          }
          if (methodName.indexOf("UIDesigner.registerAssistant") > -1) {
            return _designerIndex;
          }
          if (methodName.indexOf("Url.registerFilesystem") > -1) {
            return _vfsIndex;
          }
          if (methodName.indexOf("UrlExplorerPage.registerUrlExplorerCustomizer") > -1) {
            return _urlChooserIndex;
          }
          if (methodName.indexOf("WizardManager.registerWizardAction") > -1) {
            return _wizardIndex;
          }
        }
      }
      Integer result = suggestCategory(jstatements[index].getStatements());
      if (result != null) {
        return result;
      }
    }
    return null;
  }

  /**
   * Assemble file Url.
   *
   * From the given parameters generates a fully qualified location
   * where the "OpenTools Manifest" file will be written.
   * This is returned in the form of a Url.
   *
   * @param  project      the JBProject in which the node will be created.
   * @param  packageName  the name of the package in which file will be created.
   * @param  fileName     the name of the file to be created.
   * @return  the Url which was created.
   */
  protected static Url makeFileUrl(JBProject project,
      String packageName, String fileName) {
    try {
      ProjectPathSet paths = project.getPaths();
      Url[] sourcePaths = paths.getSourcePath();
      String projectPath = "";

      if (sourcePaths.length > 0) {
        projectPath = sourcePaths[0].getFile();
      }

      String dirName = projectPath;
      if (packageName != null && packageName.length() > 0) {
        dirName = dirName + '/' + packageName.replace('.', '/');
      }
      String filePath = dirName + '/' + fileName;
      File newFile = new File(filePath);
      return new Url(newFile);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      return null;
    }
  }

  /**
   * Creates/replaces file node.
   *
   * From the given parameters create/replace a project file node for the
   * "OpenTools Manifest" file to be written.
   *
   * @param  project      the JBProject in which the node is created.
   * @param  packageName  the name of the package in which file is created.
   * @param  fileName     the name of the file to be created.
   * @return  the FileNode which was created.
   */
  protected static FileNode createNode(JBProject project, String fileName)
      throws InvalidUrlException {
    Url filePath = new Url(new File(
      project.getPaths().getWorkingDirectory().getFullName() + "/" + fileName));
    // getNode() will create a node if does not exist
    FileNode newNode = project.getNode(filePath);
    newNode.setParent(project);
    return newNode;
  }

  /**
   * Perform code generation.
   *
   * Produces the "OpenTools Manifest" file as defined by user input on the
   * wizard pages. It is added as a node to the currently active project
   * and replaces any previously generated file of the same name.
   */
  protected void finish() throws VetoException {
    if (_tools.size() == 0) {
      return;
    }

    Browser browser = wizardHost.getBrowser();
    JBProject project = (JBProject)browser.getProjectView().getActiveProject();
    try {
      FileNode textNode = createNode(project, "classes.opentools");
      StringBuffer buffer = new StringBuffer();
      Map tools = _wizardPage1.getTools();
      if (_wizardPage1.isMerging()) {
        loadCategories(textNode, tools);
      }
      Map categories = getCategories(tools);
      Iterator iter = categories.keySet().iterator();
      while (iter.hasNext()) {
        String category = (String)iter.next();
        ArrayList classes = (ArrayList)categories.get(category);
        String prefix = category + ": ";
        for (int index = 0; index < classes.size(); index++) {
          buffer.append(prefix).append(classes.get(index)).append("\n");
          prefix = "  ";
        }
      }
      textNode.getBuffer().setContent(buffer.toString().getBytes());
      textNode.save();
      // Open the generated/updated file in the Content Pane
      browser.setActiveNode(textNode, true);
    }
    catch (Exception ex) {
      JOptionPane.showMessageDialog(wizardHost.getDialogParent(),
        ex.getMessage(), TITLE, JOptionPane.ERROR_MESSAGE);
    }
  }

  /**
   * Merge existing class-category mappings from .opentools file.
   *
   * @param  node   the existing classes.opentools file
   * @param  tools  the mapping to update
   */
  private void loadCategories(FileNode node, Map tools) throws IOException {
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    Streams.copy(node.getInputStream(), out);
    StringTokenizer tokens = new StringTokenizer(out.toString());
    String category = null;
    while (tokens.hasMoreTokens()) {
      String token = tokens.nextToken();
      if (token.charAt(token.length() - 1) == ':') {
        category = token.substring(0, token.length() - 1);
      }
      else if (tools.get(token) == null) {
        tools.put(token, category);
      }
    }
  }

  /**
   * Return the set of categories, mapped onto all the classes therein.
   *
   * @return  the category-classes mapping
   */
  private Map getCategories(Map tools) {
    Map categories = new TreeMap();
    Iterator iter = tools.entrySet().iterator();
    while (iter.hasNext()) {
      Map.Entry entry = (Map.Entry)iter.next();
      String category = (String)entry.getValue();
      ArrayList classes = (ArrayList)categories.get(category);
      if (classes == null) {
        classes = new ArrayList();
        categories.put(category, classes);
      }
      classes.add(entry.getKey());
    }
    return categories;
  }

  /**
   * Wizard termination.
   *
   * The Wizard has been completed (finished or cancelled) and any resource
   * cleanup to help the garbage collector can occur here.
   */
  public void wizardCompleted() {
    _tools.clear();
    _tools = null;
  }

  /**
   * Provide help for the wizard.
   *
   * @param  page  the current page in the wizard
   * @param  host  the wizard host
   */
  public void help(WizardPage page, WizardHost host) {
    HelpManager.showHelp(new ZipHelpTopic(
      new ZipHelpBook("OpenToolsNodeDoc.jar"),
      this.getClass().getName().replace('.', '/') + ".html"),
      page.getPageComponent(host));
  }
}
