package wood.keith.opentools.otnode;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.DefaultCellEditor;
import javax.swing.DefaultListCellRenderer;
import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.plaf.basic.BasicComboBoxEditor;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import com.borland.primetime.util.ClipPath;
import com.borland.primetime.util.VetoException;
import com.borland.primetime.wizard.BasicWizardPage;

/**
 * This class contains the wizard page which allows the user to specify
 * the details about the OpenTools manifest.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  20 May 2003
 */
public class OTManifestWizardPage1 extends BasicWizardPage {

	// Internal variables
	private OTManifestWizard _wizard;

	private BorderLayout _borderLayout1 = new BorderLayout();
	private JPanel _bodyPanel = new JPanel();
	private JScrollPane _toolsScroll = new JScrollPane();
	private JCheckBox _mergeCheck = new JCheckBox();
	private BorderLayout _borderLayout2 = new BorderLayout();
	private JTable _toolsTable = new JTable();
	private ToolsTableModel _toolsModel = new ToolsTableModel();

	/**
	 * Create a new wizard page.
	 *
	 * @param  wizard  the wizard invoking this page
	 * @param  tools   a mapping of tool class names to
	 *                 suggested OpenTool categories
	 */
	public OTManifestWizardPage1(OTManifestWizard wizard, Map tools) {
		try {
			_wizard = wizard;
			jbInit();
			loadData(tools);
		}
		catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * Define the user interface.
	 *
	 * The user interface is constructed here so the JBuilder UI Designer
	 * may be used to edit it.
	 */
	protected void jbInit() throws Exception {
		this.setPageTitle("OpenTools Classes");
		try {
			this.setLargeIcon(new ImageIcon(this.getClass().getResource(
				"/com/borland/jbuilder/wizard/unittest/images/FixtureLarge.gif")));
		}
		catch (Exception ex) {
			// Ignore
		}
		this.setInstructions("This tool has scanned your project for OpenTools candidates, " +
			"showing them in the grid below. Select an OpenTools category for each class " +
			"identified, or enter your own category. When finished, the classes.opentools " +
			"file for the project is rewritten.");
		this.setLayout(_borderLayout1);
		_mergeCheck.setMnemonic('M');
		_mergeCheck.setText("Merge with existing file");
		_toolsScroll.setPreferredSize(new Dimension(200, 200));
		_toolsTable.setModel(_toolsModel);
		_toolsTable.setRowHeight(20);
		_toolsTable.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
		_toolsModel.setTableLayout(_toolsTable);
		this.add(_bodyPanel, BorderLayout.CENTER);
		_bodyPanel.setLayout(_borderLayout2);
		_bodyPanel.add(_toolsScroll,  BorderLayout.CENTER);
		_toolsScroll.getViewport().add(_toolsTable, null);
		_bodyPanel.add(_mergeCheck,  BorderLayout.NORTH);
	}

	/**
	 * Initialise UI based on presence of tools.
	 */
	private void loadData(Map tools) {
		if (tools.size() == 0) {
			setInstructions("No candidate OpenTool classes were found in this project.\n\n" +
				"OpenTool classes have a static initOpenTool method for registration purposes.");
			_bodyPanel.setVisible(false);
		}
		else {
			_toolsModel.setTools(tools);
		}
	}

	/**
	 * Check for user input error.
	 * The user has completed this page so check for input error.
	 * If any, then report the error in a dialog and throw an exception.
	 */
	public void checkPage() throws VetoException {
	}

	/**
	 * Check for a user input error.
	 *
	 * @param  condition  a test that must be true for the page to be valid
	 * @param  message    the text to display
	 * @param  focus      the UI component to move focus to
	 * @throws  VetoException  if the condition is not met
	 */
	private void checkCondition(boolean condition, String message,
			JComponent focus) throws VetoException {
		if (!condition) {
			focus.requestFocus();
			JOptionPane.showMessageDialog(wizardHost.getDialogParent(), message,
				OTProjectWizard.TITLE, JOptionPane.ERROR_MESSAGE, null);
			throw new VetoException();
		}
	}

	/**
	 * Fetches set of OpenTools classes.
	 *
	 * @return  a mapping from the OpenTools classes to their categories
	 */
	public Map getTools() { return _toolsModel.getTools(); }

	/**
	 * Merge with an existing classes.opentools file.
	 *
	 * @return  true if merging with an existing file, false otherwise
	 */
	public boolean isMerging() { return _mergeCheck.isSelected(); }
}

/**
 * A table model for the tool details.
 */
class ToolsTableModel extends AbstractTableModel {

	private static final String[] COLUMN_NAMES = {"Class", "Category"};
	private static final Class[] COLUMN_CLASSES = {String.class, Object.class};

	private static final String EXCLUDE = "<exclude>";

	/** Editor for the tool categories. */
	private static final JComboBox CATEGORY_EDITOR = new JComboBox();
	/** Renderer for the tool categories. */
	private static final CategoryCellRenderer CATEGORY_RENDERER =
		new CategoryCellRenderer();

	static {
		// Initialise categories
		CATEGORY_EDITOR.addItem(EXCLUDE);
		for (int index = 0; index < OTScanner.KEYWORDS.length; index++) {
			CATEGORY_EDITOR.addItem(new Integer(index));
		}
		CATEGORY_EDITOR.setRenderer(CATEGORY_RENDERER);
		CATEGORY_EDITOR.setEditor(new CategoryEditor());
		CATEGORY_EDITOR.setEditable(true);
	}

	private ArrayList _data = new ArrayList();

	public String getColumnName(int column) { return COLUMN_NAMES[column]; }

	public Class getColumnClass(int columnIndex) {
		return COLUMN_CLASSES[columnIndex];
	}

	public int getColumnCount() { return COLUMN_NAMES.length; }

	public int getRowCount() { return _data.size(); }

	/**
	 * Return the set of tool classes mapped onto categories.
	 *
	 * @return  the classes-category mapping
	 */
	public Map getTools() {
		Map tools = new TreeMap();
		for (int index = 0; index < _data.size(); index++) {
			Object[] values = (Object[])_data.get(index);
			if (values[1].equals(EXCLUDE)) {
				continue;
			}
			String category = (values[1] instanceof Integer ?
				OTScanner.KEYWORDS[((Integer)values[1]).intValue()] : (String)values[1]);
			tools.put(values[0], "OpenTools-" + category);
		}
		return tools;
	}

	public Object getValueAt(int rowIndex, int columnIndex) {
		Object[] values = (Object[])_data.get(rowIndex);
		switch (columnIndex) {
			case 0:  // Fall through
			case 1:  return values[columnIndex];
			default: return null;
		}
	}

	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return (columnIndex == 1);
	}

	/**
	 * Establish the layout for the table that displays this model.
	 *
	 * @param  table  the table using this model
	 */
	public void setTableLayout(JTable table) {
		TableColumn column = table.getColumnModel().getColumn(0);
		column.setCellRenderer(new ClassCellRenderer());
		column.setPreferredWidth(300);
		column = table.getColumnModel().getColumn(1);
		column.setCellRenderer(CATEGORY_RENDERER);
		column.setCellEditor(new DefaultCellEditor(CATEGORY_EDITOR));
		column.setPreferredWidth(100);
	}

	/**
	 * Load a mapping of tool classes to categories into the table.
	 *
	 * @param  tools  the classes-categories mapping
	 */
	public void setTools(Map tools) {
		_data.clear();
		Iterator iter = tools.keySet().iterator();
		while (iter.hasNext()) {
			Object[] values = new Object[2];
			values[0] = iter.next();
			values[1] = tools.get(values[0]);
			if (values[1] == null) {
				values[1] = EXCLUDE;
			}
			_data.add(values);
		}
		fireTableDataChanged();
	}

	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		Object[] values = (Object[])_data.get(rowIndex);
		switch (columnIndex) {
			case 1:  values[columnIndex] = aValue; break;
		}
		fireTableRowsUpdated(rowIndex, rowIndex);
	}
}

/**
 * Class name table cell renderer that clips long class names.
 */
class ClassCellRenderer extends JLabel implements TableCellRenderer {

	/**
	 * Clip long class names.
	 *
	 * @param  table       the table being drawn into
	 * @param  value       the value to depict
	 * @param  isSelected  true if cell is selected
	 * @param  hasFocus    true if cell is focused
	 * @param  row         index of the table row
	 * @param  column      index of the table column
	 * @return  the rendering component
	 */
	public Component getTableCellRendererComponent(JTable table, Object value,
			boolean isSelected, boolean hasFocus, int row, int column) {
		if (isSelected) {
			setBackground(table.getSelectionBackground());
			setForeground(table.getSelectionForeground());
		} else {
			setBackground(table.getBackground());
			setForeground(table.getForeground());
		}
		setFont(table.getFont());
		ClipPath clipper = new ClipPath();
		String text = clipper.clipCenter(table.getFont(),
			(value != null ? value.toString() : "").replace('.', '/'),
			table.getColumnModel().getColumn(column).getWidth());
		setText(text.replace('/', '.'));
		return this;
	}
}

/**
 * Category renderer that shows name and JBuilder version (when applicable).
 * Works for both table cells and combobox.
 */
class CategoryCellRenderer extends DefaultListCellRenderer
		implements TableCellRenderer {

	public Component getTableCellRendererComponent(JTable table, Object value,
			boolean isSelected, boolean hasFocus, int row, int column) {
		if (isSelected) {
			setBackground(table.getSelectionBackground());
			setForeground(table.getSelectionForeground());
		} else {
			setBackground(table.getBackground());
			setForeground(table.getForeground());
		}
		setFont(table.getFont());
		setText(getDisplayText(value));
		return this;
	}

	public Component getListCellRendererComponent(JList list, Object value,
			int index, boolean isSelected, boolean cellHasFocus) {
		Component result = super.getListCellRendererComponent(
			list, value, index, isSelected, cellHasFocus);
		setText(getDisplayText(value));
		return result;
	}

	/**
	 * Prepare the entry for display.
	 * If an Integer then use as an index into the array of known categories,
	 * and show category name and JBuilder version when it first appeared.
	 * Otherwise it is a String and is displayed directly.
	 *
	 * @param  value  the value to display
	 * @return  the appropriate display text
	 */
	static String getDisplayText(Object value) {
		int index = (value instanceof Integer ? ((Integer)value).intValue() : -1);
		return (index == -1 ? value.toString() : OTScanner.KEYWORDS[index] +
			" (" + OTScanner.KEYWORD_VERSIONS[index] + ")");
	}
}

/**
 * Editor for the categories to handle Integer vs String translations.
 */
class CategoryEditor extends BasicComboBoxEditor {

	public void setItem(Object anObject) {
		editor.setText(anObject == null ? "" :
			CategoryCellRenderer.getDisplayText(anObject));
	}

	public Object getItem() {
		String value = editor.getText();
		for (int index = 0; index < OTScanner.KEYWORDS.length; index++) {
			if (value.equals(OTScanner.KEYWORDS[index] +
					" (" + OTScanner.KEYWORD_VERSIONS[index] + ")")) {
				return new Integer(index);
			}
		}
		return value;
	}
}