package wood.keith.opentools.otnode;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import javax.swing.Action;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.actions.ActionGroup;
import com.borland.primetime.actions.ActionPopupMenu;
import com.borland.primetime.editor.EditorAction;
import com.borland.primetime.editor.EditorContextActionProvider;
import com.borland.primetime.editor.EditorManager;
import com.borland.primetime.editor.EditorPane;
import com.borland.primetime.ide.Browser;

/**
 * A popup menu for the OpenTools node structure pane
 * that lets you add new categories.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  12 September 2001
 * @version  1.1  19 February 2002
 */
public class OTPopupMenu extends ActionPopupMenu {

  public static final OTMenuGroup menuGroup = new OTMenuGroup();

  private OTStructure structure;

  /**
   * When the OpenTool add-ins are loaded, a ContextActionProvider
   * is registered for the EditorPane.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    EditorManager.registerContextActionProvider(new ActionProvider());
  }

  private static class ActionProvider implements EditorContextActionProvider {
    /**
     * The priority for this menu entry - 100 towards the top, 0 towards the bottom.
     *
     * @return  4 to place in near the bottom
     */
    public int getPriority() { return 4; }

    /**
     * Return the action to add to the popup context menu in the editor.
     * Return the appropriate action only if this is an OpenTools manifest file.
     *
     * @param  editor  the editor to work with
     * @return  the new action or group of actions
     */
    public Action getContextAction(EditorPane editor) {
      if (Browser.getActiveBrowser().getActiveNode() instanceof OTFileNode) {
        menuGroup.updateActions(
          (OTStructure)editor.getDocument().getProperty(OTStructure.class));
        return menuGroup;
      }
      return null;
    }
  }

  /**
   * Create the menu from the standard list of actions.
   *
   * @param  structure  the object that extracts the current state of the node
   */
  public OTPopupMenu(OTStructure structure) {
    super(structure.getEditorPane(), menuGroup);
    this.structure = structure;
  }

  /**
   * Update the contained actions based on the current structure settings.
   */
  public void show(Component invoker, int x, int y) {
    menuGroup.updateActions(structure);
    super.show(invoker, x, y);
  }
}

/**
 * The group of actions for the OpenTools manifest file type.
 */
class OTMenuGroup extends ActionGroup {

  public OTMenuGroup() {
    super("Add category");
    putValue(Action.NAME, "Add category");
    putValue(Action.SMALL_ICON, OTFileNode.ICON);
    setPopup(true);
    for (int index = 0; index < OTScanner.KEYWORDS.length; index++) {
      add(new OTAction(
        OTScanner.KEYWORDS[index] + " (" + OTScanner.KEYWORD_VERSIONS[index] + ")",
        "OpenTools-" + OTScanner.KEYWORDS[index], OTScanner.KEYWORD_CHARS[index]));
    }
    add(new OTUserAction("User defined...", '\0'));
  }

  /**
   * Enable/disable categories based on current entries.
   *
   * @param  structure  the extracted structure of the OpenTools node
   */
  public void updateActions(OTStructure structure) {
    for (int index = 0; index < getActionCount(); index++) {
      OTAction action = (OTAction)getAction(index);
      action.setEnabled(action instanceof OTUserAction ||
        structure == null || !structure.contains(action.getCategory()));
    }
  }
}

/**
 * An action that inserts its category into the editor buffer.
 */
class OTAction extends EditorAction {

  protected String category;

  public OTAction(String name, String category, char mnemonic) {
    super(name);
    this.category = category;
    this.
    putValue(MNEMONIC_KEY, new Character(mnemonic));
  }

  public String getCategory() { return category; }

  /**
   * Add the new category at the end of the file, reposition cursor for
   * immediate entry, and move focus to the editor.
   */
  public void actionPerformed(ActionEvent e) {
    EditorPane target = getEditorTarget(e);
    target.insertAt(target.getDocument().getLength(), category + ": \n");
    target.setCaretPosition(target.getDocument().getLength() - 1);
    target.requestFocus();
  }

  public String toString() { return (String)getValue(Action.NAME); }
}

/**
 * An action that inserts a user-defined category into the editor buffer.
 */
class OTUserAction extends OTAction {

  private static String lastCategory = "";

  public OTUserAction(String name, char mnemonic) {
    super(name, lastCategory, mnemonic);
  }

  /**
   * Ask for the new category and add it at the end of the file, reposition
   * cursor for immediate entry, and move focus to the editor.
   */
  public void actionPerformed(ActionEvent e) {
    JPanel panel = new JPanel();
    JTextField catField = new JTextField(lastCategory);
    catField.setPreferredSize(new Dimension(200, 21));
    catField.selectAll();
    panel.add(new JLabel("OpenTools-"));
    panel.add(catField);
    Object[] message = {"Enter the OpenTools category", panel};
    Object[] options = {"OK", "Cancel"};
    int result = JOptionPane.showOptionDialog(null, message,
      "OpenTools Node", JOptionPane.DEFAULT_OPTION,
      JOptionPane.QUESTION_MESSAGE, null, options, options[0]);
    if (result == 0) {
      lastCategory = catField.getText();
      category = "OpenTools-" + lastCategory;
      super.actionPerformed(e);
    }
  }
}