package wood.keith.opentools.otnode;

import java.io.File;
import java.io.IOException;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import com.borland.jbuilder.node.JBProject;
import com.borland.jbuilder.paths.PathSet;
import com.borland.jbuilder.paths.PathSetManager;
import com.borland.jbuilder.paths.ProjectPathSet;
import com.borland.jbuilder.wizard.archive.ArchiveNode;
import com.borland.jbuilder.wizard.archive.ArchivePropertyGroup;
import com.borland.jbuilder.wizard.archive.DocumentationArchiver;
import com.borland.jbuilder.wizard.archive.ManifestPropertyGroup;
import com.borland.jbuilder.wizard.archive.OpenToolArchiver;
import com.borland.primetime.PrimeTime;
import com.borland.primetime.help.HelpManager;
import com.borland.primetime.help.ZipHelpBook;
import com.borland.primetime.help.ZipHelpTopic;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.DuplicateNodeException;
import com.borland.primetime.node.FileNode;
import com.borland.primetime.node.FolderNode;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.Project;
import com.borland.primetime.personality.Personality;
import com.borland.primetime.runtime.RunConfiguration;
import com.borland.primetime.runtime.RunPropertyGroup;
import com.borland.primetime.util.VetoException;
import com.borland.primetime.vfs.FileFilesystem;
import com.borland.primetime.vfs.InvalidUrlException;
import com.borland.primetime.vfs.Url;
import com.borland.primetime.wizard.BasicWizard;
import com.borland.primetime.wizard.Wizard;
import com.borland.primetime.wizard.WizardAction;
import com.borland.primetime.wizard.WizardHost;
import com.borland.primetime.wizard.WizardManager;
import com.borland.primetime.wizard.WizardPage;

/**
 * This class defines the "OpenTools Project" wizard at the highest level.
 * It sets the dialog pages to be displayed and generates the
 * Wizard classes as defined by user input.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  2.0  19 May 2003
 * @version  2.1  10 December 2003
 * @version  2.2  22 December 2003
 */
public class OTProjectWizard extends BasicWizard {

  public static final String TITLE = "OpenTools Project Wizard";

  /** JBuilder 10 minor version. */
  private static final byte JB10_MINOR = 7;

  // Internal variables
  private OTProjectWizardPage1 _wizardPage1;

  /**
   * Register the "OpenTools Project" tool.
   * Provides the needed OpenTools interface required to register the
   * WizardAction which defines and creates this wizard.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION ||
        minorVersion < JB10_MINOR) {
      return;
    }
    WizardManager.registerWizardAction(WIZARD_OpenTools_Project);
  }

  /**
   * Definition of "OpenTools Project" WizardAction.
   * This is an OpenTools registered action defining where the wizard will
   * appear in the IDE and a factory for creating each instance of this wizard.
   */
  public static final WizardAction WIZARD_OpenTools_Project = new WizardAction(
      "OpenTools Project", '\0', "Create an OpenTools project",
      null, new ImageIcon(ClassLoader.getSystemResource(
      OTProjectWizard.class.getName().replace('.', '/') + ".gif")),
      true, "Project"){

    protected Wizard createWizard() {
      return new OTProjectWizard();
    }

    /**
     * This action only applies to the OpenTools personality.
     *
     * @return  the list of applicable personalities
     */
    public Personality[] getPersonalities() {
      return OTPersonality.PERSONALITIES;
    }
  };

  /**
   * Wizard startup.
   * "OpenTools Project" wizard is to become visible,
   * create and order the wizard pages to be displayed.
   *
   * @param  host  the WizardHost that owns this wizard instance
   * @return  the initial WizardPage to show
   */
  public WizardPage invokeWizard(WizardHost host) {
    setWizardTitle(TITLE);
    Project project = host.getBrowser().getProjectView().getActiveProject();
    _wizardPage1 = new OTProjectWizardPage1(project);
    addWizardPage(_wizardPage1);
    return super.invokeWizard(host);
  }

  /**
   * Creates a new project from the given parameters.
   *
   * @param  directory    the directory in which the project is created
   * @param  projectName  the name of the project to be created
   * @param  type         the storage format for the new project (jpx or jpr)
   * @return  the project that was created
   */
  protected static JBProject createProject(String directory, String projectName,
      String type) throws DuplicateNodeException, IOException {
    return (JBProject)Project.getProject(
        new Url(new File(directory + "/" + projectName + "." + type)));
  }

  /**
   * Creates/replaces a file node.
   *
   * @param  project   the JBProject in which the node is created
   * @param  fileName  the name of the file to be created
   * @return  the FileNode that was created
   */
  protected static FileNode createFile(JBProject project, String fileName)
      throws InvalidUrlException, DuplicateNodeException {
    // getNode() will create a node if does not exist
    FileNode newNode =
      project.getNode(new Url(FileFilesystem.PROTOCOL, fileName));
    newNode.setParent(project);
    return newNode;
  }

  /** Perform the wizard's task based on user input. */
  protected void finish() throws VetoException {
    Browser browser = wizardHost.getBrowser();
    JBProject project = (JBProject)browser.getProjectView().getActiveProject();
    FileNode fileNode = null;
    try {
      // Create the project
      project = createProject(_wizardPage1.getDirectory(),
        _wizardPage1.getName(), _wizardPage1.getType());
      // And set its paths
      ProjectPathSet pathSet = project.getPaths();
      // Backup
      pathSet.setBakPath(
        new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
        File.separator + JBProject.BACKUP_DIRECTORY, ""));
      // Documentation
      if (_wizardPage1.isHelpIncluded()) {
        pathSet.setDocPath(new Url[] {
          new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
          File.separator + JBProject.DOC_DIRECTORY, ""),
          new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
          File.separator + "doco", "")});
      }
      else {
        pathSet.setDocPath(new Url[] {
          new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
          File.separator + JBProject.DOC_DIRECTORY, "")});
      }
      // Output
      pathSet.setOutPath(
        new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
        File.separator + JBProject.CLASSES_DIRECTORY, ""));
      // Source
      Url defaultSource =
        new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
        File.separator + JBProject.SOURCE_DIRECTORY, "");
      if (_wizardPage1.isHelpIncluded()) {
        pathSet.setSourcePath(new Url[] {defaultSource,
          new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
          File.separator + "doco", "")});
      }
      else {
        pathSet.setSourcePath(new Url[] {defaultSource});
      }
      pathSet.setDefaultSourcePath(defaultSource);
      // Testing
      pathSet.setTestPath(
        new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory() +
        File.separator + JBProject.TEST_DIRECTORY, ""));
      // Working
      pathSet.setWorkingDirectory(
        new Url(FileFilesystem.FILESYSTEM, "", _wizardPage1.getDirectory(), ""));
      // Required libraries
      pathSet.setRequired(
        new PathSet[] {PathSetManager.getLibrary("Open Tools SDK")});
      // Add a runtime configuration
      RunConfiguration config = new RunConfiguration(project, 0);
      config.readProperties();
      config.setName(_wizardPage1.getName());
      boolean runtimeSet = false;
      RunConfiguration.RUNNABLE_TYPE.setValue(config.getMap(),
        "com.borland.jbuilder.runtime.OpenToolRunner");
      runtimeSet = true;
      config.writeProperties();
      if (runtimeSet) {
        RunPropertyGroup.DEFAULT_CONFIGURATION.setValue(project, null);
        RunPropertyGroup.EXCLUDE_DEFAULT_FOR_ZERO.setBoolean(project, false);
      }
      // Add the archive node
      Node node = new ArchiveNode(project, project, "OpenTool");
      ArchivePropertyGroup.ARCHIVER_CLASS.setValue(
        node, OpenToolArchiver.class.getName());
      ManifestPropertyGroup.MANIFEST_OVERRIDE_PATH.setValue(
        node, "classes.opentools");
      ArchivePropertyGroup.TARGET_PATH.setValue(
        node, _wizardPage1.getName() + ".jar");
      if (_wizardPage1.isHelpIncluded()) {
        // Add the documentation node
        node = new ArchiveNode(project, project, "Documentation");
        ArchivePropertyGroup.ARCHIVER_CLASS.setValue(
          node, DocumentationArchiver.class.getName());
        ArchivePropertyGroup.TARGET_PATH.setValue(
          node, _wizardPage1.getName() + "Doc.jar");
        // Add the documentation folder
        node = new FolderNode(project, project, "Doco");
      }
      // Add the .opentools file
      fileNode = createFile(project,
        _wizardPage1.getDirectory() +	File.separator + "classes.opentools");
    }
    catch (Exception e) {
      JOptionPane.showMessageDialog(wizardHost.getBrowser(),
        "Error during generation\n" +
        (e.getMessage() == null ? e.getClass().getName() : e.getMessage()),
        this.getWizardTitle(), JOptionPane.ERROR_MESSAGE);
      throw new VetoException();
    }
    // Cause the generated/updated file to be opened in the Content Pane.
    try {
      browser.setActiveNode(fileNode, true);
    }
    catch (Exception e) {
      // Ignore
    }
  }

  /**
   * Wizard termination.
   * The Wizard has been completed (finished or cancelled) and any resource
   * cleanup to help the garbage collector can occur here.
   */
  public void wizardCompleted() {}

  /**
   * Provide help for the wizard.
   *
   * @param  page  the current page in the wizard
   * @param  host  the wizard host
   */
  public void help(WizardPage page, WizardHost host) {
    HelpManager.showHelp(new ZipHelpTopic(
      new ZipHelpBook("OpenToolsNodeDoc.jar"),
      this.getClass().getName().replace('.', '/') + ".html"),
      page.getPageComponent(host));
  }
}
