package wood.keith.opentools.otnode;

import java.util.Arrays;

import com.borland.primetime.editor.AbstractScanner;

/**
 * This class provides the OpenTools-specific text scanner.  It identifies
 * the various code elements that make up an OpenTools manifest so that color
 * coding may be done.  This version just identifies categories.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  12 September 2001
 * @version  1.1  19 February 2002
 */
public class OTScanner extends AbstractScanner {

  /** The list of standard OpenTools categories (in sorted order). */
  public static final String[] KEYWORDS =
    {"Archiver", "Build", "ClassFilter", "Core", "Debugger", "Designer",
    "Doclet", "Editor", "Ejb", "EjbImport", "EjbModeler", "ErrorInsight",
    "Info", "JavaCompiler", "JDKType", "LibKit", "Node", "NodeChooser",
    "PathSet", "PostWizard", "PreUI", "ServerServices", "Servers", "Sqlj",
    "TestRunner", "UI", "UrlChooser", "VFS", "Web", "WebServicesToolkit",
    "Wizard"};
  /** Versions where the above keywords were introduced. */
  public static final String[] KEYWORD_VERSIONS =
    {"4", "4", "8", "4", "4", "4",
    "6", "4", "5", "7", "8", "9",
    "4", "9", "5", "8", "4", "4",
    "8", "7", "4", "7", "7", "7",
    "6", "4", "4", "4", "5", "8",
    "4"};
  /** Accelerator characters for the keywords. */
  public static final char[] KEYWORD_CHARS =
   {'A', 'B', 'F', 'C', 'D', 'G',
    'L', 'E', 'J', 'M',  0 , '9',
    'I',  0 , 'Y', 'K', 'N', 'H',
     0 , 'O', 'P', '7', 'S', 'Q',
    'T', 'U', 'R', 'V', 'W', '8',
    'Z'};

  /**
   * Check to see if the buffer pointer is on a string character.
   * If so, advance the buffer point past the entire string sequence.
   *
   * @return  the defined STRING constant or -1
   */
  protected int checkString(int initialState) {
    return -1;  // No strings in OpenTools manifest
  }

  /**
   * Check to see if the buffer pointer is on a comment character.
   * If so, advance the buffer point past the entire comment.
   *
   * @return  the defined COMMENT constant or -1
   */
  protected int checkComment(int initialState) {
    return -1;  // No comments in OpenTools manifest
  }

  /**
   * The default implementation of checkIdentifier will call this method
   * to determine whether or not a particular character can be the start of
   * an identifier.
   *
   * @param  ch  the char to be checked
   * @return  true if ch can be the start of an identifier
   * @see  AbstractScanner#checkIdentifier
   */
  protected boolean isValidIdentifierStart(char ch) {
    switch (ch) {
      case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G':
      case 'H': case 'I': case 'J': case 'K': case 'L': case 'M': case 'N':
      case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U':
      case 'V': case 'W': case 'X': case 'Y': case 'Z':
      case 'a': case 'b': case 'c': case 'd': case 'e': case 'f': case 'g':
      case 'h': case 'i': case 'j': case 'k': case 'l': case 'm': case 'n':
      case 'o': case 'p': case 'q': case 'r': case 's': case 't': case 'u':
      case 'v': case 'w': case 'x': case 'y': case 'z':
        return true;
      default:
        return false;
    }
  }

  /**
   * The default implementation of checkIdentifier will call this method
   * to determine whether or not a particular character can be part of
   * an identifier.
   *
   * @param  ch  the char to be checked
   * @return  true if ch can be part of an identifier
   * @see  AbstractScanner#checkIdentifier
   */
  protected boolean isValidIdentifierPart(char ch) {
    switch (ch) {
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
      case '.': case '_': case '$': case '-':
        return true;
      default:
        return isValidIdentifierStart(ch);
    }
  }

  /**
   * The default implementation of checkSymbol will call this method
   * to determine whether or not a particular character is a valid symbol.
   *
   * @param  ch  the char to be checked
   * @return  true if ch is a valid symbol
   * @see  AbstractScanner#checkSymbol
   */
  protected boolean isSymbol(char ch) {
    switch (ch) {
      case ':':
        return true;
      default:
        return false;
    }
  }

  /**
   * The default implementation of checkIdentifier will call this method to
   * determine whether or not a particular string is a keyword.  This method
   * should take into account whether or not the language is case-sensitive.
   *
   * @param  value  the string to be checked
   * @return  true if value is a keyword
   * @see  checkIdentifier
   */
  protected boolean isKeyword(String value) {
    return isExtendedKeyword(value) &&
      (Arrays.binarySearch(KEYWORDS, value.substring(10)) > -1);
  }

  /**
   * The default implementation of checkIdentifier will call this method
   * to determine whether or not a particular string is an extended keyword.
   * This method should take into account whether or not the language is
   * case-sensitive.
   *
   * @param  value  the string to be checked
   * @return  true if value is an extended keyword
   * @see  checkIdentifier
   */
  protected boolean isExtendedKeyword(String value) {
    return (value.length() > 10 && value.substring(0, 10).equals("OpenTools-"));
  }
}
