package wood.keith.opentools.otnode;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.StringTokenizer;
import java.util.Vector;
import javax.swing.Icon;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.text.Document;
import javax.swing.tree.DefaultMutableTreeNode;

import com.borland.primetime.actions.ActionGroup;
import com.borland.primetime.node.TextStructure;
import com.borland.primetime.ui.MergeTreeNode;

/**
 * This class provides the OpenTools-specific structure component.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  12 September 2001
 * @version  1.1  19 February 2002
 */
public class OTStructure extends TextStructure {

  private JPopupMenu popup = null;
  private ArrayList categories = new ArrayList(OTScanner.KEYWORDS.length);

  public OTStructure() {
    OTStructureNodeUserObject uo = new OTStructureNodeUserObject(0, 0, "");
    treeModel.setRoot(new OTStructureNode(uo));
  }

  public JPopupMenu getPopup() {
    if (popup == null) {
      popup = new OTPopupMenu(this);
    }
    return popup;
  }

  /**
   * Supply an action group for a popup menu.
   * Used in JBuilder 8+ only!!
   *
   * @return  the popup action group
   */
  public ActionGroup getPopupActionGroup() {
    OTMenuGroup actions = ((OTPopupMenu)getPopup()).menuGroup;
    actions.updateActions(this);
    return actions;
  }

  public boolean contains(String category) {
    return categories.contains(category);
  }

  public void nodeSelected(DefaultMutableTreeNode node) {
    nodeAction(node, false);
  }

  public void nodeActivated(DefaultMutableTreeNode node) {
    nodeAction(node, true);
  }

  /**
   * Highlight the line associated with the selected node
   * and optionally move to it.
   *
   * @param  node   the selected node
   * @param  focus  true to move the focus to the line, false to just highlight
   */
  private void nodeAction(DefaultMutableTreeNode node, boolean focus) {
    if (!(node instanceof OTStructureNode)) {
      return;
    }
    OTStructureNodeUserObject uo =
      (OTStructureNodeUserObject)node.getUserObject();
    if (uo.lineNo >= 0) {
      setCaretPosition(uo.lineNo, uo.column, focus);
    }
  }

  /**
   * Gets appropriate icon for a particular tree node.
   * This routine is called when the icon needs to be rendered for a
   * particular tree node.
   *
   * @param  value  the node needing an icon
   * @return  the Icon appropriate for this node
   */
  public Icon getStructureIcon(Object value) {
    if (!(value instanceof OTStructureNode)) {
      return null;
    }

    OTStructureNode structureNode = (OTStructureNode)value;
    switch (structureNode.getType()) {
      case OTStructureNode.CATEGORY:
        return OTFileNode.ICON;
      default:
        return null;
    }
  }

  /**
   * Differentiate between standard and non-standard categories.
   *
   * @param  value        the current node
   * @param  baseFGColor  the base foreground colour
   * @param  baseBGColor  the base background color
   * @return  the foreground colour for this node
   */
  public Color getStructureForegroundColor(Object value, Color baseFGColor,
      Color baseBGColor) {
    String category = value.toString();
    if (category.startsWith("OpenTools-")) {
      for (int index = 0; index < OTScanner.KEYWORDS.length; index++) {
        if (category.endsWith(OTScanner.KEYWORDS[index])) {
          return baseFGColor;
        }
      }
    }
    return baseFGColor.equals(Color.black) ? Color.blue : Color.black;
  }

  /**
   * Update the Structure Pane when file content changed.
   *
   * This routine is called when the .opentools file content has been changed
   * and the user defined parse delay has expired.  This version that just
   * finds all the categories and presents them sorted to the user.
   *
   * @param  doc  the document containing the content
   */
  public void updateStructure(Document doc) {
    // Save this structure object for later use by popup menu
    doc.putProperty(OTStructure.class, this);
    categories.clear();
    final OTStructureNode newRoot = new OTStructureNode(
      new OTStructureNodeUserObject(0, 0, "root"));
    try {
      int lineNo = 1;
      String text = new String(fileNode.getBuffer().getContent());
      StringTokenizer st = new StringTokenizer(text, "\n", true);
      while (st.hasMoreTokens()) {
        // Scan lines for those starting with a manifest category -
        // first character is not a space and contains a colon
        String line = (String)st.nextElement();
        if (line.equals("\n")) {
          lineNo++;
        }
        else {
          int pos;
          if (line.charAt(0) != ' ' && (pos = line.indexOf(':')) > -1) {
            String name = line.substring(0, pos);
            newRoot.add(new OTStructureNode(
              new OTStructureNodeUserObject(lineNo, pos + 2, name)));
            categories.add(name);
          }
        }
      }
    }
    catch (java.io.IOException ex) {
    }

    // Add the new nodes in the main thread
    Runnable update = new Runnable() {
      public void run() {
        OTStructureNode root = (OTStructureNode)treeModel.getRoot();
        root.removeAllChildren();
        root.mergeChildren(newRoot);
        root.sortChildren();
        treeModel.nodeStructureChanged(root);
      }
    };

    if (SwingUtilities.isEventDispatchThread()) {
      update.run();
    }
    else {
      SwingUtilities.invokeLater(update);
    }
  }

  /**
   * This class provides a wrapper that records where in the source file
   * a particular OpenTools category is defined.
   */
  protected class OTStructureNodeUserObject {
    public int lineNo;
    public int column;
    public String name;

    public OTStructureNodeUserObject(
        int lineNo, int column, String name) {
      this.column = column;
      this.lineNo = lineNo;
      this.name = name;
    }

    public String toString() { return name; }
  }

  /**
   * This class provides an implementation of a TreeNode element
   * which represents a particular OpenTools category.
   */
  protected class OTStructureNode extends MergeTreeNode {
    public static final int CATEGORY = 0;

    private int type;

    public OTStructureNode(Object userObject) {
      this(userObject, CATEGORY);
    }

    public OTStructureNode(Object userObject, int type) {
      super(userObject);
      this.type = type;
    }

    public void setType(int value) { type = value; }

    public int getType() { return type; }

    public void sortChildren() {
      MergeTreeNode[] array = getChildrenArray();
      if (array == null) {
        return;
      }
      Arrays.sort(array, new Comparator() {
        public int compare(Object o1, Object o2) {
          return o1.toString().compareToIgnoreCase(o2.toString());
        }
      });
      children = new Vector(Arrays.asList(array));
    }
  }
}
