package wood.keith.opentools.tabprops;

import java.awt.event.ActionEvent;
import javax.swing.Action;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.actions.ActionGroup;
import com.borland.primetime.actions.StateAction;
import com.borland.primetime.actions.UpdateAction;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.BrowserPropertyGroup;
import com.borland.primetime.ide.ContentManager;
import com.borland.primetime.ide.ContextActionProvider;
import com.borland.primetime.node.Node;

/**
 * Add menu items to the content pane tabs pop-up to reconfigure them.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  3 August 2001
 * @version  1.1  30 August 2001
 * @version  1.2  31 March 2002
 * @version  1.3  30 September 2002
 */
public class TabProps implements ContextActionProvider {

  private static final String VERSION = "1.3";
  /* JBuilder 6 minor version */
  private static final byte JB6VERS = 3;
  /* JBuilder 8 minor version */
  private static final byte JB8VERS = 5;

  /**
   * Register the "TabProps" tool.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    ContentManager.registerContextActionProvider(new TabProps(minorVersion));
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded TabProps tool v" + VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  /**
   * The main group that contains all the other items.
   */
  public final static ActionGroup GROUP_TabProps = new ActionGroup("TabProps");

  /**
   * An action to alter the orientation of the tabs in the content manager.
   */
  public final static Action ACTION_Orientation =
      new UpdateAction("Toggle tab orientation", '\0',
      "Swap content pane tab orientation between horizontal and vertical") {

    /**
     * Invoke the action - toggle the tab orientation in the content manager.
     */
    public void actionPerformed(ActionEvent e) {
      BrowserPropertyGroup.PROP_TABS_PLACEMENT.setInteger(
          BrowserPropertyGroup.PROP_TABS_PLACEMENT.getInteger() ==
          BrowserPropertyGroup.TABS_HORIZONTAL ?
          BrowserPropertyGroup.TABS_VERTICAL :
          BrowserPropertyGroup.TABS_HORIZONTAL);
      BrowserPropertyGroup.updateContentManager();
    }
  };

  /**
   * The group for changing the label option for the tabs.
   */
  public final static ActionGroup GROUP_Labels =
      new ActionGroup("Set label type");

  /*
   * Set the contents of the tabs.
   */
  public final static Action ACTION_LabelFile = new LabelAction(
      "File name", BrowserPropertyGroup.TABS_LABEL_FILENAME);
  public final static Action ACTION_LabelShort = new LabelAction(
      "Short file name", BrowserPropertyGroup.TABS_LABEL_SHORTNAME);
  public final static Action ACTION_LabelMixed = new LabelAction(
      "Icon and short file name", BrowserPropertyGroup.TABS_LABEL_SHORTNAME_ICON);
  public final static Action ACTION_LabelIcon = new LabelAction(
      "Icon only", BrowserPropertyGroup.TABS_LABEL_ICON_ONLY);

  /**
   * The group for changing the insertion option for the tabs.
   */
  public final static ActionGroup GROUP_Insert =
      new ActionGroup("Set insertion position");

  /*
   * Set the order of the tabs.
   */
  public final static Action ACTION_InsertAlpha = new InsertAction(
      "In alphabetic order", BrowserPropertyGroup.TABS_INSERT_SORT_ALPHABETIC);
  public final static Action ACTION_InsertDisplay = new InsertAction(
      "In display order", BrowserPropertyGroup.TABS_INSERT_SORT_DISPLAY);
  public final static Action ACTION_InsertStart = new InsertAction(
      "At the start", BrowserPropertyGroup.TABS_INSERT_AT_BEGINNING);
  public final static Action ACTION_InsertLeft = new InsertAction(
      "To the left", BrowserPropertyGroup.TABS_INSERT_TO_LEFT);
  public final static Action ACTION_InsertRight = new InsertAction(
      "To the right", BrowserPropertyGroup.TABS_INSERT_TO_RIGHT);
  public final static Action ACTION_InsertEnd = new InsertAction(
      "At the end", BrowserPropertyGroup.TABS_INSERT_AT_END);

  /**
   * Set tab layout policy
   */
  public final static Action ACTION_MultiLines = new LayoutAction();

  /**
   * Create the necessary menu items.
   */
  static {
    GROUP_TabProps.add(ACTION_Orientation);
    GROUP_TabProps.add(GROUP_Labels);
    GROUP_Labels.setPopup(true);
    GROUP_Labels.add(ACTION_LabelFile);
    GROUP_Labels.add(ACTION_LabelShort);
    GROUP_Labels.add(ACTION_LabelMixed);
    GROUP_Labels.add(ACTION_LabelIcon);
    GROUP_TabProps.add(GROUP_Insert);
    GROUP_Insert.setPopup(true);
    GROUP_Insert.add(ACTION_InsertAlpha);
    GROUP_Insert.add(ACTION_InsertDisplay);
    GROUP_Insert.add(ACTION_InsertStart);
    GROUP_Insert.add(ACTION_InsertLeft);
    GROUP_Insert.add(ACTION_InsertRight);
    GROUP_Insert.add(ACTION_InsertEnd);
    GROUP_TabProps.add(ACTION_MultiLines);
  }

  /**
   * Update menu items based on OT version.
   */
  public TabProps(byte minorVersion) {
    if (minorVersion < JB6VERS) {  // Before JBuilder 6
      GROUP_TabProps.remove(GROUP_Insert);
    }
    if (minorVersion < JB8VERS) {  // Before JBuilder 8
      GROUP_TabProps.remove(ACTION_MultiLines);
    }
  }

  /**
   * Return the group of menu items from this tool.
   *
   * @param  browser  the browser hosting this action
   * @param  nodes    the list of currently open nodes
   * @return  the new action
   */
  public Action getContextAction(Browser browser, Node[] nodes) {
    return GROUP_TabProps;
  }
}

/**
 * An action for altering the label option on the tabs in the content manager.
 */
class LabelAction extends StateAction {
  private int setting;

  /**
   * Create an action for a particular label option.
   *
   * @param  name     the name to display for this action
   * @param  setting  the label option value
   */
  public LabelAction(String name, int setting) {
    super(name);
    setGrouped(true);
    this.setting = setting;
  }

  /**
   * Retrieve the current state by comparing the global setting
   * with the internal value.
   *
   * @param  object  the sending object
   */
  public boolean getState(Object object) {
    return (BrowserPropertyGroup.PROP_TABS_LABEL.getInteger() == setting);
  }

  /**
   * If selected, update the global setting from the internal value
   * and redisplay.
   *
   * @param  object  the sending object
   * @param  value   the new value for this action
   */
  public void setState(Object object, boolean value) {
    if (value) {
      BrowserPropertyGroup.PROP_TABS_LABEL.setInteger(setting);
      BrowserPropertyGroup.updateContentManager();
    }
  }
}

/**
 * An action for altering the insertion option on the tabs in the content manager.
 */
class InsertAction extends StateAction {
  private int setting;

  /**
   * Create an action for a particular insertion option.
   *
   * @param  name     the name to display for this action
   * @param  setting  the label option value
   */
  public InsertAction(String name, int setting) {
    super(name);
    setGrouped(true);
    this.setting = setting;
  }

  /**
   * Retrieve the current state by comparing the global setting
   * with the internal value.
   *
   * @param  object  the sending object
   */
  public boolean getState(Object object) {
    return (BrowserPropertyGroup.PROP_INSERT_TAB_RULE.getInteger() == setting);
  }

  /**
   * If selected, update the global setting from the internal value
   * and redisplay.
   *
   * @param  object  the sending object
   * @param  value   the new value for this action
   */
  public void setState(Object object, boolean value) {
    if (value) {
      BrowserPropertyGroup.PROP_INSERT_TAB_RULE.setInteger(setting);
      BrowserPropertyGroup.updateContentManager();
    }
  }
}

/**
 * An action for altering the layout policy on the tabs in the content manager.
 */
class LayoutAction extends StateAction {

  /**
   * Create an action for the layout option.
   */
  public LayoutAction() {
    super("Multiple rows of tabs");
  }

  /**
   * Retrieve the current state by comparing the global setting
   * with the internal value.
   *
   * @param  object  the sending object
   */
  public boolean getState(Object object) {
    return (BrowserPropertyGroup.PROP_TAB_LAYOUT_POLICY.getInteger() ==
        BrowserPropertyGroup.TABS_WRAP_LAYOUT);
  }

  /**
   * If selected, update the global setting from the internal value
   * and redisplay.
   *
   * @param  object  the sending object
   * @param  value   the new value for this action
   */
  public void setState(Object object, boolean value) {
    BrowserPropertyGroup.PROP_TAB_LAYOUT_POLICY.setInteger(value ?
        BrowserPropertyGroup.TABS_WRAP_LAYOUT :
        BrowserPropertyGroup.TABS_SCROLL_LAYOUT);
    BrowserPropertyGroup.updateContentManager();
  }
}
