package wood.keith.opentools.xmltools;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.editor.EditorManager;
import com.borland.primetime.ide.ProjectView;

/**
 * Menu extensions for opening an XML document's DTD in JBuilder.
 *
 * There are several local context menus, and this class will
 * add entries to the local menu in the Project View, and also
 * to the local menu in the "source" window in the Content pane.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  16 October 2000
 * @version  2.0  15 February 2002
 */
public class XMLOpenDTDMenu extends XMLOpenFileMenu {

  /**
   * When the OpenTool add-ins are loaded, two ContextActionProviders
   * are registered, one for the Project Pane and one for the EditorPane.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    XMLOpenDTDMenu openDTD = new XMLOpenDTDMenu();
    ProjectView.registerContextActionProvider(openDTD);
    EditorManager.registerContextActionProvider(openDTD);
  }

  /**
   * Extract the name of the related file from this line of the
   * original document.
   * Here, find the DTD name from the DOCTYPE declaration.
   *
   * @param  line  the line from the original document
   * @return  the name of the related file, or null if not found
   */
  protected String getFilenameFromLine(String line) {
    int pos;
    int endPos;
    try {
      if ((pos = line.indexOf("<!DOCTYPE")) > -1) {
        // Look for the DTD external reference (assumed to be on the same line)
        return extractField(line, line.indexOf("SYSTEM", pos) + 6);
      }
      else if ((pos = line.indexOf(":schemaLocation")) > -1) {
        // Look for the schema external reference
        return extractField(line, pos + 1);
      }
      else if ((pos = line.indexOf(":noNamespaceSchemaLocation")) > -1) {
        // Look for the schema external reference
        return extractField(line, pos + 1);
      }
    }
    catch (Exception ex) {
      ex.printStackTrace();
    }
    return null;
  }

  /**
   * Extract a quoted field from the line starting at character pos.
   * The quotes may be either single or double.
   *
   * @param  line  the line of text to search
   * @param  pos   the starting position in the line
   * @return  the field contents
   */
  private String extractField(String line, int pos) {
    char quote = '"';
    int startPos = 0;
    for (int index = pos; index < line.length(); index++) {
      if (line.charAt(index) == '"' || line.charAt(index) == '\'') {
        startPos = index + 1;
        quote = line.charAt(index);
        break;
      }
    }
    if (startPos == 0) {
      return null;
    }
    for (int index = startPos; index < line.length(); index++) {
      if (line.charAt(index) == quote) {
        return line.substring(startPos, index);
      }
    }
    return line.substring(startPos);
  }

  /**
   * Retrieve a short description of the related file type.
   * Here, identify the DTD.
   *
   * @return  the type of file to be opened
   */
  protected String getFileType() { return "DTD/Schema"; }

  /**
   * Retrieve the file name for an icon for this menu entry.
   * Here, return the DTD's icon.
   *
   * @return  the name of the file containing the image
   */
  protected String getIconFilename() {
    return "wood/keith/opentools/xmltools/XMLDTD16.gif";
  }
}
