package wood.keith.opentools.xmltools;

import java.awt.event.ActionEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.io.IOException;
import java.text.MessageFormat;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import com.borland.primetime.actions.UpdateAction;
import com.borland.primetime.editor.EditorContextActionProvider;
import com.borland.primetime.editor.EditorPane;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.BrowserAction;
import com.borland.primetime.ide.ContextActionProvider;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.TextFileNode;
import com.borland.primetime.vfs.Url;

/**
 * Menu extensions for opening an XML document's related file in JBuilder.
 * The specific file is extracted from the current document.
 * This abstract class is subclassed for particular associated files.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  16 October 2000
 * @version  2.0  15 February 2002
 */
public abstract class XMLOpenFileMenu
    implements ContextActionProvider, EditorContextActionProvider {

  /** The menu caption and description. */
  private static final String CAPTION = "Open {0}";
  private static final String LONG_DESC = "Open the {0} for this document";

  /* Messages. */
  private static final String SAVE_SOURCE =
    "Please save the file before validating";
  private static final String OPEN_FAIL = "Couldn't open {0} {1}\n{2}";
  private static final String OPENING = "Opening {0} {1}";

  /** Initialisation. */
  public XMLOpenFileMenu() {
    XMLToolsMenuGroup.getMenuGroup().add(ACTION_EDITOR_OPEN_FILE);
    try {
      // Load icon for the menu items
      Icon icon =
        new ImageIcon(ClassLoader.getSystemResource(getIconFilename()));
      ((UpdateAction)ACTION_PROJECT_OPEN_FILE).setSmallIcon(icon);
      ((UpdateAction)ACTION_EDITOR_OPEN_FILE).setSmallIcon(icon);
    }
    catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  /**
   * Execute the action - opening the XML document's related file.
   *
   * @param  browser  the browser for the current document
   * @param  node     the active file node
   */
  private void doOpenAction(Browser browser, Node node) {
    // We only want XML files
    if (XMLTools.isXmlNode(node)) {
      String relatedFilename = null;
      try {
        relatedFilename = getRelatedFilename((TextFileNode)node);
        if (relatedFilename != null) {
          browser.doOpen(new Url(new File(relatedFilename)),
            browser.getActiveProject(), false);
        }
      }
      catch (Exception ex) {
        JOptionPane.showMessageDialog(browser,
          MessageFormat.format(OPEN_FAIL,
          new Object[] {getFileType(), relatedFilename, ex.getMessage()}),
          CAPTION, JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /**
   * Extract the related file name from the XML document.
   *
   * @param  node  the node representing the XML document
   * @return  the name of its related file, or null if not found
   */
  private String getRelatedFilename(TextFileNode node) throws IOException {
    // Find the related file name
    BufferedReader reader = new BufferedReader(
      new InputStreamReader(node.getBuffer().getInputStream()));
    String line;
    int pos;
    int endPos;
    String relatedFilename = null;
    while ((line = reader.readLine()) != null) {
      // Find the related file declaration
      if ((relatedFilename = getFilenameFromLine(line)) != null) {
        // Adjust for relative references
        if (relatedFilename.indexOf(File.separatorChar) == -1) {
          relatedFilename = node.getUrl().getParent().getFile() +
            File.separatorChar + relatedFilename;
        }
        break;
      }
    }
    return relatedFilename;
  }

  /**
   * Extract the name of the related file from this line of the
   * original document.
   * This method must be overridden in subclasses to get the actual name.
   *
   * @param  line  the line from the original document
   * @return  the name of the related file, or null if not found
   */
  protected abstract String getFilenameFromLine(String line);

  /**
   * Retrieve a short description of the related file type.
   * This method must be overridden in subclasses to get the actual description.
   *
   * @return  the type of file to be opened
   */
  protected abstract String getFileType();

  /**
   * Retrieve the file name for an icon for this menu entry.
   * This method should be overridden in subclasses to get the actual filename.
   *
   * @return  the name of the file containing the image
   */
  protected String getIconFilename() { return ""; }

  /**
   * The action that will get added to the Project View local menu. The
   * actionPerformed function is triggered by selecting the item from the
   * local menu after a right click on an XML file node in the Project View.
   */
  public final Action ACTION_PROJECT_OPEN_FILE = new BrowserAction(
      MessageFormat.format(CAPTION, new Object[] {getFileType()}), '\0',
      MessageFormat.format(LONG_DESC, new Object[] {getFileType()})) {

    /**
     * Dis/enable the action - is the related file available.
     * Also show the extracted name of the related file.
     *
     * @param  browser  the currently active browser
     */
    public void update(Browser browser) {
      String filename = null;
      try {
        filename = getRelatedFilename(
          (TextFileNode)browser.getProjectView().getSelectedNode());
      }
      catch (IOException ex) {
        // Ignore
      }
      if (filename != null) {
        setLongText(MessageFormat.format(OPENING,
          new Object[] {getFileType(), filename}));
      }
      setEnabled(filename != null);
    }

    /**
     * Invoke the action - open the XML document's related file.
     *
     * @param  browser  the currently active browser
     */
    public void actionPerformed(Browser browser) {
      doOpenAction(browser, browser.getProjectView().getSelectedNode());
    }
  };

  /**
   * If the node is a single node that belongs to an XML file, the
   * appropriate action is returned that is able to deal with an XML file.
   * Part of the ContextActionProvider interface.
   *
   * @param  browser  the current browser
   * @param  nodes    the set of nodes that are currently selected
   * @return  an action to open the XML document's related file
   */
  public Action getContextAction(Browser browser, Node[] nodes) {
    if (nodes.length == 1 && XMLTools.isXmlNode(nodes[0])) {
      return ACTION_PROJECT_OPEN_FILE;
    }
    return null;
  }

  /**
   * Called when the editor right click context menu is about to appear.
   * Add the open file menu entry in the XMLTools menu group and try to
   * make the entry appear as the bottom entry of the local menu.
   * Part of the EditorContentActionProvider interface.
   *
   * @param  editor  the editor for the current source file
   * @return  an action to open the XML document's related file
   */
  public Action getContextAction(EditorPane editor) {
    // Only applies if an XML file
    if (XMLTools.isXmlNode(Browser.getActiveBrowser().getActiveNode())) {
      return XMLToolsMenuGroup.getMenuGroup();
    }
    return null;
  }

  /**
   * Make it show very low on the local context menu.
   * Priorities range from 1 - 100.  Less than 5 will probably make the menu
   * entry end up at the bottom, while 99 will make it appear at the top.
   * Priorities are shared with other menu entries, so no priority
   * will guarantee a specific place.
   * Part of the EditorContentActionProvider interface.
   *
   * @return  the priority for positioning on the menu
   */
  public int getPriority() { return 4; }

  /**
   * The action that will get added to the editor local menu.
   * The action performed function will be called when the
   * "Open File" menu entry is selected after a right
   * mouse click in an editor pane on the "source" tab.
   */
  public final Action ACTION_EDITOR_OPEN_FILE = new UpdateAction(
      MessageFormat.format(CAPTION, new Object[] {getFileType()}), '\0',
      MessageFormat.format(LONG_DESC, new Object[] {getFileType()})) {

    /**
     * Dis/enable the action - is the related file available.
     * Also show the extracted name of the related file.
     *
     * @param  source  not used - source of the action
     */
    public void update(Object source) {
      String filename = null;
      try {
        filename = getRelatedFilename(
          (TextFileNode)Browser.getActiveBrowser().getActiveNode());
      }
      catch (IOException ex) {
        // Ignore
      }
      if (filename != null) {
        setLongText(MessageFormat.format(OPENING, new Object[] {filename}));
      }
      setEnabled(filename != null);
    }

    /**
     * Perform the requested action - open the related file.
     */
    public void actionPerformed(ActionEvent e) {
      Browser browser = Browser.getActiveBrowser();
      doOpenAction(browser, browser.getActiveNode());
    }
  };
}
