package wood.keith.opentools.xmltools;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.node.Node;
import com.borland.primetime.properties.GlobalArrayProperty;
import com.borland.primetime.properties.GlobalArrayPropertyDefault;
import com.borland.primetime.properties.GlobalIntegerProperty;
import com.borland.primetime.properties.PropertyGroup;
import com.borland.primetime.properties.PropertyManager;
import com.borland.primetime.properties.PropertyPage;
import com.borland.primetime.properties.PropertyPageFactory;

/**
 * This class contains the OpenTools interface and a factory for the
 * XMLToolsPropertyGroup.
 *
 * Provides the needed OpenTools interface required to register the
 * PropertyGroup which retains settings used by the "XML Tools" package.
 * It also creates a separate UI that may be used to edit these settings
 * based on context.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  16 October 2000
 * @version  2.0  15 February 2002
 * @version  2.1  28 November 2003
 */
public class XMLToolsPropertyGroup implements PropertyGroup {

  public static final String PAGE_NAME = "XML Tools";
  public static final String CATEGORY  = "wood.keith.xmltools";
  public static final String DEFAULT_VALIDATOR =
      "wood.keith.opentools.xmltools.XMLJaxpValidator";
  public static final String DEFAULT_TRANSFORMER =
      "wood.keith.opentools.xmltools.XSLTraxTransformer";

  /*
   * Define the properties for storing the XML tools settings.
   */
  /** List of validator names and classes. */
  public static final GlobalArrayProperty VALIDATOR_LIST =
      new GlobalArrayPropertyDefault(CATEGORY, "validatorList", new String[]
      {DEFAULT_VALIDATOR, "wood.keith.opentools.xmltools.XMLXerces2Validator"});
  /** Position of the default validator in the list. */
  public static final GlobalIntegerProperty VALIDATOR =
      new GlobalIntegerProperty(CATEGORY, "validatorIndex", 0);
  /** List of transformer names and classes. */
  public static final GlobalArrayProperty TRANSFORMER_LIST =
      new GlobalArrayPropertyDefault(CATEGORY, "transformerList", new String[]
      {DEFAULT_TRANSFORMER,
      "wood.keith.opentools.xmltools.XSLXalan22Transformer",
      "wood.keith.opentools.xmltools.XSLSaxon551Transformer",
      "wood.keith.opentools.xmltools.XSLSaxon653Transformer"});
  /** Position of the default transformer in the list. */
  public static final GlobalIntegerProperty TRANSFORMER =
      new GlobalIntegerProperty(CATEGORY, "transformerIndex", 0);

  /**
   * Register the "XML Tools" property group.
   * Provides the needed OpenTools interface required to register the
   * PropertyGroup which retains/edits settings used by the wizard.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    PropertyManager.registerPropertyGroup(new XMLToolsPropertyGroup());
  }

  /**
   * Create PropertyPageFactory.
   * Provides a class used as a factory to create the "XML Tools" property
   * page whenever generic or XML node properties are requested.
   *
   * @param  topic  the topic for which property pages are desired.
   * @return  returns null or the factory class.
   */
  public PropertyPageFactory getPageFactory(Object topic) {
    if (topic == null ||  // Generic properties
        (topic instanceof Node && XMLTools.isXmlNode((Node)topic))) {
      return new PropertyPageFactory(PAGE_NAME) {
        public PropertyPage createPropertyPage() {
          return new XMLToolsPropertyPage();
        }
      };
    }
    return null;
  }

  /**
   * Invoked once for a PropertyGroup to indicate that the values of all
   * global properties have been read from permanent storage and that any
   * settings that depend on these properties can be safely made.
   *
   * Only PropertyGroup instances registered during OpenTool initialization
   * will receive this notification.
   */
  public void initializeProperties() {
  }
}
