package wood.keith.opentools.xmltools;

import java.awt.event.ActionEvent;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.actions.UpdateAction;
import com.borland.primetime.editor.EditorAction;
import com.borland.primetime.editor.EditorContextActionProvider;
import com.borland.primetime.editor.EditorManager;
import com.borland.primetime.editor.EditorPane;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.BrowserAction;
import com.borland.primetime.ide.ProjectView;
import com.borland.primetime.ide.ContextActionProvider;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.TextFileNode;

import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;

/**
 * Menu extensions for validating XML files in JBuilder.
 *
 * There are several local context menus, and this class will
 * add entries to the local menu in the Project View, and also
 * to the local menu in the "source" window in the Content pane.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  16 October 2000
 * @version  2.0  15 February 2002
 */
public class XMLValidateMenu implements ContextActionProvider {

  /** The XML validator. */
  private static XMLValidator _xmlValidator = null;

  /** The menu caption. */
  private static final String CAPTION = "Validate XML";

  /* Messages. */
  private static final String SAVE_SOURCE =
      "Please save the file before validating";
  private static final String VALIDATE_SUCCESS =
      "XML validation successful\n(checked by {0})";
  private static final String VALIDATE_FAIL =
      "XML validation failed\n(checked by {0})\n{1}";
  private static final String VALIDATE_FAIL_AT =
      "XML validation failed\n(checked by {0})\n{1}\n" +
      "at line {2,number,integer}, column {3,number,integer}";
  private static final String USING = "Using {0}";
  private static final String VALIDATOR_MISSING =
      "XMLValidator class not found\n{0}";
  private static final String VALIDATOR_BUILD =
      "XMLValidator class not created\n{0}";

  /**
   * When the OpenTool add-ins are loaded, two ContextActionProviders
   * are registered, one for the Project Pane and one for the EditorPane.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    XMLValidateMenu validateMenu = new XMLValidateMenu();
    ProjectView.registerContextActionProvider(validateMenu);
    EditorManager.registerContextActionProvider(validateMenu.ValidateXML);
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded XML Tools Validate menu v" + XMLTools.VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  /**
   * Initialisation.
   */
  public XMLValidateMenu() {
    // Find the name of the XMLValidator class
    String[] validatorList = XMLToolsPropertyGroup.VALIDATOR_LIST.getValues();
    int index = XMLToolsPropertyGroup.VALIDATOR.getInteger();
    String validatorClass = (index < validatorList.length ?
      validatorList[index] : XMLToolsPropertyGroup.DEFAULT_VALIDATOR);
    // And create an instance of it
    setValidator(validatorClass);
  }

  /**
   * Instantiate the validator class and update UI .
   *
   * @param  validatorClass  the name of the XMLValidator class
   */
  public static void setValidator(String validatorClass) {
    try {
      _xmlValidator = null;
      _xmlValidator = (XMLValidator)Class.forName(validatorClass).newInstance();
      // Set the long description to the validator name
      ((UpdateAction)ACTION_PROJECT_VALIDATE_XML).setLongText(
          MessageFormat.format(USING,
          new Object[] {_xmlValidator.getValidatorName()}));
      ((UpdateAction)ACTION_EDITOR_VALIDATE_XML).setLongText(
          MessageFormat.format(USING,
          new Object[] {_xmlValidator.getValidatorName()}));
    }
    catch (ClassNotFoundException cnfe) {
      JOptionPane.showMessageDialog(Browser.getActiveBrowser(),
          MessageFormat.format(VALIDATOR_MISSING, new Object[]
          {cnfe.getMessage()}), CAPTION, JOptionPane.ERROR_MESSAGE);
      _xmlValidator = null;
    }
    catch (Exception ex) {
      ex.printStackTrace();
      JOptionPane.showMessageDialog(Browser.getActiveBrowser(),
          MessageFormat.format(VALIDATOR_BUILD, new Object[] {ex.getMessage()}),
          CAPTION, JOptionPane.ERROR_MESSAGE);
    }
  }

  /**
   * Execute the action - validating the XML document.
   *
   * @param  browser  the browser for the current document
   * @param  node     the active file node
   */
  private static void doValidateAction(Browser browser, Node node) {
    // We only want XML files
    if (XMLTools.isXmlNode(node)) {
      try {
        // Validate the document
        InputSource xmlDocument = new InputSource(
            ((TextFileNode)node).getBuffer().getInputStream());
        xmlDocument.setSystemId(
            "file:///" + ((TextFileNode)node).getUrl().getFullName());
        _xmlValidator.validate(xmlDocument);
        // Success
        JOptionPane.showMessageDialog(browser,
            MessageFormat.format(VALIDATE_SUCCESS,
            new Object[] {_xmlValidator.getValidatorName()}),
            CAPTION, JOptionPane.INFORMATION_MESSAGE);
      }
      catch (SAXParseException spe) {
        // Failed - position given
        spe.printStackTrace();
        showError(MessageFormat.format(VALIDATE_FAIL_AT, new Object[] {
            _xmlValidator.getValidatorName(), spe.getMessage(),
            new Integer(spe.getLineNumber()),
            new Integer(spe.getColumnNumber())}),
            browser, node, spe.getLineNumber(), spe.getColumnNumber());
      }
      catch (UnknownHostException uhe) {
        // Failed
        uhe.printStackTrace();
        showError(MessageFormat.format(VALIDATE_FAIL, new Object[] {
            _xmlValidator.getValidatorName(),
            "Unknown URL host: " + uhe.getMessage()}), browser, node, -1, -1);
      }
      catch (Throwable th) {
        // Failed
        th.printStackTrace();
        showError(MessageFormat.format(VALIDATE_FAIL, new Object[] {
            _xmlValidator.getValidatorName(),
            th.getClass().getName() + " " + th.getMessage()}),
            browser, node, -1, -1);
      }
    }
  }

  /**
   * Display an error message to the user and bring up the problem node.
   *
   * @param  message  the error text to display
   * @param  browser  the current browser
   * @param  node     the XML node being validated
   * @param  line     the line number of the error, or -1 if not known
   * @param  column   the column number of the error, or -1 if not known
   */
  private static void showError(String message, Browser browser, Node node,
      int line, int column) {
    try {
      browser.openNodes(new Node[] {node}, node);
    }
    catch (Exception ex) {
      // Ignore
    }
    if (line > -1) {
      EditorPane editor = EditorAction.getFocusedEditor();
      editor.gotoPosition(line, column, true);
    }
    JOptionPane.showMessageDialog(browser, message, CAPTION,
        JOptionPane.ERROR_MESSAGE);
  }

  /**
   * The action that will get added to the Project View local menu. The
   * actionPerformed function is triggered by selecting "Validate XML" from the
   * local menu after a right click on an XML file node in the Project View.
   */
  public static final BrowserAction ACTION_PROJECT_VALIDATE_XML =
      new BrowserAction(CAPTION) {
    public void actionPerformed(Browser browser) {
      doValidateAction(browser, browser.getProjectView().getSelectedNode());
    }
  };

  /**
   * If the node is a single node that belongs to an XML file, the
   * appropriate action is returned that is able to deal with an XML file.
   * Part of the ContextActionProvider interface.
   *
   * @param  browser  the current browser
   * @param  nodes    the set of nodes that are currently selected
   * @return  an action to validate an XML file
   */
  public Action getContextAction(Browser browser, Node[] nodes) {
    if (_xmlValidator != null &&
        nodes.length == 1 && XMLTools.isXmlNode(nodes[0])) {
      return ACTION_PROJECT_VALIDATE_XML;
    }
    return null;
  }

  /**
   * Instead of implementing the EditorContextActionProvider interface in
   * the XMLValidateMenu class, and registering the whole class as the provider,
   * it is also possible to register a local class as the provider. Compared
   * with the ContextActionProvider interface, the EditorContextActionProvider
   * interface also has a getContextAction function, but with different
   * parameters, and in addition it requires a getPriority function.
   */
  private EditorContextActionProvider ValidateXML =
      new EditorContextActionProvider() {
    // Called when the editor right click context menu is about to appear.
    // Add the ValidateXML menu entry in the XMLTools menu group and try to
    // make the entry appear as the bottom entry of the local menu.
    public Action getContextAction(EditorPane editor) {
      // If this is a right click on an XML file node
      // we will return the proper menu entry
      if (XMLTools.isXmlNode(Browser.getActiveBrowser().getActiveNode())) {
        return XMLToolsMenuGroup.getMenuGroup();
      }
      return null;
    }

    // Make it show very low on the local context menu.
    // Priorities range from 1 - 100.  Less than 5 will probably make the menu
    // entry end up at the bottom, while 99 will make it appear at the top.
    // Priorities are shared with other menu entries, so no priority
    // will guarantee a specific place.
    public int getPriority() { return 4; }
  };

  /**
   * The action that will get added to the editor local menu.
   * The action performed function will be called when the
   * "Validate XML" menu entry is selected after a right
   * mouse click in an editor pane on the "source" tab.
   */
  private static final UpdateAction ACTION_EDITOR_VALIDATE_XML =
      new UpdateAction(CAPTION) {
    /**
     * Called to determine if the menu should be turned on or not.
     * If the menu entry should be shown, call setEnabled with
     * true, otherwise call setEnabled with false.
     */
    public void update(Object source) {
      Node node = Browser.getActiveBrowser().getActiveNode();
      // Only enable the menu entry for XML file nodes.
      setEnabled(_xmlValidator != null && XMLTools.isXmlNode(node));
    }

    /**
     * Perform the requested action - validate the XML document.
     */
    public void actionPerformed(ActionEvent e) {
      Browser browser = Browser.getActiveBrowser();
      doValidateAction(browser, browser.getActiveNode());
    }
  };

  /**
   * One-off initialisation.
   */
  static {
    XMLToolsMenuGroup.getMenuGroup().add(ACTION_EDITOR_VALIDATE_XML);
    try {
      // Load icon for the menu items
      Icon validateIcon = new ImageIcon(ClassLoader.getSystemResource(
          "wood/keith/opentools/xmltools/XMLVal16.gif"));
      ((UpdateAction)ACTION_PROJECT_VALIDATE_XML).setSmallIcon(validateIcon);
      ((UpdateAction)ACTION_EDITOR_VALIDATE_XML).setSmallIcon(validateIcon);
    }
    catch (Exception ex) {
      // Ignore
    }
  }
}
