package wood.keith.opentools.xmltools;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.text.MessageFormat;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileSystemView;
import javax.swing.text.Document;

import com.borland.jbuilder.editor.HTMLEditorKit;
import com.borland.jbuilder.editor.XMLEditorKit;
import com.borland.primetime.editor.EditorManager;
import com.borland.primetime.editor.EditorPane;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.node.TextFileNode;
import com.borland.primetime.vfs.Url;

import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;

/**
 * Create the viewer that shows the effect of an XSLT file on an XML document.
 * The viewer shows the XSLT source and the results of the transformation
 * (as source and display) in separate editor panes.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  16 October 2000
 * @version  2.0  15 February 2002
 */
class XSLTViewer extends JPanel implements PropertyChangeListener {

  /** The Node object of the XML document. */
  private TextFileNode _sourceNode;

  /** The XSLT transformation engine. */
  private XSLTransformer _xslTransformer;

  /* Messages. */
  private static final String CAPTION = "XSLT Processing";
  private static final String MISSING_FILE = "The XSLT file does not exist";
  private static final String UPDATES_PENDING =
    "Please save the XML file before applying the transformation";
  private static final String TRANSFORM_FAIL =
    "Transformation failed\n{0}";
  private static final String TRANSFORM_FAIL_AT =
    "Transformation failed\n{0}\n" +
    "at line {1,number,integer}, column {2,number,integer}";
  /* File chooser dialog */
  private static final String FIND_TITLE = "XSLT Document";
  private static final String XSL_FILE_DESC = "XSL files (*.xsl)";
  private static final String XSL_FILE_EXT = ".xsl";

  private BorderLayout _borderLayout = new BorderLayout();
  private JPanel jPanel1 = new JPanel();
  private JLabel jLabel1 = new JLabel();
  private JTextField _templateField = new JTextField();
  private JButton _browseButton = new JButton();
  private JButton _applyButton = new JButton();
  private GridBagLayout _gridBagLayout = new GridBagLayout();
  private JTabbedPane _resultsTabs = new JTabbedPane();
  private JScrollPane _viewScroll = new JScrollPane();
  private EditorPane _viewEditor = new EditorPane();
  private JScrollPane _sourceScroll = new JScrollPane();
  private EditorPane _sourceEditor = new EditorPane();
  private JScrollPane _templateScroll = new JScrollPane();
  private EditorPane _templateEditor = new EditorPane();
  private JPopupMenu _xsltPopup = new JPopupMenu();
  private JMenuItem _openTemplateItem = new JMenuItem();
  /* Specialised editor kits for displays */
  private HTMLEditorKit _htmlSourceKit =
      (HTMLEditorKit)EditorManager.getEditorKit(HTMLEditorKit.class);
  private XMLEditorKit _xslSourceKit =
      (XMLEditorKit)EditorManager.getEditorKit(XMLEditorKit.class);

  /**
   * Create a new viewer.
   *
   * @param  sourceNode  the Node corresponding to the original document
   */
  public XSLTViewer(TextFileNode sourceNode) {
    _sourceNode = sourceNode;
    _xslTransformer = XSLTViewerFactory.getTransformer();
    XSLTViewerFactory.addPropertyChangeListener(this);
    try {
      jbInit();
      setStylesheet();
    }
    catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  /**
   * Create the UI of the viewer.
   */
  private void jbInit() throws Exception {
    this.setLayout(_borderLayout);
    jPanel1.setLayout(_gridBagLayout);
    jLabel1.setDisplayedMnemonic('D');
    jLabel1.setHorizontalAlignment(SwingConstants.RIGHT);
    jLabel1.setLabelFor(_templateField);
    jLabel1.setText("XSLT document");
    _browseButton.setToolTipText("Find an XSLT document");
    _browseButton.setMnemonic('B');
    _browseButton.setText("Browse...");
    _browseButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        browseButton_actionPerformed(e);
      }
    });
    _applyButton.setToolTipText("Apply using " +
      _xslTransformer.getTransformerName());
    _applyButton.setMnemonic('Y');
    _applyButton.setText("Apply");
    _applyButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        applyButton_actionPerformed(e);
      }
    });
    _resultsTabs.setTabPlacement(JTabbedPane.BOTTOM);
    _viewEditor.setContentType("text/html");
    _viewEditor.setEditable(false);
    // Use built-in editor kit for HTML source
    _sourceEditor.setEditorKit(_htmlSourceKit);
    _sourceEditor.setEditable(false);
    // Use built-in editor kit for XML source
    _templateEditor.setEditorKit(_xslSourceKit);
    _templateEditor.setEditable(false);
    _templateEditor.addMouseListener(new java.awt.event.MouseAdapter() {
      public void mouseReleased(MouseEvent e) {
        templateEditor_mouseReleased(e);
      }
    });
    _viewScroll.setToolTipText("Output from the transformation (displayed)");
    _sourceScroll.setToolTipText("Output from the transformation (source)");
    _templateScroll.setToolTipText("Transformation template source");
    _openTemplateItem.setText("Open template");
    _openTemplateItem.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        openTemplateItem_actionPerformed(e);
      }
    });
    this.add(jPanel1, BorderLayout.NORTH);
    jPanel1.add(jLabel1, new GridBagConstraints(0, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(2, 4, 2, 4), 0, 0));
    jPanel1.add(_templateField, new GridBagConstraints(1, 0, 1, 1, 1.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL, new Insets(2, 4, 2, 0), 0, 0));
    jPanel1.add(_applyButton, new GridBagConstraints(3, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(2, 4, 2, 4), 0, 0));
    jPanel1.add(_browseButton, new GridBagConstraints(2, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(2, 4, 2, 4), 0, 0));
    this.add(_resultsTabs, BorderLayout.CENTER);
    _resultsTabs.addTab("View", null, _viewScroll, _viewScroll.getToolTipText());
    _resultsTabs.addTab("Source",
      null, _sourceScroll, _sourceScroll.getToolTipText());
    _resultsTabs.addTab("Template",
      null, _templateScroll, _templateScroll.getToolTipText());
    _viewScroll.getViewport().add(_viewEditor, null);
    _sourceScroll.getViewport().add(_sourceEditor, null);
    _templateScroll.getViewport().add(_templateEditor, null);
    _xsltPopup.add(_openTemplateItem);
    _templateField.requestFocus();
  }

  /**
   * Respond to changes in the transformation engine used.
   */
  public void propertyChange(PropertyChangeEvent evt) {
    if (evt.getPropertyName().equals(XSLTViewerFactory.XSL_TRANSFORMER_PROP)) {
      _xslTransformer = (XSLTransformer)evt.getNewValue();
      _applyButton.setToolTipText("Apply using " +
        _xslTransformer.getTransformerName());
    }
    else {
      throw new UnsupportedOperationException(
        "Unknown property " + evt.getPropertyName());
    }
  }

  /**
   * Set the initial value for the stylesheet from
   * the first declaration in the document.
   */
  private void setStylesheet() {
    // Default to the directory of the original document
    String stylesheet =
      _sourceNode.getUrl().getFileObject().getParent() + File.separatorChar;
    try {
      BufferedReader reader = new BufferedReader(
        new InputStreamReader(_sourceNode.getBuffer().getInputStream()));
      String line;
      while ((line = reader.readLine()) != null) {
        // Find the stylesheet PI
        int pos = line.indexOf("<?xml:stylesheet");
        if (pos == -1) {
          pos = line.indexOf("<?xml-stylesheet");
        }
        if (pos  > -1) {
          // Look for the external reference (assumed to be on the same line)
          pos = line.indexOf("href=\"", pos) + 6;
          // And extract the file name
          stylesheet = line.substring(pos, line.indexOf("\"", pos));
          // Adjust for relative references
          if (stylesheet.indexOf(File.separatorChar) == -1) {
            stylesheet = _sourceNode.getUrl().getFileObject().getParent() +
              File.separatorChar + stylesheet;
          }
          break;
        }
      }
    }
    catch (Exception ex) {
      // Ignore
    }
    _templateField.setText(stylesheet);
  }

  /**
   * Search for an appropriate XSLT document to use.
   */
  void browseButton_actionPerformed(ActionEvent e) {
    // We default to the directory that the source pane editor file lives in
    JFileChooser fileChooser = new JFileChooser(
      _sourceNode.getUrl().getFileObject().getParent() + File.separatorChar,
      FileSystemView.getFileSystemView());
    fileChooser.setSelectedFile(new File(_templateField.getText()));
    // Set filters for all and .xsl files, with the latter as the default
    fileChooser.addChoosableFileFilter(fileChooser.getAcceptAllFileFilter());
    fileChooser.setFileFilter(new FileFilter() {
      public boolean accept(File file) {
        String path = file.getPath();
        return (file.isDirectory() ||
          ((path.length() > XSL_FILE_EXT.length()) &&
          (path.substring(path.length() - XSL_FILE_EXT.length()).
          toLowerCase().equals(XSL_FILE_EXT))));
      }
      public String getDescription() {
        return XSL_FILE_DESC;
      }
    });
    fileChooser.setDialogTitle(FIND_TITLE);
    // Let the user select
    if (fileChooser.showOpenDialog(Browser.getActiveBrowser()) ==
        JFileChooser.APPROVE_OPTION) {
      if (!fileChooser.getSelectedFile().exists()) {
        JOptionPane.showMessageDialog(Browser.getActiveBrowser(),
          MISSING_FILE, CAPTION, JOptionPane.ERROR_MESSAGE);
        return;
      }
      // Copy the value back
      _templateField.setText(fileChooser.getSelectedFile().getAbsolutePath());
    }
  }

  /**
   * Open the specified XSLT document and apply it to the current XML document.
   */
  void applyButton_actionPerformed(ActionEvent e) {
    // Clear output
    _viewEditor.setText("");
    _sourceEditor.setText("");
    _templateEditor.setText("");
    // Check that the template file exists
    File templateFile = new File(_templateField.getText());
    if (!templateFile.exists()) {
      JOptionPane.showMessageDialog(Browser.getActiveBrowser(),
        MISSING_FILE, CAPTION, JOptionPane.ERROR_MESSAGE);
      return;
    }

    try {
      setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      // Load the template file source for display
      FileInputStream xsltIn = new FileInputStream(templateFile);
      Document doc = (Document)_templateEditor.getDocument();
      _templateEditor.getEditorKit().read(xsltIn, doc, 0);
      _templateEditor.gotoLine(1, false);

      // Perform the XSL transformation
      InputSource xmlDocument =
        new InputSource(_sourceNode.getBuffer().getInputStream());
      xmlDocument.setSystemId("file:///" + _sourceNode.getUrl().getFullName());
      InputSource xslDocument =
        new InputSource(new FileInputStream(templateFile));
      xslDocument.setSystemId("file:///" + templateFile.getAbsolutePath());
      String xsltResults = _xslTransformer.transform(xmlDocument, xslDocument);

      // Success - transfer the results to the output display and source pages
      _sourceEditor.setText(xsltResults);
      _sourceEditor.gotoLine(1, false);
      _viewEditor.setText(xsltResults);
      _viewEditor.gotoLine(1, false);
      _resultsTabs.setSelectedComponent(_viewScroll);
    }
    catch (SAXParseException spe) {
      // Failed - position given
      spe.printStackTrace();
      showError(
        MessageFormat.format(TRANSFORM_FAIL_AT, new Object[] {spe.getMessage(),
        new Integer(spe.getLineNumber()), new Integer(spe.getColumnNumber())}),
        spe.getLineNumber(), spe.getColumnNumber());
    }
    catch (Exception ex) {
      // Failed
      ex.printStackTrace();
      showError(MessageFormat.format(TRANSFORM_FAIL, new Object[]
        {ex.getClass().getName() + " " + ex.getMessage()}), -1, -1);
    }
    catch (Error er) {
      // Failed
      er.printStackTrace();
      showError(MessageFormat.format(TRANSFORM_FAIL, new Object[]
        {er.getClass().getName() + " " + er.getMessage()}), -1, -1);
    }
    finally {
      setCursor(Cursor.getDefaultCursor());
    }
  }

  /**
   * Display an error message and show XSL document.
   *
   * @param  message  is the text to display
   * @param  line     is the line number in error, or -1 if not known
   * @param  column   is the column number in error, or -1 if not known
   */
  private void showError(String message, int line, int column) {
    JOptionPane.showMessageDialog(Browser.getActiveBrowser(), message,
      CAPTION, JOptionPane.ERROR_MESSAGE);
    _resultsTabs.setSelectedComponent(_templateScroll);
    if (line != -1) {
      _templateEditor.gotoPosition(line, column, true);
    }
  }

  /**
   * Popup the context menu for the template.
   */
  void templateEditor_mouseReleased(MouseEvent e) {
    if (e.isPopupTrigger()) {
      _openTemplateItem.setEnabled(_templateEditor.getText().length() > 0);
      _xsltPopup.show(_templateEditor, e.getX(), e.getY());
    }
  }

  /**
   * Open a node for the XSLT template.
   */
  void openTemplateItem_actionPerformed(ActionEvent e) {
    File templateFile = new File(_templateField.getText());
    if (templateFile.exists()) {
      Browser.getActiveBrowser().doOpen(new Url(templateFile),
        Browser.getActiveBrowser().getActiveProject(), false);
    }
  }

  /**
   * Forces the viewer to a certain line number position.
   *
   * @param  lineNo  the line number the viewer should go to
   */
  public void setPosition(int lineNo) {
    // Position the editor
    _templateEditor.gotoLine(lineNo + 1, false, EditorPane.ONETHIRD_ALWAYS);
    _templateEditor.requestFocus();
  }
}