package wood.keith.opentools.xmltools;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.MessageFormat;
import java.util.EventListener;
import javax.swing.JOptionPane;
import javax.swing.event.EventListenerList;

import com.borland.primetime.PrimeTime;
import com.borland.primetime.ide.Browser;
import com.borland.primetime.ide.Context;
import com.borland.primetime.ide.NodeViewer;
import com.borland.primetime.ide.NodeViewerFactory;
import com.borland.primetime.node.Node;
import com.borland.primetime.node.TextFileNode;

/**
 * Create a page for applying XSLT transformations to XML documents.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  16 October 2000
 * @version  2.0  15 February 2002
*/
public class XSLTViewerFactory implements NodeViewerFactory {

  /** The name of the transformer property. */
  public static final String XSL_TRANSFORMER_PROP = "xslTransformer";

  /* Messages. */
  private static final String CAPTION = "XML Tools";
  private static final String TRANSFORMER_MISSING =
    "XSLTransformer class not found\n{0}";
  private static final String TRANSFORMER_BUILD =
    "XSLTransformer class not created\n{0}";

  /** Listeners for transformer changes. */
  private static EventListenerList _listenerList = new EventListenerList();

  /** The XSL transformation object. */
  private static XSLTransformer _xslTransformer = null;

  /** The viewer factory object. */
  private static XSLTViewerFactory _xsltViewerFactory = new XSLTViewerFactory();

  /** Static initialisation. */
  static {
    // Find the name of the XMLValidator class
    String[] transformerList =
        XMLToolsPropertyGroup.TRANSFORMER_LIST.getValues();
    int index = XMLToolsPropertyGroup.TRANSFORMER.getInteger();
    String transformerClass = (index < transformerList.length ?
      transformerList[index] : XMLToolsPropertyGroup.DEFAULT_TRANSFORMER);
    // And create an instance of it
    setTransformer(transformerClass);
  }

  /**
   * Register this factory as a NodeViewer factory.
   *
   * @param  majorVersion  the major version of the current OpenTools API
   * @param  minorVersion  the minor version of the current OpenTools API
   */
  public static void initOpenTool(byte majorVersion, byte minorVersion) {
    if (majorVersion != PrimeTime.CURRENT_MAJOR_VERSION) {
      return;
    }
    Browser.registerNodeViewerFactory(_xsltViewerFactory);
    if (PrimeTime.isVerbose()) {
      System.out.println("Loaded XSLT Viewer v" + XMLTools.VERSION);
      System.out.println("Written by Keith Wood (kbwood@iprimus.com.au)");
    }
  }

  /**
   * One of the functions needed to implement NodeViewerFactory.
   * Use this function to tell the Browser if this factory can
   * view a certain node.
   *
   * @param  node  the node the Browser will open
   * @return  true if this factory can create a viewer for this node,
   *          false otherwise.
   */
  public boolean canDisplayNode(Node node) { return XMLTools.isXmlNode(node); }

  /**
   * One of the functions needed to implement NodeViewerFactory.
   * Used to create the viewer for the node that is described
   * by a context. The viewer created is of type XSLTNodeViewer.
   *
   * @param  context  the context that describes the node.
   * @return  a NodeViewer capable of viewing this node, or
   *          null if the viewer creation fails.
   */
  public NodeViewer createNodeViewer(Context context) {
    Node node = context.getNode();
    if (canDisplayNode(node)) {
      return new XSLTNodeViewer(context, (TextFileNode)node);
    }
    return null;
  }

  /**
   * Instantiate the transformer class.
   *
   * @param  transformerClass  the name of the XSLTransformer class
   */
  public static void setTransformer(String transformerClass) {
    try {
      XSLTransformer oldXslt = _xslTransformer;
      _xslTransformer = null;
      _xslTransformer =
        (XSLTransformer)Class.forName(transformerClass).newInstance();
      firePropertyChange(XSL_TRANSFORMER_PROP, oldXslt, _xslTransformer);
      oldXslt = null;
    }
    catch (ClassNotFoundException cnfe) {
      JOptionPane.showMessageDialog(Browser.getActiveBrowser(),
        MessageFormat.format(TRANSFORMER_MISSING,
        new Object[] {cnfe.getMessage()}),
        CAPTION, JOptionPane.ERROR_MESSAGE);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      JOptionPane.showMessageDialog(Browser.getActiveBrowser(),
        MessageFormat.format(TRANSFORMER_BUILD, new Object[] {ex.getMessage()}),
        CAPTION, JOptionPane.ERROR_MESSAGE);
    }
  }

  /**
   * Retrieve the XSL Transformation object.
   *
   * @return  the current XSLTransformer object
   */
  public static XSLTransformer getTransformer() { return _xslTransformer; }

  /**
   * Add a PropertyChangeListener to the listener list.
   * The listener is registered for all properties.
   *
   * @param  listener  the PropertyChangeListener to be added
   */
  public static void addPropertyChangeListener(
      PropertyChangeListener listener) {
    _listenerList.add(PropertyChangeListener.class, listener);
  }

  /**
   * Remove a PropertyChangeListener from the listener list.
   * This removes a PropertyChangeListener that was registered
   * for all properties.
   *
   * @param  listener  the PropertyChangeListener to be removed
   */
  public static void removePropertyChangeListener(
      PropertyChangeListener listener) {
    _listenerList.remove(PropertyChangeListener.class, listener);
  }

  /**
   * Report a bound property update to any registered listeners.
   * No event is fired if old and new are equal and non-null.
   *
   * @param  propertyName  the programmatic name of the property
   *                       that was changed
   * @param  oldValue      the old value of the property
   * @param  newValue      the new value of the property
   */
  public static void firePropertyChange(String propertyName,
      Object oldValue, Object newValue) {
    if (oldValue != null && newValue != null && oldValue.equals(newValue)) {
      return;
    }

    EventListener[] listeners =
      _listenerList.getListeners(PropertyChangeListener.class);
    PropertyChangeEvent event = null;
    for (int index = 0; index < listeners.length; index++) {
      if (event == null) {
        event = new PropertyChangeEvent(
          _xsltViewerFactory, propertyName, oldValue, newValue);
      }
      ((PropertyChangeListener)listeners[index]).propertyChange(event);
    }
  }
}
