package wood.keith.opentools.xmltools;

import java.io.StringWriter;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.URIResolver;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.xml.sax.InputSource;

/**
 * TrAX interface for applying an XSL Transformation to a document.
 *
 * @author   Keith Wood (kbwood@iprimus.com.au)
 * @version  1.0  28 November 2003
 */
public class XSLTraxTransformer implements XSLTransformer {

  // Singleton factory
  private static final TransformerFactory FACTORY =
      TransformerFactory.newInstance();

  /**
   * Provide the factory that generates transformers.
   *
   * @return  the appropriate factory
   */
  protected TransformerFactory getFactory() { return FACTORY; }

  /**
   * Apply an XSL transformation to a document.
   *
   * @param  xmlDocument  a source representing the XML document to be transformed
   * @param  xslDocument  a source representing the XSL transformation
   * @return  the text resulting from the transformation process
   * @throws  Exception  when an error occurs, may be a SAXParseException
   *                     showing line and column where the error occurred
   */
  public String transform(InputSource xmlDocument, InputSource xslDocument)
      throws Exception {
    // Create a transformer for the stylesheet.
    Transformer transformer =
        getFactory().newTransformer(makeSource(xslDocument));
    // Initialise document source
    // Apply the transformation
    StringWriter out = new StringWriter();
    transformer.transform(makeSource(xmlDocument), new StreamResult(out));
    // Return the results
    return out.toString();
  }

  /**
   * Convert a SAX input source to a TrAX source.
   *
   * @param  inSource  the SAX input source
   * @return  the TrAX source
   */
  protected StreamSource makeSource(InputSource inSource) {
    StreamSource source = new StreamSource(inSource.getByteStream());
    source.setPublicId(inSource.getPublicId());
    source.setSystemId(inSource.getSystemId());
    return source;
  }

  /**
   * Return a description for this transformation implementation.
   *
   * @return  the implementation's description
   */
  public String getTransformerName() {
    try {
      return "TrAX (" + TransformerFactory.newInstance().
          newTransformer().getClass().getName() + ")";
    }
    catch (TransformerFactoryConfigurationError ex) {
      return "TrAX";
    }
    catch (TransformerConfigurationException ex) {
      return "TrAX";
    }
  }
}
